#include "serializable.h"

#include <util/generic/cast.h>
#include <util/string/builder.h>

IJsonSerializableTag::ITag::TPtr IJsonSerializableTag::BuildFromJson(const IDriveTagsManager& tagsManager, const NJson::TJsonValue& jsonValue, TMessagesCollector* errors /*= nullptr*/) {
    TString name;
    if (jsonValue.Has("tag_name")) {
        name = jsonValue["tag_name"].GetString();
    } else if (jsonValue.Has("tag")) {
        name = jsonValue["tag"].GetString();
    } else {
        if (errors) {
            errors->AddMessage("parsing", "Tag name is undefined");
        }
        return nullptr;
    }

    auto td = tagsManager.GetTagsMeta().GetDescriptionByName(name);
    if (!td) {
        return nullptr;
    }

    ITag::TPtr tag = TFactory::Construct(td->GetType());
    if (!tag) {
        if (errors) {
            errors->AddMessage("parsing", TStringBuilder() << "Unknown tag type: " << name);
        }
        return nullptr;
    }

    tag->SetDescriptionIndex(td->GetIndex());

    IJsonSerializableTag* serializableTag = dynamic_cast<IJsonSerializableTag*>(tag.Get());
    if (!serializableTag) {
        if (errors) {
            errors->AddMessage("parsing", TStringBuilder() << "Tag " << name << " of type " << td->GetType() << " is not serializable");
        }
        return nullptr;
    }

    if (!serializableTag->FromJson(jsonValue, errors)) {
        return nullptr;
    }

    return tag;
}

ITag::TPtr IJsonSerializableTag::BuildFromString(const IDriveTagsManager& tagsManager, const TString& data, TMessagesCollector* errors /*= nullptr*/) {
    NJson::TJsonValue tagJson;
    if (!NJson::ReadJsonFastTree(data, &tagJson)) {
        if (errors) {
            errors->AddMessage("parsing", "Erro parsing json");
        }
        return nullptr;
    }
    return BuildFromJson(tagsManager, tagJson, errors);
}

ITag::TPtr IJsonSerializableTag::Clone(const IDriveTagsManager& tagsManager) const {
    return BuildFromJson(tagsManager, SerializeToJson());
}

NDrive::TScheme IJsonSerializableTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("tag", "Название имплементации тега");
    result.Add<TFSText>("comment", "Комментарий");
    result.Add<TFSNumeric>("priority", "Приоритет тега");
    result.Add<TFSNumeric>("sla_instant", "deadline").SetVisual(NDrive::TFSNumeric::EVisualType::DateTime);
    return result;
}

void IJsonSerializableTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    if (const auto& comment = GetComment()) {
        NJson::InsertField(json, "comment", comment);
    }
    if (HasSLAInstant()) {
        TJsonProcessor::WriteInstant(json, "sla_instant", GetSLAInstant());
    }
}

TBlob IJsonSerializableTag::DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const {
    NJson::TJsonValue specialJson(NJson::JSON_MAP);
    SerializeSpecialDataToJson(specialJson);
    if (!specialJson.Has("sla_instant") && HasSLAInstant()) {
        TJsonProcessor::WriteInstant(specialJson, "sla_instant", GetSLAInstant());
    }
    return TBlob::FromString(specialJson.GetStringRobust());
}

bool IJsonSerializableTag::DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& data) {
    NJson::TJsonValue specialJson;
    if (!NJson::ReadJsonFastTree(TStringBuf(data.AsCharPtr(), data.Size()), &specialJson)) {
        return false;
    }

    return SpecialDataFromJson(specialJson, nullptr);
}

bool IJsonSerializableTag::FromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) {
    if (jsonValue.Has("tag_name")) {
        SetName(jsonValue["tag_name"].GetString());
    } else if (jsonValue.Has("tag")) {
        SetName(jsonValue["tag"].GetString());
    } else {
        if (errors) {
            errors->AddMessage("parsing", "no tag_name/tag fields");
        }
        return false;
    }

    if (jsonValue.Has("priority")) {
        if (jsonValue["priority"].IsString()) {
            i32 priority;
            if (!TryFromString(jsonValue["priority"].GetString(), priority)) {
                errors ? errors->AddMessage("parsing", "can't parse priority from string") : void();
                return false;
            }
            SetTagPriority(priority);
        } else if (jsonValue["priority"].IsInteger()) {
            SetTagPriority(jsonValue["priority"].GetInteger());
        } else {
            errors ? errors->AddMessage("parsing", "incorrect priority type") : void();
            return false;
        }
    }

    return SpecialDataFromJson(jsonValue, errors);
}

bool IJsonSerializableTag::SpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) {
    return
        NJson::ParseField(jsonValue["sla_instant"], MutableSLAInstant()) &&
        NJson::ParseField(jsonValue["comment"], MutableComment()) &&
        DoSpecialDataFromJson(jsonValue, errors);
}

bool IJsonSerializableTag::DoSpecialDataFromJson(const NJson::TJsonValue& /*json*/, TMessagesCollector* /*errors*/) {
    return true;
}
