#pragma once

#include "common.h"

#include <drive/backend/database/drive/proto/drive.pb.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/memory/blob.h>

class IDriveTagsManager;

class IJsonSerializableTag: public TCommonTag {
private:
    using TBase = TCommonTag;

public:
    using TBase::TBase;

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    template <class T>
    static ITag::TPtr BuildWithComment(const TString& name, const TString& comment) {
        THolder<T> result(new T(name));
        result->SetComment(comment);
        return result.Release();
    }

public:
    static ITag::TPtr BuildFromJson(const IDriveTagsManager& tagsManager,  const NJson::TJsonValue& jsonValue, TMessagesCollector* errors = nullptr);
    static ITag::TPtr BuildFromString(const IDriveTagsManager& tagsManager, const TString& data, TMessagesCollector* errors = nullptr);

public:
    ITag::TPtr Clone(const IDriveTagsManager& tagsManager) const;

    bool SpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors);

protected:
    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;

    TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const override;
    bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& data) override;

    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/);

private:
    bool FromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors);
};

template <typename TProtoExt, bool variativeRead = false, typename TBaseTag = IJsonSerializableTag>
class ISerializableTag: public TBaseTag {
private:
    using TBase = TBaseTag;

public:
    using TProto = TProtoExt;

protected:
    virtual TProto DoSerializeSpecialDataToProto() const {
        TProto proto;
        if (!!GetComment()) {
            proto.SetComment(GetComment());
        }
        return proto;
    }
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) {
        if (proto.HasComment()) {
            SetComment(proto.GetComment());
        }
        return true;
    }

    TProto SerializeSpecialDataToProto() const {
        TProto result = DoSerializeSpecialDataToProto();
        if (HasSLAInstant()) {
            result.SetSLAInstant(GetSLAInstantUnsafe().Seconds());
        }
        return result;
    }
    bool DeserializeSpecialDataFromProto(const TProto& proto) {
        if (proto.HasSLAInstant()) {
            SetSLAInstant(TInstant::Seconds(proto.GetSLAInstant()));
        }
        if (!DoDeserializeSpecialDataFromProto(proto)) {
            return false;
        }
        return true;
    }

    TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& h) const final {
        {
            h.SetIsTagProto(true);
            TProto proto = SerializeSpecialDataToProto();
            return TBlob::FromString(proto.SerializeAsString());
        }
    }
    bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& h, const TBlob& data) final {
        if (!variativeRead || h.GetIsTagProto()) {
            TProto proto;
            if (!proto.ParseFromArray(data.AsCharPtr(), data.Size())) {
                return false;
            }
            return DeserializeSpecialDataFromProto(proto);
        } else {
            NJson::TJsonValue value;
            return NJson::ReadJsonFastTree(TStringBuf(data.AsCharPtr(), data.Size()), &value) && SpecialDataFromJson(value, nullptr);
        }
    }

public:
    using TBase::TBase;

    using TBase::GetComment;
    using TBase::GetSLAInstantUnsafe;
    using TBase::HasSLAInstant;
    using TBase::SetComment;
    using TBase::SetSLAInstant;
    using TBase::SpecialDataFromJson;
};

template <typename TProto, typename TBaseTag = IJsonSerializableTag>
class INativeSerializationTag: public ISerializableTag<TProto, false, TBaseTag> {
private:
    using TBase = ISerializableTag<TProto, false, TBaseTag>;

public:
    using TBase::TBase;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        NProtobufJson::Proto2Json(TBase::SerializeSpecialDataToProto(), json);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* /*errors*/) override {
        typename TBase::TProto resultProto;
        NProtobufJson::Json2Proto(jsonValue, resultProto);
        return TBase::DeserializeSpecialDataFromProto(resultProto);
    }
};
