#include "temporary_tags.h"

#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

bool ITemporaryActionTag::Process(TDBTag&& dbTag, const TString& historyUserId, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    auto tag = dbTag.MutableTagAs<ITemporaryActionTag>();
    if (!tag) {
        return true;
    }
    if (!tag->GetAttempts()) {
        session.SetErrorInfo("TemporaryActionTag::Process", "action_limited", EDriveSessionResult::InconsistencyOffer);
        return false;
    }
    if (tag->GetAttempts() == 1) {
        if (!server.GetDriveDatabase().GetTagsManager().GetEntityTagManager(GetEntityType()).RemoveTag(dbTag, historyUserId, &server, session, true)) {
            return false;
        }
    } else {
        tag->SetAttempts(tag->GetAttempts() - 1);
        if (!server.GetDriveDatabase().GetTagsManager().GetEntityTagManager(GetEntityType()).UpdateTagData(dbTag, historyUserId, session)) {
            return false;
        }
    }
    return true;
}

TDBActions ITemporaryActionTag::GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& rolesManager, bool getPotential) const {
    if (!GetAttempts() || !IsActive(getPotential)) {
        return {};
    }

    auto description = tagsManager.GetTagsMeta().GetDescriptionByName(GetName());
    auto descriptionImpl = dynamic_cast<const ITemporaryActionTag::TDescription*>(description.Get());
    if (!descriptionImpl) {
        return {};
    }

    TDBActions result;
    TMap<TString, TDBAction> additionalActions = rolesManager.GetActions(descriptionImpl->GetActionIds());
    for (auto&& action : additionalActions) {
        auto impl = action.second->Clone();
        auto& sourceContext = Yensured(impl)->MutableSourceContext();
        sourceContext.SetTagId(self.GetTagId());
        sourceContext.SetLimitedPolicy(true);
        sourceContext.SetSince(GetSince());
        sourceContext.SetUntil(GetUntil());
        if (IsPromoCode()) {
            sourceContext.SetPromoCode(IsPromoCode());
        }
        result.push_back(std::move(impl));
    }
    return result;
}

NDrive::TScheme ITemporaryActionTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("max_attempts", "Количество использований").SetMin(1).SetMax(10000).SetRequired(true);
    result.Add<TFSNumeric>("since", "timestamp since").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(false);
    result.Add<TFSNumeric>("until", "timestamp until").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(false);
    result.Add<TFSBoolean>("is_promo_code", "Добавлен по промокоду").SetReadOnly(true);
    result.Add<TFSString>("promo_code", "Промокод").SetReadOnly(true);
    return result;
}

void ITemporaryActionTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE(json, "attempts", Attempts);
    JWRITE_DEF(json, "max_attempts", MaxAttempts, 0);
    JWRITE_DEF(json, "is_promo_code", IsPromoCode(), false);
    if (Since != TInstant::Zero()) {
        JWRITE(json, "since", Since.Seconds());
    }
    if (Until != TInstant::Max()) {
        JWRITE(json, "until", Until.Seconds());
    }
    JWRITE(json, "promo_code", PromoIdentifier);
}

bool ITemporaryActionTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    JREAD_INT_OPT(json, "attempts", Attempts);
    JREAD_INT_OPT(json, "max_attempts", MaxAttempts);
    if (!json.Has("attempts") && MaxAttempts) {
        Attempts = MaxAttempts;
    }
    if (!Attempts) {
        return false;
    }
    JREAD_BOOL_OPT(json, "is_promo_code", PromoCode);
    JREAD_STRING_OPT(json, "promo_code", PromoIdentifier);
    JREAD_INSTANT_OPT(json, "since", Since);
    JREAD_INSTANT_OPT(json, "until", Until);
    return TBase::DoSpecialDataFromJson(json, errors);
}

ITemporaryActionTag::TProto ITemporaryActionTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TTemporaryActionTag proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetAttempts(Attempts);
    if (MaxAttempts) {
        proto.SetMaxAttempts(MaxAttempts);
    }
    if (PromoCode) {
        proto.SetPromoCodeFlag(PromoCode);
    }
    if (PromoIdentifier) {
        proto.SetPromoCode(PromoIdentifier);
    }
    if (Since != TInstant::Zero()) {
        proto.SetSince(Since.Seconds());
    }
    if (Until != TInstant::Max()) {
        proto.SetUntil(Until.Seconds());
    }
    return proto;
}

bool ITemporaryActionTag::DoDeserializeSpecialDataFromProto(const ITemporaryActionTag::TProto& proto) {
    Attempts = proto.GetAttempts();
    if (proto.HasMaxAttempts()) {
        MaxAttempts = proto.GetMaxAttempts();
    }
    if (proto.HasPromoCodeFlag()) {
        PromoCode = proto.GetPromoCodeFlag();
    }
    if (proto.HasPromoCode()) {
        PromoIdentifier = proto.GetPromoCode();
    }
    if (proto.HasSince()) {
        Since = TInstant::Seconds(proto.GetSince());
    }
    if (proto.HasUntil()) {
        Until = TInstant::Seconds(proto.GetUntil());
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme ITemporaryActionTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("action_ids", "Идентификаторы action-ов");
    result.Add<TFSString>("promo_id", "Серия");
    result.Add<TFSString>("promo_groups", "Группы промокодов");
    return result;
}

NJson::TJsonValue ITemporaryActionTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    TJsonProcessor::WriteContainerString(jsonMeta, "action_ids", ActionIds);
    TJsonProcessor::WriteContainerString(jsonMeta, "promo_groups", PromoGroups);
    JWRITE(jsonMeta, "promo_id", PromoId);
    return jsonMeta;
}

bool ITemporaryActionTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_CONTAINER_OPT(jsonMeta, "action_ids", ActionIds);
    JREAD_CONTAINER_OPT(jsonMeta, "promo_groups", PromoGroups);
    JREAD_STRING_OPT(jsonMeta, "promo_id", PromoId);
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}
