#pragma once

#include "serializable.h"

#include <rtline/util/json_processing.h>
#include <rtline/util/instant_model.h>

#include <drive/backend/data/proto/tags.pb.h>
#include <drive/backend/proto/tags.pb.h>

class TDBAction;
using TDBActions = TVector<TDBAction>;
class TRolesManager;
class IDriveTagsManager;

class IUserActionTag {
public:
    virtual TDBActions GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& rolesManager, bool getPotential) const = 0;
};

class ITemporaryActionTag: public ISerializableTag<NDrive::NProto::TTemporaryActionTag>, public IUserActionTag {
private:
    using TBase = ISerializableTag<NDrive::NProto::TTemporaryActionTag>;

private:
    R_FIELD(ui32, Attempts, 1);
    R_FIELD(ui32, MaxAttempts, 0);
    R_FIELD(TInstant, Since, TInstant::Zero());
    R_FIELD(TInstant, Until, TInstant::Max());
    R_FIELD(bool, PromoCode, false);
    R_FIELD(TString, PromoIdentifier);

private:
    virtual NEntityTagsManager::EEntityType GetEntityType() const = 0;

public:
    using TBase::TBase;

    bool Process(TDBTag&& dbTag, const TString& objectId, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
    TDBActions GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& rolesManager, bool getPotential) const override;

public:
    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

    private:
        R_FIELD(TSet<TString>, ActionIds);
        R_FIELD(TSet<TString>, PromoGroups);
        R_FIELD(TString, PromoId);

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

    virtual bool IsActive(bool isPotentiallyActive = false) const {
        const TInstant now = ModelingNow();
        if (!Attempts) {
            return false;
        }
        return (isPotentiallyActive || now >= Since) && now < Until;
    }

    bool IsExpired() const {
        return ModelingNow() >= Until;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { GetEntityType() };
    }

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
};
