#include "communication_channel_tags.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>

void TCommunicationChannelSettings::FillScheme(NDrive::TScheme& scheme) const {
    scheme.Add<TFSBoolean>("ignore_subscription_status", "Игнорировать отписку");
}

void TCommunicationChannelSettings::SerializeToJson(NJson::TJsonValue& json) const {
    TJsonProcessor::Write(json, "ignore_subscription_status", IgnoreSubscriptionStatus);
}

bool TCommunicationChannelSettings::DeserializeFromJson(const NJson::TJsonValue& json) {
    return TJsonProcessor::Read(json, "ignore_subscription_status", IgnoreSubscriptionStatus, false);
}

const TString TLandingUserTag::TypeName = "user_landing";
ITag::TFactory::TRegistrator<TLandingUserTag> TLandingUserTag::Registrator(TLandingUserTag::TypeName);
TLandingUserTag::TDescription::TFactory::TRegistrator<TLandingUserTag::TDescription> TLandingUserTag::TDescription::Registrator(TLandingUserTag::TypeName);

NDrive::TScheme TLandingUserTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    auto gLandings = server->GetDriveAPI()->GetLandingsDB()->GetCachedObjectsMap();
    result.Add<TFSBoolean>("disposable", "Показывать только один раз").SetDefault(Disposable);
    result.Add<TFSVariants>("landing_id", "LandingId для показа").InitVariantsFromKeys(gLandings);
    NDrive::TScheme& fields = result.Add<TFSArray>("context_map_defaults", "Значения параметров лендинга по умолчанию").SetElement<NDrive::TScheme>();
    fields.Add<TFSString>("key", "Ключ");
    fields.Add<TFSString>("default_value", "Значение по умолчанию");
    CommunicationChannelSettings.FillScheme(result);
    return result;
}

NJson::TJsonValue TLandingUserTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    jsonMeta["disposable"] = Disposable;
    if (LandingId) {
        jsonMeta["landing_id"] = LandingId;
    }
    NJson::TJsonValue& mapJson = jsonMeta.InsertValue("context_map_defaults", NJson::JSON_ARRAY);
    for (const auto& [key, value] : DefaultContextParameters) {
        NJson::TJsonValue& pair = mapJson.AppendValue(NJson::JSON_MAP);
        pair["key"] = key;
        pair["default_value"] = value;
    }
    CommunicationChannelSettings.SerializeToJson(jsonMeta);
    return jsonMeta;
}

bool TLandingUserTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    if (jsonMeta["context_map_defaults"].IsArray()) {
        for (auto&& item : jsonMeta["context_map_defaults"].GetArraySafe()) {
            TString key, value;
            if (!NJson::ParseField(item, "key", key, true) || !NJson::ParseField(item, "default_value", value)) {
                return false;
            }
            if (!DefaultContextParameters.emplace(key, value).second) {
                return false;
            }
        }
    }
    return CommunicationChannelSettings.DeserializeFromJson(jsonMeta)
        && NJson::ParseField(jsonMeta["disposable"], Disposable)
        && NJson::ParseField(jsonMeta["landing_id"], LandingId)
        && TBase::DoDeserializeMetaFromJson(jsonMeta);
}

void TLandingUserTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    NJson::TJsonValue& mapJson = json.InsertValue("context_map", NJson::JSON_ARRAY);
    for (const auto& [key, value] : ContextParameters) {
        NJson::TJsonValue& pair = mapJson.AppendValue(NJson::JSON_MAP);
        pair["key"] = ToString(key);
        pair["value"] = value;
    }
    TBase::SerializeSpecialDataToJson(json);
}

bool TLandingUserTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (json["context_map"].IsArray()) {
        for (auto&& item : json["context_map"].GetArraySafe()) {
            TString key, value;
            if (!NJson::ParseField(item, "key", key) || !NJson::ParseField(item, "value", value)) {
                return false;
            }
            if (!SetContextParameter(key, value)) {
                return false;
            }
        }
    }
    return TBase::DoSpecialDataFromJson(json, errors);
}

NDrive::NProto::TLandingUserTagData TLandingUserTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TLandingUserTagData proto = TBase::DoSerializeSpecialDataToProto();
    for (const auto& [key, value] : ContextParameters) {
        TProtoPair* paramProto = proto.AddContextParameters();
        paramProto->SetKey(key);
        paramProto->SetValue(value);
    }
    return proto;
}

bool TLandingUserTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TLandingUserTagData& proto) {
    for (ui32 i = 0; i < proto.ContextParametersSize(); ++i) {
        ContextParameters[proto.GetContextParameters(i).GetKey()] = proto.GetContextParameters(i).GetValue();
    }

    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TLandingUserTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme scheme = TBase::GetScheme(server);
    NDrive::TScheme& fields = scheme.Add<TFSArray>("context_map", "Параметры заполнения лендинга").SetElement<NDrive::TScheme>();
    fields.Add<TFSString>("key", "Ключ");
    fields.Add<TFSString>("value", "Значение");
    return scheme;
}

TMaybe<NJson::TJsonValue> TLandingUserTag::SubstituteTemplate(const NJson::TJsonValue& templateJson, const NDrive::IServer* server, const TString& escPrefix /* = "["*/, const TString& escSuffix /* = "]"*/) const {
    TString resultStr = NJson::WriteJson(templateJson, false);
    bool substituted = false;
    for (const auto& [key, value] : ContextParameters) {
        auto substCount = SubstGlobal(resultStr, escPrefix + key + escSuffix, NEscJ::EscapeJ<false>(value));
        substituted = substituted || substCount > 0;
    }
    if (server) {
        auto description = std::dynamic_pointer_cast<const TDescription>(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName()));
        if (description) {
            for (const auto& [key, value] : description->GetDefaultContextParameters()) {
                auto substCount = SubstGlobal(resultStr, escPrefix + key + escSuffix, NEscJ::EscapeJ<false>(value));
                substituted = substituted || substCount > 0;
            }
        }
    }
    if (substituted) {
        NJson::TJsonValue result;
        if (NJson::ReadJsonTree(resultStr, &result)) {
            return result;
        } else {
            return Nothing();
        }
    }
    return templateJson;
}
