#include "car_complaint.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/database/drive/url.h>
#include <drive/library/cpp/mds/client.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/common/scheme_adapters.h>
#include <drive/backend/database/history/session_builder.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/json_processing.h>


const TString TCarComplaintTag::TypeName = "car_complaint_tag";
ITag::TFactory::TRegistrator<TCarComplaintTag> TCarComplaintTag::Registrator(TCarComplaintTag::TypeName);
TCarComplaintTag::TDescription::TFactory::TRegistrator<TCarComplaintTag::TDescription> TCarComplaintTag::TDescription::Registrator(TCarComplaintTag::TypeName);

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TCarComplaintConfig& config) {
    bool lostItemParsed = true;
    const auto& lostItem = value["lost_item"];
    if (lostItem.IsDefined()) {
        lostItemParsed &= NJson::ParseField(lostItem, "type", NJson::Stringify(config.LostItemType), false);
        lostItemParsed &= NJson::ParseField(lostItem, "description", config.ItemDescription, false);
        lostItemParsed &= NJson::ParseField(lostItem, "document", config.Document, false);
    }

    return
        lostItemParsed &&
        NJson::ParseField(value, "complaint_id", config.Id, true) &&
        NJson::ParseField(value, "description", config.Description) &&
        NJson::ParseField(value, "complaint_tag", config.TagName, true) &&
        NJson::ParseField(value, "is_user_available", config.IsUserAvailable, false) &&
        NJson::ParseField(value, "location", config.Location, false) &&
        NJson::ParseField(value, "complaint_weight", config.Weight, false);
}

template<>
NJson::TJsonValue NJson::ToJson(const TCarComplaintTag::TLostItemSource& source) {
    NJson::TJsonValue json;
    source.SerializeToJson(json);
    return json;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TCarComplaintTag::TLostItemSource& source) {
    return source.DeserializeFromJson(value);
}

void TCarComplaintTag::TComplaintSource::SerializeToJson(NJson::TJsonValue &json) const {
    NJson::InsertField(json, "type", NJson::Stringify(Type));
    NJson::InsertField(json, "added_weight", AddedWeight);
    NJson::InsertField(json, "topic_link", TopicLink);
    NJson::InsertField(json, "source_name", SourceName);
    NJson::InsertField(json, "image_paths", ImagePaths);
    NJson::InsertField(json, "comment", Comment);
    NJson::InsertField(json, "user_id", UserId);
    NJson::InsertField(json, "is_confirmed", Confirmed);
    NJson::InsertField(json, "created_at", CreatedAt);
    NJson::InsertField(json, "location", Location);
    NJson::InsertField(json, "is_user_available", UserAvailable);
}

void TCarComplaintTag::TLostItemSource::SerializeToJson(NJson::TJsonValue &json) const {
    TBase::SerializeToJson(json);
    NJson::InsertField(json, "lost_item_type", NJson::Stringify(LostItemType));
    NJson::InsertField(json, "item_description", ItemDescription);
    NJson::InsertField(json, "is_document", Document);
}

bool TCarComplaintTag::TComplaintSource::DeserializeFromJson(const NJson::TJsonValue &json) {
    return
        NJson::ParseField(json, "type", NJson::Stringify(Type), false) &&
        NJson::ParseField(json, "created_at", CreatedAt, false) &&
        NJson::ParseField(json, "added_weight", AddedWeight, false) &&
        NJson::ParseField(json, "topic_link", TopicLink) &&
        NJson::ParseField(json, "source_name", SourceName, true) &&
        NJson::ParseField(json, "image_paths", ImagePaths) &&
        NJson::ParseField(json, "comment", Comment) &&
        NJson::ParseField(json, "is_confirmed", Confirmed, false) &&
        NJson::ParseField(json, "user_id", UserId) &&
        NJson::ParseField(json, "location", Location) &&
        NJson::ParseField(json, "is_user_available", UserAvailable, false);
}

bool TCarComplaintTag::TLostItemSource::DeserializeFromJson(const NJson::TJsonValue &json) {
    return
        TBase::DeserializeFromJson(json) &&
        NJson::ParseField(json, "lost_item_type", NJson::Stringify(LostItemType)) &&
        NJson::ParseField(json, "item_description", ItemDescription) &&
        NJson::ParseField(json, "is_document", Document, false);
}

template<>
NJson::TJsonValue NJson::ToJson(const TCarComplaintTag::TAction& action) {
    NJson::TJsonValue result;
    NJson::InsertField(result, "action_type", NJson::Stringify(action.GetActionType()));
    NJson::InsertField(result, "from_status", NJson::Stringify(action.GetFromStatus()));
    NJson::InsertField(result, "to_status", NJson::Stringify(action.GetToStatus()));
    NJson::InsertField(result, "sources", action.GetSources());
    NJson::InsertField(result, "object", action.GetObject());
    NJson::InsertField(result, "tag", action.GetTag());
    NJson::InsertField(result, "topic_link", action.GetTopicLink());
    NJson::InsertField(result, "node_id", action.GetNodeId());
    NJson::InsertField(result, "only_for_new_sources", action.GetOnlyForNewSources());
    NJson::InsertField(result, "notifier_name", action.GetNotifierName());
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TCarComplaintTag::TAction& action) {
    if (!NJson::ParseField(value, "action_type", NJson::Stringify(action.MutableActionType()))
        || !NJson::ParseField(value, "tag", action.MutableTag())
        || !NJson::ParseField(value, "topic_link", action.MutableTopicLink())
        || !NJson::ParseField(value, "node_id", action.MutableNodeId())) {
        return false;
    }
    if (action.GetActionType() == TCarComplaintTag::EActionType::AddTag && action.GetTag().empty()) {
        return false;
    } else if (action.GetActionType() == TCarComplaintTag::EActionType::MoveChatToNode && action.GetNodeId().empty()) {
        return false;
    }
    return
        NJson::ParseField(value, "from_status", NJson::Stringify(action.MutableFromStatus())) &&
        NJson::ParseField(value, "to_status", NJson::Stringify(action.MutableToStatus())) &&
        NJson::ParseField(value, "sources", action.MutableSources()) &&
        NJson::ParseField(value, "object", action.MutableObject(), true) &&
        NJson::ParseField(value, "only_for_new_sources", action.MutableOnlyForNewSources(), false) &&
        NJson::ParseField(value, "notifier_name", action.MutableNotifierName());
}

NDrive::TScheme TCarComplaintTag::TComplaintSource::GetScheme(const TDescription& /* description */) const {
    NDrive::TScheme result;
    result.Add<TFSVariants>("type", "Тип источника")
        .InitVariants<ESourceType>()
        .SetDefault(::ToString(ESourceType::Simple))
        .SetReadOnly(true);
    result.Add<TFSNumeric>("added_weight", "Вес обращения");
    result.Add<TFSString>("topic_link", "Полный id чата-источника обращения");
    result.Add<TFSString>("source_name", "Название источника обращения");
    result.Add<TFSString>("user_id", "id обратившегося пользователя");
    result.Add<TFSBoolean>("is_user_available", "Пользователь доступен");
    result.Add<TFSBoolean>("is_confirmed", "Источник подтвержден");
    result.Add<TFSArray>("image_paths", "Фотографии").SetElement<TFSString>();
    result.Add<TFSString>("location", "Местоположение");
    result.Add<TFSString>("comment", "Комментарий");
    return result;
}

NDrive::TScheme TCarComplaintTag::TLostItemSource::GetScheme(const TDescription& description) const {
    auto result = TBase::GetScheme(description);
    auto* type = Yensured(dynamic_cast<TFSVariants*>(result.Get("type").Get()));
    type->SetDefault(::ToString(ESourceType::LostItem));

    result.Add<TFSVariants>("lost_item_type", "Тип пропажи").SetVariants({
            ::ToString(ELostItemType::ForgottenItem),
            ::ToString(ELostItemType::FoundItem)
        })
        .SetDefault(::ToString(description.GetDefaultLostItem()))
        .SetReadOnly(true);
    result.Add<TFSString>("item_description", "Описание предмета");
    result.Add<TFSBoolean>("is_document", "Документ");
    return result;
}

TString TCarComplaintTag::TAction::GetUnescapedTopicLink(const TDBTag& self) const {
    TString result = GetTopicLink();
    SubstGlobal(result, "<current_timestamp>", ToString(TInstant::Now().Seconds()));
    SubstGlobal(result, "<tag_id>", self.GetTagId());
    SubstGlobal(result, "<car_id>", self.GetObjectId());
    return result;
}

TString TCarComplaintTag::TAction::GetUnescapedTag(const TDBTag& self) const {
    TString result = GetTag();
    SubstGlobal(result, "<current_timestamp>", ToString(TInstant::Now().Seconds()));
    SubstGlobal(result, "<tag_id>", self.GetTagId());
    SubstGlobal(result, "<car_id>", self.GetObjectId());
    return result;
}

NDrive::TScheme TCarComplaintTag::TAction::GetScheme() {
    NDrive::TScheme result;
    result.Add<TFSVariants>("from_status", "При переходе из статуса").InitVariants<EComplaintStatus>();
    result.Add<TFSVariants>("to_status", "При переходе в статус").InitVariants<EComplaintStatus>();
    result.Add<TFSArray>("sources", "При наличии источников").SetElement<TFSString>();
    result.Add<TFSVariants>("object", "С кем выполнять (можно указать id пользователя)").InitVariants<EObjectType>().SetEditable(true);
    result.Add<TFSVariants>("action_type", "Тип действия").InitVariants<EActionType>().SetEditable(false);
    result.Add<TFSText>("tag", "Какой тег навесить");
    result.Add<TFSString>("topic_link", "Полный id чата, можно не указывать для пожаловавшихся");
    result.Add<TFSString>("node_id", "В какую ноду перевести чат");
    result.Add<TFSString>("notifier_name", "Имя notify чата");
    result.Add<TFSBoolean>("only_for_new_sources", "Только для новых источников").SetDefault(false);
    return result;
}

NDrive::TScheme TCarComplaintTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSNumeric>("min_complaint_weight", "Суммарный вес для перевода в confirmed");
    result.Add<TFSString>("max_session_age", "Искать сессии, которые были завершены не позже, чем");
    result.Add<TFSArray>("status_changed_actions", "Действия при смене статуса").SetElement<TFSStructure>().SetStructure(TAction::GetScheme());

    result.Add<TFSVariants>("default_lost_item").InitVariants<ELostItemType>();
    return result;
}

NJson::TJsonValue TCarComplaintTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
    NJson::InsertField(jsonMeta, "min_complaint_weight", MinComplaintWeight);
    NJson::InsertField(jsonMeta, "max_session_age", NJson::Hr(MaxSessionAge));
    NJson::InsertField(jsonMeta, "status_changed_actions", StatusChangedActions);

    NJson::InsertField(jsonMeta, "default_lost_item", NJson::Stringify(DefaultLostItem));

    return jsonMeta;
}

bool TCarComplaintTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    return
        NJson::ParseField(jsonMeta, "min_complaint_weight", MinComplaintWeight, false) &&
        NJson::ParseField(jsonMeta, "max_session_age", NJson::Hr(MaxSessionAge), false) &&
        NJson::ParseField(jsonMeta, "status_changed_actions", StatusChangedActions) &&
        NJson::ParseField(jsonMeta, "default_lost_item", NJson::Stringify(MutableDefaultLostItem()));
}

bool TCarComplaintTag::MoveChatToNode(const TCarComplaintTag* toTag, const TDBTag& self, const TCarComplaintTag::TAction& action, const NDrive::IServer& server, NDrive::TEntitySession& chatSession) const {
    TVector<TContextMapInfo> contextMap;
    if (!toTag->GetProblemUserId().empty()) {
        contextMap.push_back(TContextMapInfo("complaint_user_id", toTag->GetProblemUserId()));
    }
    contextMap.push_back(TContextMapInfo("complaint_car_id", self.GetObjectId()));
    contextMap.push_back(TContextMapInfo("complaint_tag_id", self.GetTagId()));
    TCarComplaintTag::EObjectType objectType;
    if (TryFromString(action.GetObject(), objectType)) {
        if (objectType == TCarComplaintTag::EObjectType::ComplainedUsers) {
            for (auto&& complaintSource : toTag->GetComplaintSources()) {
                if (!action.GetSources().empty() && !action.GetSources().contains(complaintSource->GetSourceName())) {
                    continue;
                }
                if (action.IsOnlyForNewSources() && !complaintSource->IsNew()) {
                    continue;
                }
                TString topicLink = action.GetTopicLink().empty() ? complaintSource->GetTopicLink() : action.GetUnescapedTopicLink(self);
                if (!IChatRobotImpl::MoveToStep(complaintSource->GetUserId(), action.GetNodeId(), topicLink, server, chatSession, contextMap)) {
                    return false;
                }
            }
        } else if (objectType == TCarComplaintTag::EObjectType::ProblemUser) {
            if (!IChatRobotImpl::MoveToStep(ProblemUserId, action.GetNodeId(), action.GetUnescapedTopicLink(self), server, chatSession, contextMap)) {
                return false;
            }
        }
    } else {
        if (!IChatRobotImpl::MoveToStep(action.GetObject(), action.GetNodeId(), action.GetUnescapedTopicLink(self), server, chatSession, contextMap)) {
            return false;
        }
    }
    return true;
}

bool TCarComplaintTag::AddTag(const TCarComplaintTag* toTag, const TDBTag& self, const TString& actorUserId, const TCarComplaintTag::TAction& action, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    ITag::TPtr tagToAdd = IJsonSerializableTag::BuildFromString(server.GetDriveAPI()->GetTagsManager(), action.GetUnescapedTag(self));
    if (!tagToAdd) {
        session.SetErrorInfo("TCarComplaintTag::AddTag", "cannot construct tag");
        return false;
    }
    TVector<TString> userIds;
    TCarComplaintTag::EObjectType objectType;
    if (TryFromString(action.GetObject(), objectType)) {
        if (objectType == TCarComplaintTag::EObjectType::ComplainedUsers) {
            for (auto&& complaintSource : toTag->GetComplaintSources()) {
                if (!action.GetSources().empty() && !action.GetSources().contains(complaintSource->GetSourceName())) {
                    continue;
                }
                if (action.IsOnlyForNewSources() && !complaintSource->IsNew()) {
                    continue;
                }
                if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tagToAdd, actorUserId, complaintSource->GetUserId(), &server, session)) {
                    return false;
                }
            }
        } else if (objectType == TCarComplaintTag::EObjectType::ProblemUser) {
            if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tagToAdd, actorUserId, toTag->GetProblemUserId(), &server, session)) {
                return false;
            }
        } else if (objectType == TCarComplaintTag::EObjectType::Car) {
            if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tagToAdd, actorUserId, self.GetObjectId(), &server, session)) {
                return false;
            }
        }
    } else if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tagToAdd, actorUserId, action.GetObject(), &server, session)) {
        return false;
    }
    return true;
}

void TCarComplaintTag::FillUserReport(TStringBuilder& report, const TUsersDB::TOptionalUser& user, const TString& userAlias, const bool available) {
    if (!user) {
        report << userAlias << "нет информации" << Endl << Endl;
        return;
    }

    report << userAlias << user->GetFullName() << Endl;
    report << "Телефон: " << user->GetPhone() << Endl;
    report << "Доступен: " << ::ToString(available ? "Да" : "Нет") << Endl;
    report << "Ссылка на профиль: " << TCarsharingUrl().ClientInfo(user->GetUserId()) << Endl << Endl;
}

bool TCarComplaintTag::Notify(const TAction& action, const TDBTag& self, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    const auto notifier = server.GetNotifier(action.GetNotifierName());
    if (!notifier) {
        session.SetErrorInfo("TCarComplaintTag::Notify", "Notifier " + action.GetNotifierName() + " not found");
        return false;
    }

    const auto* driveApi = server.GetDriveAPI();
    if (!driveApi) {
        session.SetErrorInfo("TCarComplaintTag::Notify", "DriveAPI is not defined");
        return false;
    }

    const auto* userData = driveApi->GetUsersData();
    if (!userData) {
        session.SetErrorInfo("TCarComplaintTag::Notify", "UserData is not defined");
        return false;
    }

    const auto* description = GetDescription(server);
    if (!description) {
        session.SetErrorInfo("TCarComplaintTag::Notify", "Failed to get description");
        return false;
    }
    TStringBuilder reportHead;
    {
        switch (description->GetDefaultLostItem()) {
            case ELostItemType::ForgottenItem:
                reportHead << "Потерян предмет";
                break;
            case ELostItemType::FoundItem:
                reportHead << "Найден предмет";
                break;
            default:
                reportHead << "Жалоба";
        }
        reportHead << Endl << Endl;
    }

    TUsersDB::TOptionalUser problemUser = ProblemUserId ? userData->RestoreUser(ProblemUserId, session) : Nothing();
    FillUserReport(reportHead, problemUser, GetProblemUserAlias(*description), ProblemUserAvailable);

    TString carNumber;
    {
        const auto* carsData = driveApi->GetCarsData();
        if (!carsData) {
            session.SetErrorInfo("TCarComplaintTag::Notify", "CarNumbers is not defined");
            return false;
        }
        auto carInfo = carsData->FetchInfo(self.GetObjectId(), session);
        if (!carInfo || carInfo.size() != 1) {
            session.SetErrorInfo("TCarComplaintTag::Notify", "Failed to get car info");
            return false;
        }
        carNumber = carInfo.begin()->second.GetNumber();
        reportHead << "Номер машины: " << carNumber << Endl << Endl;
    }


    for (const auto& source: ComplaintSources) {
        TStringBuilder report = source->BuildReport(reportHead, userData, session);

        TString binaryImage;
        if (!source->GetImagePaths().empty() && driveApi->HasMDSClient()) {
            const auto& mdsClient = driveApi->GetMDSClient();

            TString tmpImage;
            TString imagePath;
            TString bucketName;
            TMessagesCollector errors;

            const TS3Client::TBucket* bucket;
            if (mdsClient.ParseMdsLink(*source->GetImagePaths().begin(), errors, bucketName, imagePath) &&
                (bucket = mdsClient.GetBucket(bucketName)) &&
                (bucket->GetFile(imagePath, binaryImage, errors) / 100 == 2))
            {
                binaryImage = tmpImage;
            }
        }

        if (binaryImage) {
            NDrive::INotifier::TMessage message(report.Data(), binaryImage);
            NDrive::INotifier::SendPhoto(notifier, message);
        } else {
            NDrive::INotifier::Notify(notifier, report);
        }
    }

    return true;
}

TStringBuilder TCarComplaintTag::TComplaintSource::BuildReport(const TStringBuilder& reportHead, const TUsersDB* userData, NDrive::TEntitySession& session) const {
    TStringBuilder report;
    report << reportHead.copy();

    TUsersDB::TOptionalUser sourceUser =  UserId ? userData->RestoreUser(UserId, session) : Nothing();
    TCarComplaintTag::FillUserReport(report, sourceUser, GetUserAlias(), UserAvailable);

    report << "Подтверждено: " << ::ToString(Confirmed ? "Да" : "Нет") << Endl;
    report << "Комментарий: " << Comment << Endl;
    report << "Местоположение: " << Location << Endl;
    report << "Ссылки на фото: " << Endl << JoinSeq('\n', ImagePaths) << Endl;

    return report;
}

TStringBuilder TCarComplaintTag::TLostItemSource::BuildReport(const TStringBuilder& reportHead, const TUsersDB* userData, NDrive::TEntitySession& session) const {
    TStringBuilder report = TBase::BuildReport(reportHead, userData, session);

    report << "Описание предмета: " << ItemDescription << Endl;
    report << "Документ: " << ::ToString(Document ? "Да" : "Нет") << Endl;

    return report;
}

TString TCarComplaintTag::GetProblemUserAlias(const TCarComplaintTag::TDescription& description) const {
    TString problemUserAlias = "Проблемный пользователь: ";
    if (description.GetDefaultLostItem() != ELostItemType::None) {
        problemUserAlias = description.GetDefaultLostItem() == ELostItemType::ForgottenItem
            ? "Нашедший пользователь: " : "Потерявший пользователь: ";
    }
    return problemUserAlias;
}

const TCarComplaintTag::TDescription* TCarComplaintTag::GetDescription(const NDrive::IServer& server) const {
    auto baseDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    return baseDescription ? baseDescription->GetAs<TCarComplaintTag::TDescription>() : nullptr;
}

bool TCarComplaintTag::PerformStatusChangedActions(const TCarComplaintTag* tag, const TDBTag& self, const TString& actorUserId, TCarComplaintTag::EComplaintStatus fromStatus, const NDrive::IServer& server,  NDrive::TEntitySession& session) const {
    auto chatEngine = server.GetChatEngine();
    if (Yensured(server.GetDriveAPI())->GetDatabaseName() == Yensured(chatEngine)->GetDatabaseName()) {
        return PerformStatusChangedActions(tag, self, actorUserId, fromStatus, tag->GetStatus(), server, session, session);
    } else {
        auto chatSession = chatEngine->BuildSession(false);
        if (!PerformStatusChangedActions(tag, self, actorUserId, fromStatus, tag->GetStatus(), server, session, chatSession) || !chatSession.Commit()) {
            if (!session.GetMessages().HasMessages()) {
                session.SetErrorInfo("TCarComplaintTag::PerformStatusChangedActions", "cannot perform status changed actions");
            }
            session.MergeErrorMessages(chatSession.GetMessages(), "TCarComplaintTag::PerformStatusChangedActions");
            return false;
        }
    }
    return true;
}

bool TCarComplaintTag::PerformStatusChangedActions(const TCarComplaintTag* toTag, const TDBTag& self, const TString& actorUserId, TCarComplaintTag::EComplaintStatus fromStatus, TCarComplaintTag::EComplaintStatus toStatus, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const {
    auto description = GetDescription(server);
    if (!description) {
        session.SetErrorInfo("TCarComplaintTag::PerformStatusChangedActions", "cannot get tag description");
        return false;
    }
    for (auto&& action : description->GetStatusChangedActions()) {
        if ((action.GetFromStatus() != TCarComplaintTag::EComplaintStatus::None && action.GetFromStatus() != fromStatus) || (action.GetToStatus() != TCarComplaintTag::EComplaintStatus::None && action.GetToStatus() != toStatus)) {
            continue;
        }
        bool matchedSources = true;
        if (!action.GetSources().empty() || action.IsOnlyForNewSources()) {
            matchedSources = false;
            for (auto&& source : toTag->GetComplaintSources()) {
                matchedSources |= !action.GetSources().empty() && action.GetSources().contains(source->GetSourceName());
                matchedSources |= action.IsOnlyForNewSources() && source->IsNew();
            }
        }
        if (!matchedSources) {
            continue;
        }
        switch(action.GetActionType()) {
            case EActionType::AddTag:
                if (!AddTag(toTag, self, actorUserId, action, server, session)) {
                    return false;
                }
                break;
            case EActionType::MoveChatToNode:
                if (!MoveChatToNode(toTag, self, action, server, chatSession)) {
                    return false;
                }
                break;
            case EActionType::Notify:
                if (!Notify(action, self, server, chatSession)) {
                    return false;
                }
                break;
        }
    }
    return true;
}

void UpdateStatusIfConfirmed(TCarComplaintTag& toTag, TCarComplaintTag::EComplaintStatus fromStatus, const TCarComplaintTag::TDescription& description) {
    TSet<TCarComplaintTag::EComplaintStatus> autoUpdatedStatuses = {
            TCarComplaintTag::EComplaintStatus::None,
            TCarComplaintTag::EComplaintStatus::Created,
            TCarComplaintTag::EComplaintStatus::ConfirmedNoUser,
            TCarComplaintTag::EComplaintStatus::ConfirmedWithUser
    };
    if (!autoUpdatedStatuses.contains(toTag.GetStatus())) {
        return;
    }
    bool isConfirmed = false;
    for (auto&& source : toTag.GetComplaintSources()) {
        if (source->IsConfirmed()) {
            isConfirmed = true;
            break;
        }
    }
    if (isConfirmed || toTag.GetWeight() >= description.GetMinComplaintWeight() && fromStatus == TCarComplaintTag::EComplaintStatus::Created || fromStatus == TCarComplaintTag::EComplaintStatus::ConfirmedNoUser) {
        if (toTag.GetProblemUserId().empty()) {
            toTag.SetStatus(TCarComplaintTag::EComplaintStatus::ConfirmedNoUser);
        } else {
            toTag.SetStatus(TCarComplaintTag::EComplaintStatus::ConfirmedWithUser);
        }
    }
}

void MergeTagData(const TCarComplaintTag* fromTag, TCarComplaintTag* toTag, const NDrive::IServer* server) {
    if (fromTag->HasSLAInstant()) {
        toTag->SetSLAInstant(fromTag->GetSLAInstantUnsafe());
    }
    if (toTag->GetProblemSessionId().empty()) {
        toTag->SetProblemSessionId(fromTag->GetProblemSessionId());
    }
    if (toTag->GetProblemUserId().empty()) {
        toTag->SetProblemUserId(fromTag->GetProblemUserId());
    }
    if (toTag->GetWeight() == 0) {
        toTag->SetWeight(fromTag->GetWeight());
    }
    if (toTag->GetStatus() == TCarComplaintTag::EComplaintStatus::None) {
        toTag->SetStatus(fromTag->GetStatus());
    }
    if (toTag->GetStatus() == TCarComplaintTag::EComplaintStatus::Denied && fromTag->IsComplaintConfirmed()) {
        toTag->SetStatus(fromTag->GetStatus());
    }
    for (auto&& existingSource : fromTag->GetComplaintSources()) {
        bool updatedSource = false;
        for (auto& newSource : toTag->GetComplaintSources()) {
            if (existingSource->GetUserId() == newSource->GetUserId() && existingSource->GetSourceName() == newSource->GetSourceName()) {
                updatedSource = true;
                newSource->SetConfirmed(existingSource->IsConfirmed() | newSource->IsConfirmed());
                newSource->MutableImagePaths().insert(existingSource->GetImagePaths().begin(), existingSource->GetImagePaths().end());
                newSource->SetNew(false);
                break;
            }
        }
        if (!updatedSource) {
            auto newSource = existingSource->DeepCopy();
            newSource->SetNew(false);
            toTag->GetComplaintSources().push_back(std::move(newSource));
        }
    }
    toTag->SetWeight(0);
    for (auto&& source : toTag->GetComplaintSources()) {
        toTag->MutableWeight() += source->GetAddedWeight();
    }
    auto description = toTag->GetDescription(*server);
    if (description) {
        UpdateStatusIfConfirmed(*toTag, fromTag->GetStatus(), *description);
    }
}

bool TCarComplaintTag::IsComplaintConfirmed() const {
    return Status == TCarComplaintTag::EComplaintStatus::ConfirmedWithUser || Status == TCarComplaintTag::EComplaintStatus::ConfirmedNoUser;
}

bool TCarComplaintTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeAdd(objectId, userId, server, session)) {
        return false;
    }
    TVector<TDBTag> tags;
    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({objectId}, {GetName()}, tags, session)) {
        return false;
    }
    if (tags.size() > 1) {
        session.SetErrorInfo("TCarComplaintTag::OnBeforeAdd", "more than one tag on object " + GetName());
        return false;
    }

    if (tags.size() == 0) {
        if (Status == TCarComplaintTag::EComplaintStatus::None) {
            SetStatus(TCarComplaintTag::EComplaintStatus::Created);
        }
        for (auto&& complaintSource : GetComplaintSources()) {
            Weight += complaintSource->GetAddedWeight();
        }
        auto description = GetDescription(*server);
        if (!description) {
            session.SetErrorInfo("TCarComplaintTag::OnBeforeAdd", "cannot get tag description");
            return false;
        }
        TInstant since = description->GetMaxSessionAge() ? Now() - description->GetMaxSessionAge() : TInstant::Zero();
        THistoryRidesContext ridesContext(*server, since);
        auto ydbTx = server->GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("car_complaint_tag", server);
        if (!ridesContext.InitializeCars({ objectId }, session, ydbTx)) {
            return false;
        }
        TVector<THistoryRideObject> rides = ridesContext.GetSessions(Now(), 100);
        if (!rides.empty()) {
            SetProblemUserId(rides.front().GetUserId());
            SetProblemSessionId(rides.front().GetSessionId());
        }
        UpdateStatusIfConfirmed(*this, Status, *description);
    } else {
        const auto* tagImpl = tags[0].GetTagAs<TCarComplaintTag>();
        if (!tagImpl) {
            session.SetErrorInfo("TCarComplaintTag::OnBeforeAdd", "cannot get tag description");
            return false;
        }
        MergeTagData(tagImpl, this, server);
    }
    return true;
}

bool TCarComplaintTag::OnBeforeUpdate(const TDBTag& self, ITag::TPtr toTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    Y_UNUSED(userId);
    Y_UNUSED(session);
    const auto* from = self.GetTagAs<TCarComplaintTag>();
    auto* to = dynamic_cast<TCarComplaintTag*>(toTag.Get());
    if (!from || !to) {
        session.SetErrorInfo("TCarComplaintTag::OnBeforeUpdate", "cannot cast tag");
        return false;
    }
    MergeTagData(from, to, server);
    return true;
}

bool TCarComplaintTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto* to = self.GetTagAs<TCarComplaintTag>();
    if (!to) {
        session.SetErrorInfo("TCarComplaintTag::OnAfterAdd", "cannot cast tag");
        return false;
    }
    return PerformStatusChangedActions(to, self, userId, TCarComplaintTag::EComplaintStatus::Created, *server, session);
}

bool TCarComplaintTag::OnAfterUpdate(const TDBTag& self, ITag::TPtr toTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto from = self.GetTagAs<TCarComplaintTag>();
    auto to = dynamic_cast<const TCarComplaintTag*>(toTag.Get());
    if (!from || !to) {
        session.SetErrorInfo("TCarComplaintTag::OnAfterUpdate", "cannot cast tag");
        return false;
    }
    return PerformStatusChangedActions(to, self, userId, from->GetStatus(), *server, session);
}

NDrive::TScheme TCarComplaintTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);

    auto description = GetDescription(*server);
    Y_ENSURE(description);

    auto& sources = result.Add<TFSArray>("complaint_sources", "Источники информации");
    if (description->GetDefaultLostItem() != ELostItemType::None) {
        TLostItemSource source;
        source.SetLostItemType(description->GetDefaultLostItem());

        sources.SetElement(source.GetScheme(*description));
        sources.AddDefault<TLostItemSource>(source);
        sources.SetRequired(true);
    } else {
        sources.SetElement(TComplaintSource().GetScheme(*description));
    }

    result.Add<TFSString>("problem_session_id", "Cессия для исходящей связи");
    result.Add<TFSString>("problem_user_id", "Пользователь для исходящей связи");
    result.Add<TFSBoolean>("is_problem_user_available", "Пользователь доступен");
    result.Add<TFSNumeric>("weight", "Суммарный вес жалоб");
    result.Add<TFSVariants>("status", "Статус").InitVariants<EComplaintStatus>().SetEditable(false);
    return result;
}

void TCarComplaintTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    for (const auto& source: ComplaintSources) {
        NJson::TJsonValue sourceJson;
        source->SerializeToJson(sourceJson);
        json["complaint_sources"].AppendValue(std::move(sourceJson));
    }
    NJson::InsertField(json, "problem_session_id", ProblemSessionId);
    NJson::InsertField(json, "problem_user_id", ProblemUserId);
    NJson::InsertField(json, "is_problem_user_available", ProblemUserAvailable);
    NJson::InsertField(json, "weight", Weight);
    NJson::InsertField(json, "status", NJson::Stringify(Status));
}

bool TCarComplaintTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(json, errors)) {
        return false;
    }

    for (const auto& sourceJson: json["complaint_sources"].GetArray()) {
        auto sourceType = DeductSourceTypeFromJson(sourceJson);
        auto source = BuildComplaintSource(sourceType);
        if (!source || !source->DeserializeFromJson(sourceJson)) {
            return false;
        }
        ComplaintSources.push_back(std::move(source));
    }

    return
        NJson::ParseField(json, "problem_session_id", ProblemSessionId, false, errors) &&
        NJson::ParseField(json, "problem_user_id", ProblemUserId, false, errors) &&
        NJson::ParseField(json, "is_problem_user_available", ProblemUserAvailable, false, errors) &&
        NJson::ParseField(json, "weight", Weight, false, errors) &&
        NJson::ParseField(json, "status", NJson::Stringify(Status), false, errors);
}

EUniquePolicy TCarComplaintTag::GetUniquePolicy() const {
    return EUniquePolicy::Rewrite;
}

TSet<NEntityTagsManager::EEntityType> TCarComplaintTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::Car };
}

void TCarComplaintTag::TComplaintSource::SerializeToProto(NDrive::NProto::TCarComplaintTag::TComplaintSource* proto) const {
    proto->SetType(::ToString(Type));
    proto->SetUserId(UserId);
    proto->SetTopicLink(TopicLink);
    proto->SetSourceName(SourceName);
    for (auto&& imagePath : ImagePaths) {
        proto->AddImagePaths(imagePath);
    }
    proto->SetComment(Comment);
    proto->SetAddedWeight(AddedWeight);
    proto->SetConfirmed(Confirmed);
    proto->SetCreatedAt(CreatedAt.Seconds());
    proto->SetLocation(Location);
    proto->SetUserAvailable(UserAvailable);
}

void TCarComplaintTag::TLostItemSource::SerializeToProto(NDrive::NProto::TCarComplaintTag::TComplaintSource* proto) const {
    TBase::SerializeToProto(proto);
    NDrive::NProto::TCarComplaintTag::TLostItem* protoLostItem = proto->AddLostItem();
    protoLostItem->SetType(::ToString(LostItemType));
    protoLostItem->SetItemDescription(ItemDescription);
    protoLostItem->SetDocument(Document);
}

TCarComplaintTag::TProto TCarComplaintTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TCarComplaintTag proto = TBase::DoSerializeSpecialDataToProto();
    for (const auto& source : GetComplaintSources()) {
        NDrive::NProto::TCarComplaintTag::TComplaintSource* protoInfo = proto.AddComplaintSources();
        source->SerializeToProto(protoInfo);
    }
    proto.SetProblemSessionId(GetProblemSessionId());
    proto.SetProblemUserId(GetProblemUserId());
    proto.SetProblemUserAvailable(GetProblemUserAvailable());
    proto.SetWeight(GetWeight());
    proto.SetStatus(ToString(GetStatus()));
    return proto;
}

bool TCarComplaintTag::TComplaintSource::DeserializeFromProto(const TCarComplaintTag::TProto::TComplaintSource& proto) {
    UserId = proto.GetUserId();
    TopicLink = proto.GetTopicLink();
    SourceName = proto.GetSourceName();
    for (auto&& imagePath : proto.GetImagePaths()) {
        ImagePaths.insert(imagePath);
    }
    Comment = proto.GetComment();
    AddedWeight = proto.GetAddedWeight();
    Confirmed = proto.GetConfirmed();
    CreatedAt = TInstant::Seconds(proto.GetCreatedAt());
    Location = proto.GetLocation();
    UserAvailable = proto.GetUserAvailable();
    return true;
}

bool TCarComplaintTag::TLostItemSource::DeserializeFromProto(const TCarComplaintTag::TProto::TComplaintSource &proto) {
    if (!TBase::DeserializeFromProto(proto) || proto.GetLostItem().empty()) {
        return false;
    }

    for (const auto& protoLostItem: proto.GetLostItem()) {
        LostItemType = ::FromString(protoLostItem.GetType());
        ItemDescription = protoLostItem.GetItemDescription();
        Document = protoLostItem.GetDocument();
    }

    return true;
}

bool TCarComplaintTag::DoDeserializeSpecialDataFromProto(const TCarComplaintTag::TProto& proto) {
    for (const auto& protoSource : proto.GetComplaintSources()) {
        ESourceType type = DeductSourceTypeFromProto(protoSource);
        auto source = BuildComplaintSource(type);
        if (!source || !source->DeserializeFromProto(protoSource)) {
            return false;
        }
        ComplaintSources.push_back(std::move(source));
    }
    SetProblemSessionId(proto.GetProblemSessionId());
    SetProblemUserId(proto.GetProblemUserId());
    SetProblemUserAvailable(proto.GetProblemUserAvailable());
    SetWeight(proto.GetWeight());
    if (!TryFromString(proto.GetStatus(), Status)) {
        return false;
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

TCarComplaintTag::ESourceType TCarComplaintTag::DeductSourceTypeFromJson(const NJson::TJsonValue& json) const {
    ESourceType type;
    if (json["type"].IsString() && ::TryFromString(json["type"].GetString(), type)) {
        return type;
    }

    return ESourceType::Simple;
}

TCarComplaintTag::ESourceType TCarComplaintTag::DeductSourceTypeFromProto(const TCarComplaintTag::TProto::TComplaintSource& proto) const {
    ESourceType type;
    if (proto.HasType() && ::TryFromString(proto.GetType(), type)) {
        return type;
    }

    return ESourceType::Simple;
}

THolder<TCarComplaintTag::TComplaintSource> TCarComplaintTag::BuildComplaintSource(const TCarComplaintTag::ESourceType type) const {
    THolder<TCarComplaintTag::TComplaintSource> source;

    switch (type) {
        case TCarComplaintTag::ESourceType::LostItem:
            source = MakeHolder<TCarComplaintTag::TLostItemSource>();
            break;
        default:
            source = MakeHolder<TCarComplaintTag::TComplaintSource>();
    }

    return source;
}
