#pragma once

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/chat_robots/configuration/context.h>
#include <drive/backend/data/proto/tags.pb.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tag_description.h>

#include <rtline/util/types/accessor.h>

class TCarComplaintTag: public ISerializableTag<NDrive::NProto::TCarComplaintTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TCarComplaintTag>;
    static TFactory::TRegistrator<TCarComplaintTag> Registrator;

public:
    using TBase::TBase;
    static const TString TypeName;
    class TDescription;

    enum class ELostItemType {
        None /* "none" */,
        FoundItem /* "found" */,
        ForgottenItem /* "forgotten" */,
    };

    enum class ESourceType {
        Simple /* "generic" */,
        LostItem /* "lost_item" */,
    };

    class TComplaintSource {
    protected:
        R_FIELD(ESourceType, Type);
        R_FIELD(TString, UserId);
        R_FIELD(TString, TopicLink);
        R_FIELD(TString, SourceName);
        R_FIELD(TSet<TString>, ImagePaths);
        R_FIELD(TString, Comment);
        R_FIELD(ui32, AddedWeight, 0);
        R_FIELD(bool, Confirmed, false);
        R_FIELD(TInstant, CreatedAt);
        R_FIELD(bool, New, true);
        R_FIELD(TString, Location);
        R_FIELD(bool, UserAvailable, false);

    public:
        TComplaintSource() : Type(ESourceType::Simple) {
        }

    protected:
        TComplaintSource(ESourceType type) : Type(type) {
        }

    public:

        virtual ~TComplaintSource() = default;

        virtual NDrive::TScheme GetScheme(const TDescription& description) const;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json);
        virtual void SerializeToJson(NJson::TJsonValue& json) const;
        virtual bool DeserializeFromProto(const TCarComplaintTag::TProto::TComplaintSource& proto);
        virtual void SerializeToProto(NDrive::NProto::TCarComplaintTag::TComplaintSource* proto) const;

        virtual TStringBuilder BuildReport(const TStringBuilder& reportHead, const TUsersDB* userData, NDrive::TEntitySession& session) const;
        virtual TString GetUserAlias() const {
            return "Пожаловавшийся пользователь: ";
        }

        virtual THolder<TComplaintSource> DeepCopy() const {
            return MakeHolder<TComplaintSource>(*this);
        }
    };

    class TLostItemSource : public TComplaintSource {
    private:
        using TBase = TComplaintSource;

    private:
        R_FIELD(ELostItemType, LostItemType, ELostItemType::FoundItem);
        R_FIELD(TString, ItemDescription);
        R_FIELD(bool, Document, true);

    public:
        TLostItemSource() : TBase(ESourceType::LostItem) {
        }

        virtual NDrive::TScheme GetScheme(const TDescription& description) const override;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual void SerializeToJson(NJson::TJsonValue& json) const override;
        virtual bool DeserializeFromProto(const TCarComplaintTag::TProto::TComplaintSource& proto) override;
        virtual void SerializeToProto(NDrive::NProto::TCarComplaintTag::TComplaintSource* proto) const override;

        virtual TStringBuilder BuildReport(const TStringBuilder& reportHead, const TUsersDB* userData, NDrive::TEntitySession& session) const override;
        virtual TString GetUserAlias() const override {
            return LostItemType == ELostItemType::ForgottenItem ? "Потерявший пользователь: " : "Нашедший пользователь: ";
        }

        virtual THolder<TComplaintSource> DeepCopy() const override {
            return MakeHolder<TLostItemSource>(*this);
        }

    };

    enum class EComplaintStatus {
        None /* "none" */,
        Created /* "created" */,
        ConfirmedWithUser /* "confirmed_with_user" */,
        ConfirmedNoUser /* "confirmed_no_user" */,
        Escalated /* "escalated" */,
        Resolved /* "resolved" */,
        Denied /* "denied" */,
        FixedByUser /* "fixed_by_user" */,
    };

    enum class EActionType {
        AddTag /* "add_tag" */,
        MoveChatToNode /* "move_chat_to_node" */,
        Notify /* "notify" */,
    };

    enum class EObjectType {
        ProblemUser /* "problem_user" */,
        ComplainedUsers /* "complained_users" */,
        Car /* "car" */,
    };

    class TAction {
    private:
        R_FIELD(EActionType, ActionType);
        R_FIELD(EComplaintStatus, FromStatus);
        R_FIELD(EComplaintStatus, ToStatus);
        R_FIELD(TSet<TString>, Sources);
        R_FIELD(TString, Object);
        R_FIELD(TString, Tag);
        R_FIELD(TString, NodeId);
        R_FIELD(TString, TopicLink);
        R_FIELD(bool, OnlyForNewSources, false);
        R_FIELD(TString, NotifierName);

    public:
        TString GetUnescapedTopicLink(const TDBTag& self) const;
        TString GetUnescapedTag(const TDBTag& self) const;
        static NDrive::TScheme GetScheme();
    };

    class TDescription: public TTagDescription {
    private:
        R_FIELD(ui32, MinComplaintWeight, 1);
        R_FIELD(TDuration, MaxSessionAge, TDuration::Days(2));
        R_FIELD(TVector<TAction>, StatusChangedActions);
        R_FIELD(ELostItemType, DefaultLostItem);

        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

private:
    TVector<THolder<TComplaintSource>> ComplaintSources;

    R_FIELD(TString, ProblemSessionId);
    R_FIELD(TString, ProblemUserId);
    R_FIELD(bool, ProblemUserAvailable, false);
    R_FIELD(ui32, Weight, 0);
    R_FIELD(EComplaintStatus, Status, EComplaintStatus::None);

    static void FillUserReport(TStringBuilder& report, const TUsersDB::TOptionalUser& user, const TString& userAlias, const bool available);
    bool Notify(const TAction& action, const TDBTag& self, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
    TString GetProblemUserAlias(const TDescription& description) const;

    bool MoveChatToNode(const TCarComplaintTag* toTag, const TDBTag& self, const TAction& action, const NDrive::IServer& server, NDrive::TEntitySession& chatSession) const;
    bool AddTag(const TCarComplaintTag* toTag, const TDBTag& self, const TString& actorUserId, const TAction& action, const NDrive::IServer& server, NDrive::TEntitySession& session) const;
    bool PerformStatusChangedActions(const TCarComplaintTag* tag, const TDBTag& self, const TString& actorUserId, EComplaintStatus fromStatus, const NDrive::IServer& server,  NDrive::TEntitySession& session) const;
    bool PerformStatusChangedActions(const TCarComplaintTag* toTag, const TDBTag& self, const TString& actorUserId, EComplaintStatus fromStatus, EComplaintStatus toStatus, const NDrive::IServer& server, NDrive::TEntitySession& session, NDrive::TEntitySession& chatSession) const;

    ESourceType DeductSourceTypeFromJson(const NJson::TJsonValue& json) const;
    ESourceType DeductSourceTypeFromProto(const TCarComplaintTag::TProto::TComplaintSource& proto) const;
    THolder<TComplaintSource> BuildComplaintSource(const ESourceType type) const;

public:
    const TVector<THolder<TComplaintSource>>& GetComplaintSources() const {
        return ComplaintSources;
    }

    TVector<THolder<TComplaintSource>>& GetComplaintSources() {
        return ComplaintSources;
    }

    void SetComplaintSources(const TVector<TCarComplaintTag::TComplaintSource>& sources) {
        for (const auto& source: sources) {
            auto sourceHolder = source.DeepCopy();
            ComplaintSources.push_back(std::move(sourceHolder));
        }
    }

    void SetComplaintSources(TMap<TString, THolder<TCarComplaintTag::TComplaintSource>>&& sources) {
        for (auto&& [_, source]: sources) {
            ComplaintSources.push_back(std::move(source));
        }
    }

    virtual bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;
    virtual bool OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterUpdate(const TDBTag& self, ITag::TPtr toTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual EUniquePolicy GetUniquePolicy() const override;
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    const TDescription* GetDescription(const NDrive::IServer& server) const;
    bool IsComplaintConfirmed() const;
};


struct TCarComplaintConfig {
    TString Id;
    TString Description;
    TString TagName;
    ui32 Weight = 1;

    bool IsUserAvailable = false;
    TString Location;
    TCarComplaintTag::ELostItemType LostItemType = TCarComplaintTag::ELostItemType::None;
    TString ItemDescription;
    bool Document = false;
};
