#include "container_tag.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

const TString TUserContainerTag::TypeName = "container_tag";
const TString TCarContainerTag::TypeName = "car_container_tag";

bool IContainerTag::OnSLAExpired(const TDBTag& dbTag, const TUserPermissions& permissions, const IEntityTagsManager* manager, const NDrive::IServer* /*server*/, NDrive::TEntitySession& session) const {
    auto tag = RestoreTag();
    if (!tag) {
        session.SetErrorInfo("IContainerTag::OnSLAExpired", "Cannot restore tag: " + StoredTagType + " - " + dbTag.GetTagId(), EDriveSessionResult::InconsistencySystem);
        return false;
    }
    if (!manager->DirectEvolveTag(permissions.GetUserId(), dbTag, RestoreTag(), session)) {
        session.SetErrorInfo("IContainerTag::OnSLAExpired", "Incorrect tag type: " + StoredTagType + " - " + dbTag.GetTagId(), EDriveSessionResult::InconsistencySystem);
        return false;
    }
    return true;
}

NDrive::NProto::IContainerTag IContainerTag::DoSerializeSpecialDataToProto() const {
    TProto result = TBase::DoSerializeSpecialDataToProto();
    result.SetStoredTagData(StoredTagData);
    result.SetStoredTagType(StoredTagType);
    return result;
}

bool IContainerTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (!HasSLAInstant()) {
        return false;
    }
    StoredTagData = proto.GetStoredTagData();
    StoredTagType = proto.GetStoredTagType();
    return true;
}

ITag::TPtr IContainerTag::RestoreTag() const {
    THolder<ITag> storedTag(ITag::TFactory::Construct(StoredTagType));
    if (!storedTag) {
        return nullptr;
    }
    NStorage::TTableRecord tr;

    NJson::TJsonValue jsonTag;
    if (!NJson::ReadJsonFastTree(StoredTagData, &jsonTag)) {
        return nullptr;
    }

    if (!tr.DeserializeFromJson(jsonTag)) {
        return nullptr;
    }

    if (!TTagDecoder::DeserializeFromJson(*storedTag, jsonTag, &NDrive::GetServerAs<NDrive::IServer>().GetDriveDatabase().GetTagsManager().GetContext())) {
        return nullptr;
    }
    return storedTag.Release();
}

void IContainerTag::SetStoredTag(ITag::TConstPtr tag) {
    if (!tag) {
        StoredTagData = "";
        StoredTagType = "";
    } else {
        StoredTagData = tag->SerializeToTableRecord().SerializeToJson().GetStringRobust();
        StoredTagType = NDrive::GetServerAs<NDrive::IServer>().GetDriveDatabase().GetTagsManager().GetTagsMeta().GetTagTypeByName(tag->GetName());
    }
}

TMaybe<TDBTag> IContainerTag::UndeferContainer(const TDBTag& tag, TUserPermissionsConstPtr permissions, const NDrive::IServer* server, const IEntityTagsManager& tagsManager, NDrive::TEntitySession& session) {
    if (!tag->OnSLAExpired(tag, *permissions, &tagsManager, server, session)) {
        return Nothing();
    }
    auto optionalTag = tagsManager.RestoreTag(tag.GetTagId(), session);
    return optionalTag;
}

ITag::TFactory::TRegistrator<TUserContainerTag> TUserContainerTag::Registrator(TUserContainerTag::TypeName);
ITag::TFactory::TRegistrator<TCarContainerTag> TCarContainerTag::Registrator(TCarContainerTag::TypeName);
