#include "dedicated_fleet.h"

#include "user_tags.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/offers/actions/dedicated_fleet.h>

#include <rtline/library/json/field.h>
#include <rtline/library/json/proto/adapter.h>
#include <rtline/util/algorithm/type_traits.h>


const TString TSpecialFleetTag::TypeName = "special_fleet_tag";
ITag::TFactory::TRegistrator<TSpecialFleetTag> TSpecialFleetTag::Registrator(TSpecialFleetTag::TypeName);

const TString TSpecialFleetTag::Hold = "dedicated_fleet_hold";
const TString TSpecialFleetTag::Preparing = "dedicated_fleet_preparing";
const TString TSpecialFleetTag::Prepared = "dedicated_fleet_prepared";
const TString TSpecialFleetTag::Delivery = "dedicated_fleet_delivery";
const TString TSpecialFleetTag::Actual = "dedicated_fleet_actual";

bool TSpecialFleetTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeRemove(self, userId, server, session)) {
        return false;
    }

    if (OfferId.Empty() || UnitOfferId.Empty() || OfferHolderTagId.Empty()) {
        return true;
    }

    const auto& database = server->GetDriveDatabase();
    const auto& tagsManager = database.GetTagsManager();
    const auto& accountTagsManager = tagsManager.GetAccountTags();

    auto&& holderTagId = GetOfferHolderTagId();
    auto holderDbTag = accountTagsManager.RestoreTag(holderTagId, session);
    if (!holderDbTag) {
        return false;
    }

    auto holderTag = holderDbTag->MutableTagAs<TDedicatedFleetOfferHolderTag>();
    if (holderTag == nullptr) {
        return true;
    }

    const auto offer = holderTag->GetOffer();
    const auto fleetOffer = offer ? std::dynamic_pointer_cast<const TDedicatedFleetOffer>(offer) : nullptr;
    if (!fleetOffer) {
        session.SetErrorInfo("TSpecialFleetTag::OnBeforeRemove", "cannot restore fleet offer");
        return false;
    }

    if (auto report = fleetOffer->GetUnitsOffers().FindPtr(UnitOfferId); report && report->Get()) {
        auto unitFleetOffer = report->Get()->GetOfferPtrAs<TDedicatedFleetUnitOffer>();
        if (!unitFleetOffer) {
            session.SetErrorInfo("TSpecialFleetTag::OnBeforeRemove", "cannot restore unit offer " + UnitOfferId);
            return false;
        }

        if (unitFleetOffer->GetCarId() != self.GetObjectId()) {
            return true;
        }
    } else {
        session.SetErrorInfo("TSpecialFleetTag::OnBeforeRemove", "cannot match unit offer id '" + UnitOfferId + "'");
        return false;
    }

    return false;
}

bool TSpecialFleetTag::OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    if (!TBase::OnBeforeEvolve(fromTag, toTag, permissions, server, session, eContext)) {
        return false;
    }

    const auto& database = server->GetDriveDatabase();
    const auto& tagsManager = database.GetTagsManager();
    const auto& deviceTagsManager = tagsManager.GetDeviceTags();
    const auto& tagsMeta = tagsManager.GetTagsMeta();

    const TString dstTagName = toTag->GetName();

    auto toTagImpl = std::dynamic_pointer_cast<TSpecialFleetTag>(toTag);
    if (!toTagImpl) {
        session.SetErrorInfo("TSpecialFleetTag::OnBeforeEvolve", "incorrect destination tag");
        return false;
    }

    toTagImpl->SetOfferId(OfferId);
    toTagImpl->SetUnitOfferId(UnitOfferId);
    toTagImpl->SetOnAssignAddedTags(OnAssignAddedTags);
    toTagImpl->SetOnAssignEvolvedTags(OnAssignAddedTags);
    toTagImpl->SetOfferHolderTagId(OfferHolderTagId);

    TDedicatedFleetUnitOffer::TPtr unitOffer;
    if (!GetFleetUnitOffer(unitOffer, GetUnitOfferId(), server, session) || !unitOffer) {
        session.AddErrorMessage("TSpecialFleetTag::OnBeforeEvolver", "cannot restore fleet unit offer");
        return false;
    }

    if (dstTagName == TSpecialFleetTag::Preparing) {
        for (const auto& tagName : unitOffer->GetServiceTagsToPerform()) {
            auto tag = tagsMeta.CreateTag(tagName);
            if (!tag) {
                session.SetErrorInfo("TSpecialFleetTag::OnBeforeEvolve", TStringBuilder() << "cannot create tag " << tagName);
                return false;
            }

            if (!deviceTagsManager.AddTag(tag, permissions.GetUserId(), fromTag.GetObjectId(), server, session)) {
                session.AddErrorMessage("TSpecialFleetTag::OnBeforeEvolve", TStringBuilder() << "cannot add tag " << tagName);
                return false;
            }
        }
    } else if (dstTagName == TSpecialFleetTag::Prepared) {
        if (unitOffer->GetServiceTagsToCheck()) {
            TVector<TDBTag> tags;
            if (!deviceTagsManager.RestoreTags({fromTag.GetObjectId()}, ::MakeVector(unitOffer->GetServiceTagsToCheck()), tags, session)) {
                session.AddErrorMessage("TSpecialFleetTag::OnBeforeEvolve", "cannot Restore tags for" + fromTag.GetObjectId());
                return false;
            }

            auto missedTags  = unitOffer->GetServiceTagsToCheck();
            for (const auto& tag : tags) {
                missedTags.erase(tag->GetName());
            }

            if (missedTags) {
                session.AddErrorMessage("TSpecialFleetTag::OnBeforeEvolve", "incomplete car status, missed check tags: " + JoinSeq(", ", missedTags));
                return false;
            }
        }
    } else if (dstTagName == TSpecialFleetTag::Delivery) {
        auto tag = tagsMeta.CreateTagAs<TFleetDeliveryTag>(TFleetDeliveryTag::Type());
        if (!tag) {
            session.AddErrorMessage("TSpecialFleetTag::OnBeforeEvolve", "cannot create fleet delivery tag: " + TFleetDeliveryTag::Type());
            return false;
        }

        if (!unitOffer->HasDeliveryLocation()) {
            session.SetErrorInfo("TSpecialFleetTag::OnBeforeEvolve", "no DeliveryLocation provided");
            return false;
        }

        tag->SetLatitude(unitOffer->GetDeliveryLocationRef().Y);
        tag->SetLongitude(unitOffer->GetDeliveryLocationRef().X);
        tag->SetFleetTagId(fromTag.GetTagId());
    } else if (dstTagName == TSpecialFleetTag::Actual) {
        // TODO: prepare status checks
    }

    return true;
}

void TSpecialFleetTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    IOrganizationTag::SerializeToJson(json);
    TBase::SerializeSpecialDataToJson(json);
    NJson::FieldsToJson(json, GetFields());
}

bool TSpecialFleetTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return IOrganizationTag::DeserializeFromJson(json, errors)
        && TBase::DoSpecialDataFromJson(json, errors)
        && NJson::TryFieldsFromJson(json, GetFields());
}

NDrive::NProto::TSpecialFleetTagData TSpecialFleetTag::DoSerializeSpecialDataToProto() const {
    auto proto = TBase::DoSerializeSpecialDataToProto();
    IOrganizationTag::SerializeToProto(proto);

    proto.SetOfferId(OfferId);
    proto.SetUnitOfferId(UnitOfferId);
    proto.SetOfferHolderTagId(OfferHolderTagId);

    for (auto&& tagId : OnAssignAddedTags) {
        proto.AddOnAssignAddedTags(tagId);
    }

    for (auto&& tagId : OnAssignEvolvedTags) {
        proto.AddOnAssignEvolvedTags(tagId);
    }

    return proto;
}

bool TSpecialFleetTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TSpecialFleetTagData& proto) {
    if (proto.HasOfferId()) {
        SetOfferId(proto.GetOfferId());
    }

    if (proto.HasUnitOfferId()) {
        SetUnitOfferId(proto.GetUnitOfferId());
    }

    if (proto.HasOfferHolderTagId()) {
        SetOfferHolderTagId(proto.GetOfferHolderTagId());
    }

    for (auto&& tagId : proto.GetOnAssignAddedTags()) {
        OnAssignAddedTags.push_back(tagId);
    }

    for (auto&& tagId : proto.GetOnAssignEvolvedTags()) {
        OnAssignEvolvedTags.push_back(tagId);
    }

    return IOrganizationTag::DeserializeFromProto(proto)
        && TBase::DoDeserializeSpecialDataFromProto(proto);
}

bool TSpecialFleetTag::GetFleetUnitOffer(TDedicatedFleetUnitOffer::TPtr& unitOffer,const TString& id, const NDrive::IServer* server, NDrive::TEntitySession& session,  const TDedicatedFleetOffer::TPtr offer) const {
    auto fleetOffer = offer;
    if (!fleetOffer) {
        if (!GetFleetOffer(fleetOffer, server, session)) {
            session.AddErrorMessage("TSpecialFleetTag::GetFleetUnitOffer", "cannot restore fleet offer");
            return false;
        }
    }

    if (auto report = fleetOffer->GetUnitsOffers().FindPtr(id); report && report->Get()) {
        auto unitFleetOffer = report->Get()->GetOfferPtrAs<TDedicatedFleetUnitOffer>();
        if (!unitFleetOffer) {
            session.SetErrorInfo("TSpecialFleetTag::GetFleetUnitOffer", "cannot restore unit offer " + id);
            return false;
        }

        unitOffer = unitFleetOffer;
        return true;
    }

    session.SetErrorInfo("TSpecialFleetTag::GetFleetUnitOffer", "cannot match unit offer id '" + id + "'");
    return false;
}

bool TSpecialFleetTag::GetFleetOffer(TDedicatedFleetOffer::TPtr& offer, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& database = server->GetDriveDatabase();
    const auto& tagsManager = database.GetTagsManager();
    const auto& accountTagsManager = tagsManager.GetAccountTags();

    auto&& holderTagId = GetOfferHolderTagId();
    auto holderDbTag = accountTagsManager.RestoreTag(holderTagId, session);
    if (!holderDbTag) {
        return false;
    }

    auto holderTag = holderDbTag->MutableTagAs<TDedicatedFleetOfferHolderTag>();
    if (!holderTag) {
        return false;
    }

    const auto fleetOffer = offer ? std::dynamic_pointer_cast<TDedicatedFleetOffer>(holderTag->GetOffer()) : nullptr;
    if (!fleetOffer) {
        session.SetErrorInfo("TSpecialFleetTag::GetFleetOffer", "cannot restore fleet offer");
        return false;
    }

    offer = fleetOffer;
    return true;
}

NDrive::TScheme TSpecialFleetTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    IOrganizationTag::AddScheme(result);
    return result;
}


NDrive::TScheme TDedicatedFleetOfferHolderTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    if (server) {
        auto&& tagsMeta = server->GetDriveDatabase().GetTagsManager().GetTagsMeta();
        {
            auto deviceTags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car);
            auto evolveTargets = NContainer::Keys(deviceTags);
            result.Add<TFSVariants>("tags_to_evolve_with_suffix", "Tags to evolve with suffix").SetVariants(evolveTargets).SetMultiSelect(true);
        }

        {
            auto tags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car, {TSpecialFleetTag::TypeName});
            auto fleetTags = NContainer::Keys(tags);
            result.Add<TFSVariants>("fleet_tag", "Dedicated fleet tag").SetVariants(fleetTags).SetRequired(true);
        }

        {
            auto tags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::User, {TFleetSupportOutgoingCommunicationTag::TypeName});
            auto supportTags = NContainer::Keys(tags);
            result.Add<TFSVariants>("communication_tag", "Communication tag").SetVariants(supportTags).SetRequired(true);
        }

        result.Add<TFSBoolean>("start_billing_task", "Start billing").SetDefault(StartBillingTask);
        result.Add<TFSString>("evolve_target_suffix", "Evolve targets suffix").SetDefault(EvolveTargetSuffix);
    }

    return result;
}

NJson::TJsonValue TDedicatedFleetOfferHolderTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TDedicatedFleetOfferHolderTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return TBase::DoDeserializeMetaFromJson(value)
        && NJson::TryFieldsFromJson(value, GetFields());
}

bool TDedicatedFleetOfferHolderTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return TBase::DoSpecialDataFromJson(value, errors) && NJson::TryFieldsFromJson(value, GetFields());
}

void TDedicatedFleetOfferHolderTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}

NDrive::NProto::TOfferHolderTag TDedicatedFleetOfferHolderTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TOfferHolderTag result = TBase::DoSerializeSpecialDataToProto();
    if (CommunicationTagId) {
        result.SetCommunicationTagId(CommunicationTagId);
    }

    auto& fleetHolderInfo = *result.MutableDedicatedFleetOfferHolderTagInfo();
    for (auto&& tagId : FleetTagIds) {
         fleetHolderInfo.AddFleetTagsIds(tagId);
    }

    if (ConstructOffer) {
        fleetHolderInfo.SetConstructOffer(ConstructOffer);
        fleetHolderInfo.SetFleetSize(FleetSize);
        fleetHolderInfo.SetBuilder(Builder);
    }

    return result;
}

bool TDedicatedFleetOfferHolderTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TOfferHolderTag& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }

    if (proto.HasCommunicationTagId()) {
        CommunicationTagId = proto.GetCommunicationTagId();
    }

    if (proto.HasDedicatedFleetOfferHolderTagInfo()) {
        auto fleetHolderInfo = proto.GetDedicatedFleetOfferHolderTagInfo();
        for (auto&& tagId : fleetHolderInfo.GetFleetTagsIds()) {
            FleetTagIds.emplace(tagId);
        }

        if (fleetHolderInfo.HasConstructOffer()) {
            ConstructOffer = fleetHolderInfo.GetConstructOffer();

            if (fleetHolderInfo.HasFleetSize()) {
                FleetSize = fleetHolderInfo.GetFleetSize();
            }

            if (fleetHolderInfo.HasBuilder()) {
                Builder = fleetHolderInfo.GetBuilder();
            }
        }
    }

    return true;
}

NDrive::TScheme TDedicatedFleetOfferHolderTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSBoolean>("construct_offer", "Построить оффер").SetDefault(false);
    result.Add<TFSNumeric>("fleet_size", "Количество машин в парке").SetDefault(0);

    auto builderActions = server
        ? ::MakeSet(server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionNamesWithType<TDedicatedFleetOfferBuilder>(/*reportDeprecated=*/false))
        : TSet<TString>();
    result.Add<TFSVariants>("builder", "Билдер оффера").SetVariants(builderActions);

    return result;
}

bool TDedicatedFleetOfferHolderTag::BuildOfferFromTagData(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
    auto builders = permissions->GetOfferBuilders();
    TSharedPtr<const TDedicatedFleetOfferBuilder, TAtomicCounter> action(nullptr);
    for (auto&& builder : builders) {
        action = std::dynamic_pointer_cast<const TDedicatedFleetOfferBuilder>(builder);
        if (action && action->GetIsPublish() && Builder == action->GetName()) {
             break;
        }

        action.Reset();
    }

    if (action == nullptr) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::BuildOfferFromTagData", "could not create offer invalid check builder " + Builder);
        return false;
    }

    TUserOfferContext uoc(server, permissions, nullptr);
    TOffersBuildingContext buildingContext(server);
    buildingContext.SetUserHistoryContext(uoc);
    buildingContext.MutableVariables().emplace("since", NJson::TJsonValue(TInstant::Now().Seconds()));
    buildingContext.MutableVariables().emplace("until", NJson::TJsonValue(TInstant::Now().Seconds()));
    buildingContext.MutableVariables().emplace("count", FleetSize);

    if (ui64 accountId = 0; TryFromString(objectId, accountId)) {
        buildingContext.MutableVariables().emplace("account_id", accountId);
    } else {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::BuildOfferFromTagData", "bad account id '" + objectId + "'");
        return false;
    }

    buildingContext.MutableVariables().emplace("account_id", objectId);

    TVector<IOfferReport::TPtr> buildedOffers;
    auto resultCode = action->BuildOffers(*permissions, permissions->GetOfferCorrections(), buildedOffers, buildingContext, server, session);
    if (resultCode != EOfferCorrectorResult::Success || buildedOffers.empty()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::BuildOfferFromTagData", "could not create offer builder code: '" + ::ToString(resultCode) + "', builded offers count " + ::ToString(buildedOffers.size()));
        return false;
    }

    auto report = buildedOffers.front();
    auto offer = report->GetOfferPtrAs<TDedicatedFleetOffer>();
    if (!offer) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::BuildOfferFromTagData", "could not cast offer to fleet offer");
        return false;
    }

    offer->SetUserId(userId);
    SetOffer(offer);

    return true;
}

TDBTag TDedicatedFleetOfferHolderTag::Book(ICommonOffer::TPtr offer, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, const TBookOptions& bookOptions) {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "Book")
            ("offer", NJson::ToJson(offer))
            ("user_id", permissions.GetUserId())
            ("options", NJson::ToJson(NJson::Pass(bookOptions)))
        );
    }

    auto fleetOffer = std::dynamic_pointer_cast<TDedicatedFleetOffer>(offer);
    if (!fleetOffer) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::Book", "null offer", EDriveSessionResult::InconsistencyOffer);
        return {};
    }

    if (fleetOffer->GetUserId() != permissions.GetUserId()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::Book", "offer belongs to another user", EDriveSessionResult::InconsistencyOffer);
        return {};
    }

    session.Committed().Subscribe([fleetOffer](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            SendGlobalMessage<TOfferBookingCompleted>(fleetOffer);
        }
    });

    const auto& database = server.GetDriveDatabase();
    const auto& tagsManager = database.GetTagsManager();
    const auto& accountTagsManager = tagsManager.GetAccountTags();
    const auto& tagsMeta = tagsManager.GetTagsMeta();

    const TString& userId = permissions.GetUserId();

    if (fleetOffer->GetTargetHolderTag().empty()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::Book", TStringBuilder() << "no holder tag in offer " << fleetOffer->GetOfferId());
        return {};
    }

    auto tag = tagsMeta.CreateTag(fleetOffer->GetTargetHolderTag());
    if (!tag) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::Book", TStringBuilder() << "cannot create tag " << fleetOffer->GetTargetHolderTag());
        return {};
    }
    auto offerHolderTag = std::dynamic_pointer_cast<TDedicatedFleetOfferHolderTag>(tag);
    if (!offerHolderTag) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::Book", TStringBuilder() << "cannot cast tag " << fleetOffer->GetTargetHolderTag());
        return {};
    }
    offerHolderTag->SetOffer(fleetOffer);

    auto optionalAddedTags = accountTagsManager.AddTag(tag, userId, ToString(fleetOffer->GetAccountId()), &server, session);
    if (!optionalAddedTags) {
        return {};
    }
    if (optionalAddedTags->size() != 1) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::Book", TStringBuilder() << "added " << optionalAddedTags->size() << " tags");
        return {};
    }

    auto result = std::move(optionalAddedTags->front());
    return result;
}

bool TDedicatedFleetOfferHolderTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!GetOffer() && ConstructOffer) {
        if (!BuildOfferFromTagData(objectId, userId, server, session)) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::OnBeforeAdd", "failed to build offer from tag data");
            return false;
        }
    }

    if (!TBase::OnBeforeAdd(objectId, userId, server, session)) {
        return false;
    }

    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& tagMeta = database.GetTagsManager().GetTagsMeta();
    const auto& userTagManager = database.GetTagsManager().GetUserTags();

    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        return false;
    }

    if (const auto& communicationTagName = description->GetCommunicationTag()) {
        auto tag = tagMeta.CreateTag(communicationTagName);
        auto communicationTag = std::dynamic_pointer_cast<TFleetSupportOutgoingCommunicationTag>(tag);
        if (!communicationTag) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeAdd", "cannot create tag " + communicationTagName);
            return false;
        }
        auto added = userTagManager.AddTag(tag, userId, userId, server, session);
        if (!added) {
            return false;
        }
        if (added->size() != 1) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeAdd", "could not AddTag " + communicationTagName);
            return false;
        }
        CommunicationTagId = added->front().GetTagId();
    }

    return true;
}

bool TDedicatedFleetOfferHolderTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterAdd(self, userId, server, session)) {
        return false;
    }

    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& userTagManager = database.GetTagsManager().GetUserTags();

    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        return false;
    }

    if (CommunicationTagId) {
        auto communicationDbTag = userTagManager.RestoreTag(CommunicationTagId, session);
        if (!communicationDbTag) {
            return false;
        }
        auto communicationTag = communicationDbTag->MutableTagAs<TFleetSupportOutgoingCommunicationTag>();
        if (!communicationTag) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnAfterAdd", "cannot cast tag " + communicationDbTag->GetTagId() + " to SupportOutgoingCommunicationTag");
            return false;
        }
        communicationTag->SetLinkedTagId(self.GetTagId());
        if (!userTagManager.UpdateTagData(*communicationDbTag, userId, session)) {
            return false;
        }

        const auto offer = GetOffer();
        const auto fleetOffer = offer ? std::dynamic_pointer_cast<const TDedicatedFleetOffer>(offer) : nullptr;
        if (!fleetOffer) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnAfterAdd", "fleet offer is undefined");
            return false;
        }

        if (fleetOffer && fleetOffer->GetNeedDeferCommunication()) {
            if (!fleetOffer->GetTimeIntervalValues().HasBeginTime()) {
                session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnAfterAdd", "begin time is undefined");
                return false;
            }

            auto deferUntil = fleetOffer->GetTimeIntervalValues().GetBeginTimeRef() - fleetOffer->GetDeferThreshold();
            if (deferUntil > Now()) {
                auto permissions = server->GetDriveAPI()->GetUserPermissions(self.GetObjectId(), TUserPermissionsFeatures());
                if (!communicationTag->OnDeferUntil(*communicationDbTag, {}, deferUntil, *permissions, session, server)) {
                    return false;
                }
            }
        }
    }

    //TODO: create billing task
    // if (description->GetStartBillingTask()) {
    //     if (!StartBillingTask(server, session)) {
    //         return false;
    //     }
    // }

    return true;
}

bool TDedicatedFleetOfferHolderTag::OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnBeforeUpdate(self, to, userId, server, session)) {
        return false;
    }

    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& deviceTagsManager = database.GetTagsManager().GetDeviceTags();
    TUserPermissions::TPtr permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());

    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        return false;
    }

    auto&& specialFleetTagName = description->GetFleetTag();
    if (!specialFleetTagName) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot get special fleet tag name");
        return false;
    }

    auto recipient = std::dynamic_pointer_cast<TDedicatedFleetOfferHolderTag>(to);
    if (!recipient) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot get TDedicatedFleetOfferHolderTag update target");
        return false;
    }

    const auto fleetOffer = recipient->GetOffer() ? std::dynamic_pointer_cast<const TDedicatedFleetOffer>(recipient->GetOffer()) : nullptr;
    if (!fleetOffer) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot get fleet offer");
        return false;
    }

    const auto selfTagImpl = self.GetTagAs<TDedicatedFleetOfferHolderTag>();
    if (!selfTagImpl) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot restore self implementation tag");
        return false;
    }

    const auto selfFleetOffer = selfTagImpl->GetOffer() ? std::dynamic_pointer_cast<const TDedicatedFleetOffer>(selfTagImpl->GetOffer()) : nullptr;
    if (!selfFleetOffer) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot get self fleet offer");
        return false;
    }

    const auto offerIds = MakeSet(NContainer::Keys(fleetOffer->GetUnitsOffers()));
    const auto selfOfferIds = MakeSet(NContainer::Keys(selfFleetOffer->GetUnitsOffers()));
    if (offerIds != selfOfferIds) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "unit offer ids missmatch");
        return false;
    }

    TSet<TString> carsIds;
    for (auto&& offerId : selfOfferIds) {
        TDedicatedFleetUnitOffer::TPtr unitFleetOffer;
        if (auto report = fleetOffer->GetUnitsOffers().FindPtr(offerId); report && report->Get()) {
            unitFleetOffer = report->Get()->GetOfferPtrAs<TDedicatedFleetUnitOffer>();
        }
        if (!unitFleetOffer) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot restore unit offer " + offerId);
            return false;
        }

        TDedicatedFleetUnitOffer::TPtr selfUnitFleetOffer;
        if (auto report = selfFleetOffer->GetUnitsOffers().FindPtr(offerId); report && report->Get()) {
            selfUnitFleetOffer = report->Get()->GetOfferPtrAs<TDedicatedFleetUnitOffer>();
        }
        if (!selfUnitFleetOffer) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot restore unit offer " + offerId);
            return false;
        }

        if (!UpdateUnitOffer(self, selfUnitFleetOffer, unitFleetOffer, *permissions, server, session, unitFleetOffer->GetForceUpdate())) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot update unit offer data");
            return false;
        }

        if (unitFleetOffer->GetCarId()) {
            if (!carsIds.emplace(unitFleetOffer->GetCarId()).second) {
                session.AddErrorMessage("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "car id already linked to another unit offer");
                return false;
            }
        }
    }

    recipient->MutableFleetTagIds().clear();
    if (!carsIds.empty()) {
        TVector<TDBTag> tags;
        if (!deviceTagsManager.RestoreTags(carsIds, {specialFleetTagName}, tags, session)) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::OnBeforeUpdate", "cannot Restore tags " + JoinSeq(", ", carsIds));
            return false;
        }

        for (auto&& dbTag : tags) {
            recipient->MutableFleetTagIds().emplace(dbTag.GetTagId());
        }
    }

    return true;
}

bool TDedicatedFleetOfferHolderTag::OnAfterUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterUpdate(self, to, userId, server, session)) {
        return false;
    }

    return DoDeferredClean(self, userId, server, session);
}

bool TDedicatedFleetOfferHolderTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeRemove(self, userId, server, session)) {
        return false;
    }

    TUserPermissions::TPtr permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());

    const auto& database = server->GetDriveDatabase();
    const auto& deviceTagsManager = database.GetTagsManager().GetDeviceTags();
    const auto& userTagManager = database.GetTagsManager().GetUserTags();

    auto& fleetTagIds = GetFleetTagIds();
    if (fleetTagIds) {
        TVector<TDBTag> tags;
        if (!deviceTagsManager.RestoreTags(fleetTagIds, tags, session)) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::OnBeforeRemove", "cannot Restore tags " + JoinSeq(", ", fleetTagIds));
            return false;
        }

        for (auto&& dbTag : tags) {
            if (!RemoveCar(self, dbTag.GetObjectId(), *permissions, server, session, false)) {
                session.AddErrorMessage("TDedicatedFleetOfferHolderTag::OnBeforeRemove", "cannot remove all cars from fleet");
                return false;
            }
        }
    }

    auto& communicationTagId = GetCommunicationTagId();
    if (!communicationTagId.Empty()) {
        auto optCommunicationTag = userTagManager.RestoreTag(communicationTagId, session);
        if (optCommunicationTag) {
            if (!userTagManager.RemoveTag(*optCommunicationTag, userId, server, session)) {
                session.AddErrorMessage("TDedicatedFleetOfferHolderTag::OnBeforeRemove", "error while removing communication tag");
                return false;
            }
        }
    }

    return true;
}

bool TDedicatedFleetOfferHolderTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterRemove(self, userId, server, session)) {
        return false;
    }

    return DoDeferredClean(self, userId, server, session);
}

bool TDedicatedFleetOfferHolderTag::DoDeferredClean(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& database = server->GetDriveDatabase();
    const auto& deviceTagsManager = database.GetTagsManager().GetDeviceTags();

    auto selfImpl = std::dynamic_pointer_cast<const TDedicatedFleetOfferHolderTag>(self.GetData());
    if (!selfImpl) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::DoDeferredClean", "cannot cast offer holder tag");
        return false;
    }

    auto&& removeSet = selfImpl->GetTagsForDeferredCleanup();

    if (!removeSet.empty()) {
        auto tagsToRemove = deviceTagsManager.RestoreTags(::MakeSet(removeSet), session);
        if (!tagsToRemove) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::DoDeferredClean", "cannot Restore fleet tags: " + JoinSeq(", ", removeSet));
            return false;
        }

        if (!deviceTagsManager.RemoveTags(*tagsToRemove, userId, server, session)) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::DoDeferredClean", "cannot remove fleet tags: " + JoinSeq(", ", removeSet));
            return false;
        }
    }

    return true;
}

bool TDedicatedFleetOfferHolderTag::UpdateUnitOffer(
    const TDBTag& selfTag, const TDedicatedFleetUnitOffer::TPtr& selfUOffer, const TDedicatedFleetUnitOffer::TPtr& toUOffer, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, bool force) const {
    if (selfUOffer->GetCarId() != toUOffer->GetCarId()) {
        if(!selfUOffer->GetCarId().Empty()) {
            if (!RemoveCar(selfTag, selfUOffer->GetCarId(), permissions, server, session, force)) {
                return false;
            }
        }

        if (!toUOffer->GetCarId().Empty()) {
            if (!AddCar(selfTag, toUOffer->GetCarId(), toUOffer->GetOfferId(), permissions, server, session)) {
                return false;
            }
        }
    }

    return true;
}

bool TDedicatedFleetOfferHolderTag::AddCar(const TDBTag& selfTag, const TString& carId, const TString& offerId, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& deviceTagsManager = database.GetTagsManager().GetDeviceTags();
    const auto& tagsMeta = database.GetTagsManager().GetTagsMeta();
    const auto& deviceTags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car);

    if (carId.Empty()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::AddCar", "cannot add empty car id");
        return false;
    }

    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        return false;
    }

    auto&& specialFleetTagName = description->GetFleetTag();
    if (!specialFleetTagName) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::AddCar", "cannot get special fleet tag name");
        return false;
    }

    TVector<TDBTag> tags;
    if (!deviceTagsManager.RestoreTags({carId}, {specialFleetTagName}, tags, session)) {
        session.AddErrorMessage("TDedicatedFleetOfferHolderTag::AddCar", "cannot Restore tags " + carId);
        return false;
    }

    if (!tags.empty()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::AddCar", "car " + carId + " already in fleet");
        return false;
    }

    auto fleetTag = PrepareFleetTag(selfTag, offerId, server, session);
    if (!fleetTag) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::AddCar", "cannot create fleet tag");
        return false;
    }

    {
        TVector<TString> addedTagIds;
        addedTagIds.reserve(description->GetCarTags().size());
        for (auto&& tagName : description->GetCarTags()) {
            auto tag = tagsMeta.CreateTag(tagName);
            if (!tag) {
                session.SetErrorInfo("TDedicatedFleetOfferHolderTag::AddCar", "cannot create tag " + tagName);
                return false;
            }

            if (auto added = deviceTagsManager.AddTag(tag, permissions.GetUserId(), carId, server, session)) {
                for (auto&& i : *added) {
                    addedTagIds.push_back(i.GetTagId());
                }
            } else {
                return false;
            }
        }

        fleetTag->SetOnAssignAddedTags(std::move(addedTagIds));
    }

    {
        TVector<TString> evolvedTags;
        auto&& tagsToEvolve = description->GetTagsToEvolveWithSuffix();
        if (tagsToEvolve) {
            TVector<TDBTag> evolveTargets;
            if (!deviceTagsManager.RestoreTags({carId}, ::MakeVector(tagsToEvolve), evolveTargets, session)) {
                session.AddErrorMessage("TDedicatedFleetOfferHolderTag::AddCar", "cannot Restore tags " + carId);
                return false;
            }

            evolvedTags.reserve(evolveTargets.size());
            for (auto&& evolveTarget : evolveTargets) {
                auto newTagName = TStringBuilder() << evolveTarget->GetName() << description->GetEvolveTargetSuffix();
                if (!deviceTags.contains(newTagName)) {
                    session.SetErrorInfo("TDedicatedFleetOfferHolderTag::AddCar", "evolve target " + newTagName + " is not registred");
                    return false;
                }

                if (auto destinationTag = tagsMeta.CreateTag(newTagName)) {
                    if (!deviceTagsManager.EvolveTag(evolveTarget, destinationTag, permissions, server,session)) {
                        session.AddErrorMessage("TDedicatedFleetOfferHolderTag::AddCar", "cannot evolve tag " + evolveTarget.GetTagId() + " to suffixed *" + description->GetEvolveTargetSuffix());
                        return false;
                    }
                } else {
                    session.SetErrorInfo("TDedicatedFleetOfferHolderTag::AddCar", "cannot create tag " + newTagName);
                    return false;
                }

                evolvedTags.emplace_back(evolveTarget.GetTagId());
            }
        }

        fleetTag->SetOnAssignEvolvedTags(evolvedTags);
    }

    if (auto added = deviceTagsManager.AddTag(fleetTag, permissions.GetUserId(), carId, server, session); !added || added->empty()) {
        session.AddErrorMessage("TDedicatedFleetOfferHolderTag::AddCar", "cannot add tag " + fleetTag->GetName());
        return false;
    }

    return true;
}

bool TDedicatedFleetOfferHolderTag::RemoveCar(const TDBTag& selfTag, const TString& carId, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, bool force) const {
    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& deviceTagsManager = database.GetTagsManager().GetDeviceTags();
    const auto& tagsMeta = database.GetTagsManager().GetTagsMeta();
    const auto& deviceTags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car);

    if (carId.empty()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot remove empty car id");
        return false;
    }

    const auto offer = GetOffer();
    const auto fleetOffer = offer ? std::dynamic_pointer_cast<const TDedicatedFleetOffer>(offer) : nullptr;
    if (!fleetOffer) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot restore fleet offer");
        return false;
    }

    auto description = GetDescriptionAs<TDedicatedFleetOfferHolderTag::TDescription>(*server,  session);
    if (!description) {
        return false;
    }

    auto&& specialFleetTagName = description->GetFleetTag();
    TVector<TDBTag> tags;
    if (!deviceTagsManager.RestoreTags({carId}, {specialFleetTagName}, tags, session)) {
        session.AddErrorMessage("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot Restore tags " + carId);
        return false;
    }

    if (tags.empty()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "car " + carId + " not in fleet");
        return false;
    } else if (tags.size() != 1) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "multiply fleet tags found for car " + carId + " process car in manual mode");
        return false;
    }

    auto fleetTag = tags.front().MutableTagAs<TSpecialFleetTag>();
    if (!fleetTag) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "unable to recognize fleet tag for car " + carId);
        return false;
    }

    if (fleetTag->GetOfferId() != fleetOffer->GetOfferId()) {
        session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot match special fleet tag offer " + fleetTag->GetOfferId() + " != " + fleetOffer->GetOfferId());
        return false;
    }

    const auto& carTagsIds = fleetTag->GetOnAssignAddedTags();
    if (!carTagsIds.empty()) {
        auto subTags = deviceTagsManager.RestoreTags(::MakeSet(carTagsIds), session);
        if (!subTags) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot Restore tags " + carId);
            return false;
        }

        if (!force) {
            auto idsSet = ::MakeSet(carTagsIds);
            for (auto subTag : *subTags) {
                idsSet.erase(subTag.GetTagId());
            }

            if (!idsSet.empty()) {
                session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar",
                    "cannot remove car " + carId + " auto assigned tags missmatch, this tags was performed or removed < " + JoinSeq(", ", idsSet) + "> try force to skip check.");
                return false;
            }
        }

        auto selfImpl = std::dynamic_pointer_cast<const TDedicatedFleetOfferHolderTag>(selfTag.GetData());
        if (!selfImpl) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot cast offer holder tag");
            return false;
        }

        selfImpl->MutableTagsForDeferredCleanup().emplace(tags.front().GetTagId());

        TVector<TDBTag> tagsToRemove;
        tagsToRemove.reserve(subTags->size());
        for (auto subTag : *subTags) {
            if (subTag->GetPerformer() && !force) {
                session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot remove car " + carId + " cos tag " + subTag->GetName() + " has performer");
                return false;
            }

            tagsToRemove.emplace_back(subTag);
        }

        if (!deviceTagsManager.RemoveTags(tagsToRemove, permissions.GetUserId(), server, session)) {
            session.AddErrorMessage("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot remove sub tags for car " + carId);
            return false;
        }
    }

    auto evolvedTags = deviceTagsManager.RestoreTags(::MakeSet(fleetTag->GetOnAssignEvolvedTags()), session);
    if (!evolvedTags) {
        session.AddErrorMessage("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot Restore tags " + carId);
        return false;
    }

    for (auto&& tag : *evolvedTags) {
        if (!tag->GetName().EndsWith(description->GetEvolveTargetSuffix())) {
            continue;
        }

        auto tagName = tag->GetName();
        tagName = tagName.substr(0, tagName.find(description->GetEvolveTargetSuffix()));
        if (!deviceTags.contains(tagName)) {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "evolve target " + tagName + " is not registred");
            return false;
        }

        if (auto destinationTag = tagsMeta.CreateTag(tagName)) {
            if (!deviceTagsManager.EvolveTag(tag, destinationTag, permissions, server,session)) {
                session.AddErrorMessage("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot evolve tag " + tag.GetTagId() + " to suffixed *" + description->GetEvolveTargetSuffix());
                return false;
            }
        } else {
            session.SetErrorInfo("TDedicatedFleetOfferHolderTag::RemoveCar", "cannot create tag " + tagName);
            return false;
        }
    }

    return true;
}

TSpecialFleetTag::TPtr TDedicatedFleetOfferHolderTag::PrepareFleetTag(const TDBTag& selfTag, const TString& offerId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& tagsMeta = database.GetTagsManager().GetTagsMeta();
    auto description = GetDescriptionAs<TDedicatedFleetOfferHolderTag::TDescription>(*server,  session);
    if (!description) {
        return TSpecialFleetTag::TPtr();
    }

    auto&& specialFleetTagName = description->GetFleetTag();
    auto tag = tagsMeta.CreateTagAs<TSpecialFleetTag>(specialFleetTagName);
    if (!tag) {
        session.AddErrorMessage("TDedicatedFleetOfferHolderTag::PrepareFleetTag", "cannot create tag " + specialFleetTagName);
        return TSpecialFleetTag::TPtr();
    }

    const auto offer = GetOffer();
    const auto fleetOffer = offer ? std::dynamic_pointer_cast<const TDedicatedFleetOffer>(offer) : nullptr;
    if (!fleetOffer) {
        session.AddErrorMessage("TDedicatedFleetOfferHolderTag::PrepareFleetTag", "cannot restore fleet offer");
        return TSpecialFleetTag::TPtr();
    }

    tag->SetParentId(fleetOffer->GetAccountId());
    tag->SetUnitOfferId(offerId);
    tag->SetOfferId(fleetOffer->GetOfferId());
    tag->SetOfferHolderTagId(selfTag.GetTagId());

    return tag;
}

ITag::TFactory::TRegistrator<TDedicatedFleetOfferHolderTag> TDedicatedFleetOfferHolderTag::Registrator(TDedicatedFleetOfferHolderTag::Type());
TTagDescription::TFactory::TRegistrator<TDedicatedFleetOfferHolderTag::TDescription> TDedicatedFleetOfferHolderTag::TDescription::Registrator(TDedicatedFleetOfferHolderTag::Type());

template <>
NJson::TJsonValue NJson::ToJson(const TDedicatedFleetOfferHolderTag::TBookOptions& value) {
    NJson::TJsonValue result;
    result["check_blocked"] = value.CheckBlocked;
    return result;
}

const TString TFleetSupportOutgoingCommunicationTag::TypeName = "fleet_user_outgoing_communication";
ITag::TFactory::TRegistrator<TFleetSupportOutgoingCommunicationTag> TFleetSupportOutgoingCommunicationTag::Registrator(TFleetSupportOutgoingCommunicationTag::TypeName);

bool TFleetSupportOutgoingCommunicationTag::OnAfterEvolve(const TDBTag& /*from*/, ITag::TPtr /*to*/, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* /*eContext*/) const {
    auto fleetHolderTagId = GetLinkedTagId();
    if (!fleetHolderTagId) {
        return false;
    }

    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& tagManager = database.GetTagsManager().GetAccountTags();

    auto expectedFleetHolder = tagManager.RestoreTag(fleetHolderTagId, session);
    if (!expectedFleetHolder) {
        session.SetErrorInfo("TFleetSupportOutgoingCommunicationTag::OnAfterEvolve", "cannot RestoreTag " + fleetHolderTagId);
        return false;
    }

    auto destinationTag = expectedFleetHolder->Clone(server->GetDriveAPI()->GetTagsHistoryContext());
    TDedicatedFleetOfferHolderTag::TPtr fleetHolderTagImpl = std::dynamic_pointer_cast<TDedicatedFleetOfferHolderTag>(destinationTag.GetData());
    if (!fleetHolderTagImpl) {
        session.SetErrorInfo("TFleetSupportOutgoingCommunicationTag::OnAfterEvolve", "error while clone Fleet holder tag");
        return false;
    }

    const auto offer = fleetHolderTagImpl->GetOffer();
    const auto fleetOffer = offer ? std::dynamic_pointer_cast<const TDedicatedFleetOffer>(offer) : nullptr;
    if (!fleetOffer) {
        session.SetErrorInfo("TFleetSupportOutgoingCommunicationTag::OnAfterEvolve", "cannot restore fleet offer");
        return false;
    }

    auto&& uOffers = fleetOffer->GetUnitsOffers();
    TSet<TString> uniqueCarsCache;
    for (auto&& [offerId, data] : RawData.GetMap()) {
        TDedicatedFleetUnitOffer::TPtr unitFleetOffer;
        if (auto report = uOffers.FindPtr(offerId); report && report->Get()) {
            unitFleetOffer = report->Get()->GetOfferPtrAs<TDedicatedFleetUnitOffer>();
        }

        if (!unitFleetOffer) {
            session.SetErrorInfo("TFleetSupportOutgoingCommunicationTag::OnAfterEvolve", "cannot restore unit offer " + offerId);
            return false;
        }

        auto carId = data["car_id"].GetString();
        if (uniqueCarsCache.contains(carId)) {
            session.SetErrorInfo("TFleetSupportOutgoingCommunicationTag::OnAfterEvolve", "duplicating cars in raw data " + offerId);
            return false;
        }

        if (carId) {
            uniqueCarsCache.emplace(carId);
        }

        unitFleetOffer->SetCarId(carId);
        unitFleetOffer->SetForceUpdate(data["force_update"].GetBoolean());
    }

    return tagManager.UpdateTagData((*expectedFleetHolder), destinationTag.GetData(), permissions.GetUserId(), server, session);
}

bool TFleetSupportOutgoingCommunicationTag::OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    if (!TBase::OnBeforeEvolve(from, to, permissions, server, session, eContext)) {
        return false;
    }
    return true;
}

TFleetSupportOutgoingCommunicationTag::TProto TFleetSupportOutgoingCommunicationTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    return proto;
}

bool TFleetSupportOutgoingCommunicationTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

void TFleetSupportOutgoingCommunicationTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
}

bool TFleetSupportOutgoingCommunicationTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return TBase::DoSpecialDataFromJson(json, errors);
}

ITag::TFactory::TRegistrator<TFleetDeliveryTag> TFleetDeliveryTag::Registrator(TFleetDeliveryTag::Type());
TTagDescription::TFactory::TRegistrator<TFleetDeliveryTag::TDescription> FleetDeliveryTag(TFleetDeliveryTag::Type());

bool TFleetDeliveryTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeAdd(objectId, userId, server, session)) {
        return false;
    }

    return true;
}

bool TFleetDeliveryTag::OnBeforePerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforePerform(self, permissions, server, session)) {
        return false;
    }

    TSpecialFleetTag::TPtr fleetTagImpl;
    if (!GetFleetTag(fleetTagImpl, GetFleetTagId(), server, session)) {
        session.AddErrorMessage("TFleetDeliveryTag::OnBeforePerform", "cannot restore fleet tag " + FleetTagId);
        return false;
    }

    if (fleetTagImpl->GetName() != TSpecialFleetTag::Delivery) {
        session.SetErrorInfo("TFleetDeliveryTag::OnBeforePerform", "incompatible fleet tag state" + fleetTagImpl->GetName());
        return false;
    }

    return true;
}

bool TFleetDeliveryTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterRemove(self, userId, server, session)) {
        return false;
    }

    if (GetPerformer()) {
        const auto driveApi = Yensured(server)->GetDriveAPI();
        const auto& database = server->GetDriveDatabase();
        const auto& tagsManager = database.GetTagsManager();
        const auto& deviceTagsManager = tagsManager.GetDeviceTags();
        const auto& tagsMeta = tagsManager.GetTagsMeta();

        auto permissions = driveApi->GetUserPermissions(userId, {});
        if (!permissions) {
            session.SetErrorInfo("TFleetDeliveryTag::OnAfterRemove", "cannot recreate permissions");
            return false;
        }

        TSpecialFleetTag::TPtr fleetTagImpl;
        if (!GetFleetTag(fleetTagImpl, GetFleetTagId(), server, session)) {
            session.AddErrorMessage("TFleetDeliveryTag::OnAfterRemove", "cannot restore fleet tag " + FleetTagId);
            return false;
        }

        if (fleetTagImpl->GetName() == TSpecialFleetTag::Delivery) {
            if (auto dstTag = tagsMeta.CreateTag(TSpecialFleetTag::Actual)) {
                auto fleetTag = deviceTagsManager.RestoreTag(GetFleetTagId(), session);
                if (!fleetTag) {
                    session.AddErrorMessage("TFleetDeliveryTag::OnAfterRemove", "cannot restore fleet tag " + GetFleetTagId());
                    return false;
                }

                if (!deviceTagsManager.EvolveTag(*fleetTag, dstTag, *permissions, server,session)) {
                    session.AddErrorMessage("TFleetDeliveryTag::OnAfterRemove", "cannot evolve tag " + GetFleetTagId());
                    return false;
                }
            }
        }
    }

    return true;
}

bool TFleetDeliveryTag::GetFleetTag(TSpecialFleetTag::TPtr& tag, const TString& id, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& tagsManager = database.GetTagsManager();
    const auto& deviceTagsManager = tagsManager.GetDeviceTags();

    if (!FleetTagId) {
        session.SetErrorInfo("TFleetDeliveryTag::GetFleetTag", "cannot perform tag: empty fleet_tag_id");
        return false;
    }

    auto fleetTag = deviceTagsManager.RestoreTag(id, session);
    if (!fleetTag) {
        session.AddErrorMessage("TFleetDeliveryTag::GetFleetTag", "cannot restore fleet tag " + id);
        return false;
    }

    auto fleetTagImpl = std::dynamic_pointer_cast<TSpecialFleetTag>(fleetTag->GetData());
    if (!fleetTagImpl) {
        session.SetErrorInfo("TFleetDeliveryTag::GetFleetTag", "cannot cast fleet tag " + id);
        return false;
    }

    tag = fleetTagImpl;

    return true;
}

bool TFleetDeliveryTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return TBase::DoSpecialDataFromJson(value, errors)
        && NJson::TryFieldsFromJson(value, GetFields());
}

void TFleetDeliveryTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}
