#include "delegation.h"

#include "chargable.h"
#include "user_tags.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/device_snapshot/snapshot.h>
#include <drive/backend/offers/actions/flexipack.h>
#include <drive/backend/offers/actions/pack.h>

const TString TFreeDelegationTag::TypeName = "free_delegation_tag";
ITag::TFactory::TRegistrator<TFreeDelegationTag> TFreeDelegationTag::Registrator(TFreeDelegationTag::TypeName);

const TString TP2PDelegationTag::TypeName = "p2p_delegation_tag";
ITag::TFactory::TRegistrator<TP2PDelegationTag> TP2PDelegationTag::Registrator(TP2PDelegationTag::TypeName);

const TString TDelegationUserTag::TypeName = "user_delegation_tag";
ITag::TFactory::TRegistrator<TDelegationUserTag> TDelegationUserTag::Registrator(TDelegationUserTag::TypeName);

const TString TIncomingDelegationUserTag::TypeName = "user_incoming_delegation_tag";
ITag::TFactory::TRegistrator<TIncomingDelegationUserTag> TIncomingDelegationUserTag::Registrator(TIncomingDelegationUserTag::TypeName);

NDrive::TScheme TP2PDelegationTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("p2p_user_id", "Целевой пользователь передачи");
    result.Add<TFSString>("p2p_user_phone", "Телефон целевого пользователя");
    result.Add<TFSString>("p2p_user_name", "Имя целевого пользователя");
    result.Add<TFSBoolean>("p2p_preserve_offer", "Сохранять оффер?");
    return result;
}

NJson::TJsonValue TP2PDelegationTag::GetReport(const TDBTag& self, const NDrive::IServer* server) const {
    NJson::TJsonValue result = TBase::GetReport(self, server);

    if (!P2PUserId) {
        TJsonProcessor::Write(result, "p2p_qr_code", server->GetSettings().GetValueDef<TString>("delegation.p2p.qr_deeplink_prefix", "") + "QR" + self.GetTagId());
    } else {
        TJsonProcessor::Write(result, "p2p_user_phone", P2PUserPhone);
        TJsonProcessor::Write(result, "p2p_user_name", P2PUserName);
        TJsonProcessor::Write(result, "p2p_target_user_id", P2PUserId);
    }
    TJsonProcessor::Write(result, "p2p_rejected", Rejected);
    TJsonProcessor::Write(result, "p2p_preserve_offer", PreserveOffer);

    auto now = Now();
    TJsonProcessor::Write(result, "p2p_expired", now >= GetSLAInstant());
    TJsonProcessor::Write(result, "p2p_time_remaining", (GetSLAInstant() - now).Seconds());
    TJsonProcessor::Write(result, "p2p_end_timestamp", GetSLAInstant().Seconds());

    return result;
}

void TP2PDelegationTag::SerializeToJson(NJson::TJsonValue& json) const {
    IDelegationTag::SerializeToJson(json);
    TJsonProcessor::Write(json, "p2p_user_id", P2PUserId);
    TJsonProcessor::Write(json, "p2p_user_phone", P2PUserPhone);
    TJsonProcessor::Write(json, "p2p_user_name", P2PUserName);
    TJsonProcessor::Write(json, "p2p_rejected", Rejected);
    TJsonProcessor::Write(json, "p2p_preserve_offer", PreserveOffer);
}

bool TP2PDelegationTag::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/) {
    TJsonProcessor::Read(json, "p2p_user_id", P2PUserId);
    TJsonProcessor::Read(json, "p2p_user_phone", P2PUserPhone);
    TJsonProcessor::Read(json, "p2p_user_name", P2PUserName);
    JREAD_BOOL_OPT(json, "p2p_rejected", Rejected);
    JREAD_BOOL_OPT(json, "p2p_preserve_offer", PreserveOffer);
    return true;
}

NTagActions::TTagActions TP2PDelegationTag::GetVisibilityFeatures(const TUserPermissions& permissions) const {
    if (P2PUserId == permissions.GetUserId()) {
        return (TTagAction::TTagActions)TTagAction::ETagAction::ObserveBusyObject;
    }
    return 0;
}

TP2PDelegationTag::TProto TP2PDelegationTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TP2PDelegationTag proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetP2PUserId(P2PUserId);
    proto.SetP2PUserPhone(P2PUserPhone);
    proto.SetP2PUserName(P2PUserName);
    proto.SetPreserveOffer(PreserveOffer);
    proto.SetRejected(Rejected);
    IDelegationTag::SerializeToProto(proto);
    return proto;
}

bool TP2PDelegationTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!IDelegationTag::DeserializeFromProto(proto)) {
        return false;
    }
    P2PUserId = proto.GetP2PUserId();
    P2PUserPhone = proto.GetP2PUserPhone();
    P2PUserName = proto.GetP2PUserName();
    Rejected = proto.GetRejected();
    PreserveOffer = proto.GetPreserveOffer();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

bool TP2PDelegationTag::OnAfterPerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterPerformImpl(self, permissions, server, session)) {
        return false;
    }

    TVector<TDBTag> tags;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({GetBaseUserId()}, {TDelegationUserTag::TypeName}, tags, session)) {
        session.SetErrorInfo("delegation", "delegation.no_base_user_tags", EDriveSessionResult::InconsistencyUser);
        return false;
    }
    for (auto&& tag : tags) {
        auto tagImpl = tag.MutableTagAs<TDelegationUserTag>();
        if (!tagImpl) {
            continue;
        }
        if (tagImpl->GetObjectId() == self.GetObjectId()) {
            tagImpl->SetTargetUserId(permissions.GetUserId());
            if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(tag, permissions.GetUserId(), session)) {
                session.SetErrorInfo("delegation", "delegation.base_user_tag.update_failed", EDriveSessionResult::InternalError);
                return false;
            }
        }
    }

    if (!!P2PUserId && (GetOffer()->GetUserId() != P2PUserId || P2PUserId != permissions.GetUserId())) {
        session.SetErrorInfo("delegation", "delegation.incorrect_user", EDriveSessionResult::InconsistencyUser);
        return false;
    }
    return true;
}

bool TP2PDelegationTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterAddImpl(self, userId, server, session)) {
        return false;
    }

    auto selfImpl = self.GetTagAs<TP2PDelegationTag>();
    if (!selfImpl->GetP2PUserId()) {
        return true;
    }

    auto userData = server->GetDriveAPI()->GetUsersData();
    auto delegatorFetchResult = userData->FetchInfo(userId, session);
    if (!delegatorFetchResult) {
        return false;
    }
    auto delegator = delegatorFetchResult.GetResultPtr(userId);
    if (!delegator) {
        session.SetErrorInfo("P2PDelegationTag::OnAfterAdd", "cannot FetchInfo for user " + userId);
        return false;
    }

    THolder<TIncomingDelegationUserTag> incomingTag(new TIncomingDelegationUserTag(TIncomingDelegationUserTag::TypeName));
    incomingTag->SetObjectId(self.GetObjectId());
    incomingTag->SetType(GetDelegationType());
    incomingTag->SetDelegatorName(delegator->GetShortName());
    incomingTag->SetSLAInstant(selfImpl->GetSLAInstant());

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(incomingTag.Release(), userId, selfImpl->GetP2PUserId(), server, session)) {
        return false;
    }
    return true;
}

bool TP2PDelegationTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterRemoveImpl(self, userId, server, session)) {
        return false;
    }

    auto selfImpl = self.GetTagAs<TP2PDelegationTag>();

    if (!selfImpl->GetP2PUserId()) {
        return true;
    }

    TTaggedObject obj;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(selfImpl->GetP2PUserId(), obj, session)) {
        session.SetErrorInfo("p2p_delegation_tag", "could not restore tagged user", EDriveSessionResult::InternalError);
        return false;
    }
    TVector<TDBTag> userTags = obj.GetTagsByClass<TIncomingDelegationUserTag>();
    for (auto&& tag : userTags) {
        auto tagImpl = tag.GetTagAs<TIncomingDelegationUserTag>();
        if (!tagImpl) {
            continue;
        }
        if (tagImpl->GetObjectId() == self.GetObjectId()) {
            if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(tag, userId, server, session)) {
                session.SetErrorInfo("p2p_delegation_tag", "could not remove incoming delegation tag", EDriveSessionResult::InternalError);
                return false;
            }
        }
    }

    return true;
}

bool IDelegationTag::ShrinkPackOfferOnDelegation(ICommonOffer::TPtr& offer, const TBillingSession& bSession) const {
    double delegatedDistance = 0;
    TDuration delegatedDuration = TDuration::Zero();
    auto currentOffer = bSession.GetCurrentOffer();

    auto packOffer = dynamic_cast<TPackOffer*>(currentOffer.Get());
    auto billingCompilation = bSession.GetCompilationAs<TBillingSession::TBillingCompilation>();
    if (packOffer) {
        auto packOfferState = billingCompilation->GetCurrentOfferStateAs<TPackOfferState>();
        delegatedDistance = packOfferState->GetRemainingDistance();
        delegatedDuration = packOfferState->GetPackRemainingTime();
    }

    auto newPackOffer = dynamic_cast<TPackOffer*>(offer.Get());
    if (!newPackOffer) {
        return false;
    }

    newPackOffer->SetMileageLimit(delegatedDistance);
    newPackOffer->SetDuration(delegatedDuration);
    newPackOffer->SetExtraMileageLimit(0);
    newPackOffer->SetExtraDuration(TDuration::Zero());
    newPackOffer->SetPackPrice(0);
    newPackOffer->SetReturningDuration(TDuration::Zero());
    newPackOffer->SetDepositAmount(0);
    newPackOffer->SetUseDeposit(false);
    if (packOffer) {
        newPackOffer->SetOverrunKm(packOffer->GetOverrunKm());
        newPackOffer->SetOvertimeParking(packOffer->GetOvertimeParking());
        newPackOffer->SetOvertimeRiding(packOffer->GetOvertimeRiding());
    }

    return DoShrinkPackOfferOnDelegation(offer, bSession);
}

bool TP2PDelegationTag::DoShrinkPackOfferOnDelegation(ICommonOffer::TPtr& offer, const TBillingSession& /*bSession*/) const {
    offer->SetDeadline(GetSLAInstant());
    return true;
}

bool IDelegationTag::FinishOriginalSession(const TDBTag& cTag, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    NDrive::ITag::TEvolutionContext forceContext;
    forceContext.SetMode(EEvolutionMode::Force);
    if ((cTag->GetName() == "old_state_parking" || cTag->GetName() == "old_state_reservation")) {
        auto next = MakeAtomicShared<TChargableTag>(TChargableTag::Reservation);
        next->SetDelegationType(GetDelegationType());
        auto ownerPermissions = server->GetDriveAPI()->GetUserPermissions(cTag->GetPerformer(), TUserPermissionsFeatures());
        if (!ownerPermissions) {
            session.SetErrorInfo("DelegationTag::FinishOriginalSession", "cannot create permissions for " + cTag->GetPerformer(), EDriveSessionResult::InternalError);
            return false;
        }
        return TChargableTag::DirectEvolve(cTag, next, *ownerPermissions, *server, session, &forceContext);
    } else {
        session.SetErrorInfo("delegation", "delegation.incorrect_state", EDriveSessionResult::InconsistencyUser);
        return false;
    }
}

void IDelegationTag::PatchScheme(const NDrive::IServer* /*server*/, NDrive::TScheme& result) const {
    result.Add<TFSString>("BaseUserId", "пользователь, инициировавший передачу");
}

bool IDelegationTag::OnAfterRemoveImpl(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!GetBaseUserId()) {
        return true;
    }
    TTaggedObject obj;
    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(GetBaseUserId(), obj, session)) {
        return false;
    }
    TVector<TDBTag> userTags = obj.GetTagsByClass<TDelegationUserTag>();
    bool needPush = false;
    for (auto&& i : userTags) {
        if (i.GetTagAs<TDelegationUserTag>()->GetObjectId() == self.GetObjectId()) {
            if (userId != GetBaseUserId()) {
                i.MutableTagAs<TDelegationUserTag>()->SetFinishInstant(ModelingNow());
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(i, userId, session)) {
                    return false;
                }
                needPush = true;
            } else {
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(i, userId, server, session)) {
                    return false;
                }
            }
        }
    }
    if (needPush) {
        auto pushTagName = server->GetSettings().GetValueDef<TString>("evolution.delegation." + ToString(GetDelegationType()) + ".finish_push_tag", "");
        if (pushTagName) {
            auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(pushTagName, "the car had been taken");
            if (tag && !server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, GetBaseUserId(), server, session)) {
                session.SetErrorInfo("delegation", "delegation.finished.push_failed", EDriveSessionResult::InternalError);
                return false;
            }
        }
    }
    return true;
};

bool IDelegationTag::OnAfterPerformImpl(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!GetOffer() || GetOffer()->GetObjectId() != self.GetObjectId() || GetOffer()->GetUserId() != permissions.GetUserId()) {
        session.SetErrorInfo("delegation", "delegation.incorrect_offer", EDriveSessionResult::InconsistencyUser);
        return false;
    }
    TTaggedObject obj;
    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreObject(self.GetObjectId(), obj, session)) {
        return false;
    }

    TDBTag cTag = obj.GetFirstTagByClass<TChargableTag>();
    if (!cTag) {
        session.SetErrorInfo("delegation", "delegation.incorrect_chargable", EDriveSessionResult::InconsistencyUser);
        return false;
    }

    if (!FinishOriginalSession(cTag, server, session)) {
        return false;
    }

    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTag(self, permissions.GetUserId(), server, session, false)) {
        return false;
    }

    GetOffer()->SetTransferredFrom(SessionId);
    GetOffer()->SetTransferType(GetDelegationType());
    TChargableTag::TBookOptions bookOptions;
    bookOptions.MultiRent = true;
    if (!TChargableTag::Book(GetOffer(), permissions, *server, session, bookOptions)) {
        return false;
    }

    return true;
}

bool IDelegationTag::OnAfterAddImpl(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    TTaggedObject obj;
    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreObject(self.GetObjectId(), obj, session)) {
        session.SetErrorInfo("delegation", "delegation.cannot_restore_object", EDriveSessionResult::InternalError);
        return false;
    }

    const TVector<TDBTag> dTags = obj.GetTagsByClass<IDelegationTag>();
    for (auto&& i : dTags) {
        if (i.GetTagId() != self.GetTagId()) {
            session.SetErrorInfo("delegation", "delegation.exists_already", EDriveSessionResult::InconsistencyUser);
            return false;
        }
    }

    TDBTag cTag = obj.GetFirstTagByClass<TChargableTag>();
    if (!cTag) {
        session.SetErrorInfo("delegation", "delegation.incorrect_chargable", EDriveSessionResult::InconsistencyUser);
        return false;
    }

    if (cTag->GetName() != "old_state_parking" && cTag->GetName() != "old_state_reservation") {
        session.SetErrorInfo("delegation", "delegation.incorrect_state", EDriveSessionResult::InconsistencyUser);
        return false;
    }

    if (cTag->GetPerformer() != GetBaseUserId()) {
        session.SetErrorInfo("delegation", "delegation.incorrect_user", EDriveSessionResult::InconsistencyUser);
        return false;
    }

    THolder<TDelegationUserTag> udTag(new TDelegationUserTag(TDelegationUserTag::TypeName));
    udTag->SetObjectId(self.GetObjectId());
    udTag->SetSessionId(GetSessionId());
    udTag->SetType(GetDelegationType());

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(udTag.Release(), userId, userId, server, session)) {
        return false;
    }

    return true;
}

NJson::TJsonValue TDelegationUserTag::GetReport(const TDBTag& self, const NDrive::IServer* server) const {
    NJson::TJsonValue result;
    result.InsertValue("object_id", ObjectId);
    result.InsertValue("session_id", SessionId);
    result.InsertValue("tag_id", self.GetTagId());
    result.InsertValue("finished", !!FinishInstant);
    result.InsertValue("type", ToString(Type));
    if (TargetUserId) {
        auto maybeUser = server->GetDriveAPI()->GetUsersData()->GetCachedObject(TargetUserId);
        if (maybeUser) {
            result.InsertValue("p2p_user_phone", maybeUser->GetPhone());
            result.InsertValue("p2p_user_name", maybeUser->GetShortName());
            result.InsertValue("p2p_user_id", maybeUser->GetUserId());
        }
    }
    return result;
}

NDrive::TScheme TDelegationUserTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("object_id", "Делегированный объект");
    result.Add<TFSString>("target_user_id", "Целевой пользователь, если он есть");
    result.Add<TFSString>("session_id", "id сессии, из которой объект был делегирован");
    result.Add<TFSVariants>("type", "Тип делегирования").InitVariants<ECarDelegationType>();
    result.Add<TFSNumeric>("finished_instant", "Завершилось").SetVisual(TFSNumeric::EVisualType::DateTime);
    return result;
}

bool TDelegationUserTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    TTaggedObject object;
    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreObject(ObjectId, object, session)) {
        return false;
    }

    TVector<TDBTag> deviceTags = object.GetTagsByClass<IDelegationTag>();
    for (const auto& objectTag : deviceTags) {
        if (!objectTag) {
            continue;
        }
        if (Yensured(objectTag.GetTagAs<IDelegationTag>())->GetBaseUserId() != self.GetObjectId()) {
            NDrive::TEventLog::Log("DelegationError", NJson::TMapBuilder
                ("func", "TDelegationUserTag::OnAfterRemove")
                ("error", "BaseUserId in cer tag != ObjectId in user tag")
                ("object_tag", objectTag.SerializeToJson())
                ("user_tag", self.SerializeToJson())
            );
            continue;
        }
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTag(objectTag, userId, server, session, false, true)) {
            session.SetErrorInfo("delegation", "delegation.cancel_delegation.object_is_busy", EDriveSessionResult::InconsistencyUser);
            return false;
        }
    }
    return true;
}

NDrive::NProto::TDelegationUserTag TDelegationUserTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetObjectId(ObjectId);
    proto.SetSessionId(SessionId);
    proto.SetType(ToString(Type));
    proto.SetTargetUserId(TargetUserId);
    if (!!FinishInstant) {
        proto.SetFinishInstant(FinishInstant->Seconds());
    }
    return proto;
}

bool TDelegationUserTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    ObjectId = proto.GetObjectId();
    SessionId = proto.GetSessionId();
    TargetUserId = proto.GetTargetUserId();
    if (proto.HasFinishInstant()) {
        FinishInstant = TInstant::Seconds(proto.GetFinishInstant());
    }
    if (!TryFromString(proto.GetType(), Type)) {
        WARNING_LOG << "Could not parse delegation type: " << proto.GetType() << Endl;
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TIncomingDelegationUserTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("delegated_object_id", "Делегированный объект");
    result.Add<TFSString>("delegator_name", "От кого пришло предложение на делегирование");
    result.Add<TFSVariants>("type", "Тип делегирования").InitVariants<ECarDelegationType>();
    return result;
}

NDrive::NProto::TIncomingDelegationUserTag TIncomingDelegationUserTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetObjectId(ObjectId);
    proto.SetDelegatorName(DelegatorName);
    proto.SetType(ToString(Type));
    return proto;
}

bool TIncomingDelegationUserTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    ObjectId = proto.GetObjectId();
    DelegatorName = proto.GetDelegatorName();
    if (!TryFromString(proto.GetType(), Type)) {
        WARNING_LOG << "Could not parse delegation type: " << proto.GetType() << Endl;
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

bool TIncomingDelegationUserTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    Y_UNUSED(userId);
    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        return false;
    }
    TString notifierName = description->GetNotifierName();
    if (notifierName.empty()) {
        notifierName = "drive_push";
    }
    auto notifier = server->GetNotifier(notifierName);
    if (!notifier) {
        return true;
    }

    auto maybeCar = server->GetDriveAPI()->GetCarsData()->GetObject(ObjectId);
    if (!maybeCar) {
        return false;
    }
    auto modelCode = maybeCar->GetModel();
    auto modelFetchResult = server->GetDriveAPI()->GetModelsData()->FetchInfo(modelCode, TInstant::Zero());
    auto modelPtr = modelFetchResult.GetResultPtr(modelCode);
    if (!modelPtr) {
        return false;
    }

    auto targetUser = server->GetDriveAPI()->GetUsersData()->RestoreUser(self.GetObjectId(), session);
    if (!targetUser) {
        session.AddErrorMessage("IncomingDelegationUserTag::OnAfterAdd", "cannot RestoreUser");
        return false;
    }

    if (notifier) {
        notifier->Notify(server->GetLocalization()->FormatIncomingDelegationMessage(ELocalization::Rus, DelegatorName, modelPtr->GetName()), NDrive::INotifier::TContext().SetRecipients(TVector<TUserContacts>({*targetUser})));
    }
    return true;
}

bool TIncomingDelegationUserTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/) {
    TJsonProcessor::Read(json, "delegated_object_id", ObjectId);
    TJsonProcessor::Read(json, "delegator_name", DelegatorName);
    JREAD_FROM_STRING(json, "type", Type);
    return true;
}

NDrive::TScheme TIncomingDelegationUserTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSString>("notifier_name", "имя нотификатора").SetDefault("drive_push").SetRequired(true);
    return result;
}

NJson::TJsonValue TIncomingDelegationUserTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeMetaToJson();
    NJson::InsertField(json, "notifier_name", NotifierName);
    return json;
}

bool TIncomingDelegationUserTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "notifier_name", NotifierName);
        TBase::DoDeserializeMetaFromJson(json);
}

TTagDescription::TFactory::TRegistrator<TIncomingDelegationUserTag::TDescription> TIncomingDelegationUserTag::TDescription::Registrator(TIncomingDelegationUserTag::TypeName);
