#pragma once

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/offers/abstract.h>
#include <drive/backend/proto/tags.pb.h>
#include <drive/backend/tags/abstract.h>

#include <rtline/util/json_processing.h>

class TBillingSession;

enum class ECarDelegationType: ui32 {
    Free = 0 /* "free" */,
    P2P = 1 /* "p2p" */,
    P2PPassOffer = 2 /* "p2p_pass_offer" */,
};

class IDelegationTag {
private:
    R_FIELD(IOffer::TPtr, Offer);
    R_FIELD(TString, BaseUserId);
    R_FIELD(TString, SessionId);

protected:
    virtual bool FinishOriginalSession(const TDBTag& cTag, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    virtual bool DoShrinkPackOfferOnDelegation(ICommonOffer::TPtr& /*offer*/, const TBillingSession& /*bSession*/) const {
        return true;
    }

public:
    virtual bool IsAutoChargable() const {
        return true;
    }

    virtual ECarDelegationType GetDelegationType() const = 0;

    virtual NJson::TJsonValue GetReport(const TDBTag& self, const NDrive::IServer* /*server*/) const {
        NJson::TJsonValue result;
        result.InsertValue("tag_id", self.GetTagId());
        result.InsertValue("type", ToString(GetDelegationType()));
        return result;
    }

    void PatchScheme(const NDrive::IServer* /*server*/, NDrive::TScheme& result) const;

    void SerializeToJson(NJson::TJsonValue& json) const {
        TJsonProcessor::Write(json, "base_user_id", BaseUserId);
        TJsonProcessor::Write(json, "session_id", SessionId);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/) {
        if (!TJsonProcessor::Read(json, "base_user_id", BaseUserId)) {
            return false;
        }
        TJsonProcessor::Read(json, "session_id", SessionId);
        return true;
    }

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        proto.SetBaseUserId(BaseUserId);
        proto.SetSessionId(SessionId);
    }

    template <class TProto>
    bool DeserializeFromProto(const TProto& proto) {
        BaseUserId = proto.GetBaseUserId();
        SessionId = proto.GetSessionId();
        return true;
    }

    bool OnAfterRemoveImpl(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    bool OnAfterPerformImpl(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    bool OnAfterAddImpl(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const;

    virtual bool ShrinkPackOfferOnDelegation(ICommonOffer::TPtr& offer, const TBillingSession& bSession) const final;
};

class TDelegationUserTag: public INativeSerializationTag<NDrive::NProto::TDelegationUserTag> {
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TDelegationUserTag>;
    R_FIELD(TString, ObjectId);
    R_FIELD(TString, SessionId);
    R_OPTIONAL(TInstant, FinishInstant);
    R_FIELD(ECarDelegationType, Type, ECarDelegationType::Free);

    R_FIELD(TString, TargetUserId);

    static TFactory::TRegistrator<TDelegationUserTag> Registrator;
public:
    static const TString TypeName;

    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    NJson::TJsonValue GetReport(const TDBTag& self, const NDrive::IServer* server) const;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    virtual bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

    virtual TProto DoSerializeSpecialDataToProto() const override;

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
};

class TIncomingDelegationUserTag: public ISerializableTag<NDrive::NProto::TIncomingDelegationUserTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TIncomingDelegationUserTag>;

    R_FIELD(TString, ObjectId);
    R_FIELD(TString, DelegatorName);

    R_FIELD(ECarDelegationType, Type, ECarDelegationType::P2P);

    static TFactory::TRegistrator<TIncomingDelegationUserTag> Registrator;

public:
    static const TString TypeName;

    class TDescription : public TTagDescription {
    public:
        using TBase = TTagDescription;
    private:
        R_FIELD(TString, NotifierName);

    private:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& json) override;

    private:
        static TFactory::TRegistrator<TDescription> Registrator;
    };

    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    NJson::TJsonValue GetReport(const TDBTag& self) const {
        NJson::TJsonValue result;
        result.InsertValue("object_id", ObjectId);
        result.InsertValue("tag_id", self.GetTagId());
        result.InsertValue("type", ToString(Type));
        result.InsertValue("delegator_name", DelegatorName);

        auto now = Now();
        result.InsertValue("expired", now >= GetSLAInstant());
        result.InsertValue("time_remaining", (GetSLAInstant() - now).Seconds());

        return result;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::User};
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
};

template <class TProto>
class TDelegationTag: public INativeSerializationTag<TProto>, public IDelegationTag {
private:
    using TBase = INativeSerializationTag<TProto>;
public:
    using TBase::TBase;

    virtual TMaybe<NEntityTagsManager::EMultiplePerformersPolicy> GetMultiPerformingAbility() const override {
        return NEntityTagsManager::EMultiplePerformersPolicy::Allow;
    }

    virtual TString GetPerformGroup() const override {
        return "delegation";
    }

    virtual bool OnBeforeAdd(const TString& /*objectId*/, const TString& userId, const NDrive::IServer* /*server*/, NDrive::TEntitySession& /*session*/) override {
        IDelegationTag::SetBaseUserId(userId);
        return true;
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        IDelegationTag::PatchScheme(server, result);
        return result;
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Car};
    }

    virtual TProto DoSerializeSpecialDataToProto() const override {
        TProto proto = TBase::DoSerializeSpecialDataToProto();
        IDelegationTag::SerializeToProto(proto);
        return proto;
    }

    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        if (!DeserializeFromProto(proto)) {
            return false;
        }
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }
};

class TFreeDelegationTag: public TDelegationTag<NDrive::NProto::TFreeDelegationTag> {
private:
    using TBase = TDelegationTag<NDrive::NProto::TFreeDelegationTag>;
    static TFactory::TRegistrator<TFreeDelegationTag> Registrator;
public:
    static const TString TypeName;

    using TBase::TBase;

    ECarDelegationType GetDelegationType() const override {
        return ECarDelegationType::Free;
    }

    virtual bool OnAfterPerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const override {
        return TBase::OnAfterPerformImpl(self, permissions, server, session);
    }

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override {
        return TBase::OnAfterAddImpl(self, userId, server, session);
    }

    virtual bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override {
        return TBase::OnAfterRemoveImpl(self, userId, server, session);
    }
};

class TP2PDelegationTag: public TDelegationTag<NDrive::NProto::TP2PDelegationTag> {
private:
    using TProto = NDrive::NProto::TP2PDelegationTag;
    using TBase = TDelegationTag<NDrive::NProto::TP2PDelegationTag>;
    static TFactory::TRegistrator<TP2PDelegationTag> Registrator;

    R_FIELD(TString, P2PUserId);
    R_FIELD(TString, P2PUserPhone);
    R_FIELD(TString, P2PUserName);
    R_FIELD(bool, Rejected, false);
    R_FIELD(bool, PreserveOffer, false);

protected:
    virtual bool DoShrinkPackOfferOnDelegation(ICommonOffer::TPtr& offer, const TBillingSession& bSession) const override;

public:
    static const TString TypeName;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    using TBase::TBase;

    NTagActions::TTagActions GetVisibilityFeatures(const TUserPermissions& permissions) const override;

    ECarDelegationType GetDelegationType() const override {
        return PreserveOffer ? ECarDelegationType::P2PPassOffer : ECarDelegationType::P2P;
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    NJson::TJsonValue GetReport(const TDBTag& self, const NDrive::IServer* server) const override;
    void SerializeToJson(NJson::TJsonValue& json) const;
    bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/);

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Car};
    }
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual bool OnAfterPerform(TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
};
