#include "delivery.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/field.h>

NDrive::TScheme TCarDeliveryTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSVariants>("deliveree_tags", "User tags to add to the deliveree on evolution").SetReference("user_tags").SetMultiSelect(true);
    return result;
}

NJson::TJsonValue TCarDeliveryTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TCarDeliveryTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TTagDescription::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

NDrive::TScheme TCarDeliveryTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("distance", "Allowed distance from the car to the delivery point for completition").SetDefault(Distance);
    result.Add<TFSNumeric>("latitude").SetPrecision(6);
    result.Add<TFSNumeric>("longitude").SetPrecision(6);
    return result;
}

bool TCarDeliveryTag::HasLocation() const {
    return std::abs(Latitude) > 0.001 || std::abs(Longitude) > 0.001;
}

bool TCarDeliveryTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(userId);
    if (!HasLocation()) {
        auto snapshot = server->GetSnapshotsManager().GetSnapshot(objectId);
        auto location = snapshot.GetLocation();
        if (!location) {
            session.SetErrorInfo("CarDeliveryTag::OnBeforeAdd", "cannot get location");
            return false;
        }
        Latitude = location->Latitude;
        Longitude = location->Longitude;
    }
    return true;
}

bool TCarDeliveryTag::OnBeforeEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    Y_UNUSED(fromTag);
    Y_UNUSED(permissions);
    Y_UNUSED(server);
    Y_UNUSED(session);
    Y_UNUSED(eContext);
    auto targetDeliveryTag = std::dynamic_pointer_cast<TCarDeliveryTag>(toTag);
    if (targetDeliveryTag && !targetDeliveryTag->HasLocation()) {
        targetDeliveryTag->SetDistance(Distance);
        targetDeliveryTag->SetLatitude(Latitude);
        targetDeliveryTag->SetLongitude(Longitude);
    }
    return true;
}

bool TCarDeliveryTag::OnBeforeRemove(const TDBTag& self, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (GetPerformer()) {
        auto snapshot = Yensured(server)->GetSnapshotsManager().GetSnapshot(self.GetObjectId());
        auto destination = TGeoCoord(Longitude, Latitude);
        auto location = snapshot.GetLocation();
        if (!location) {
            session.SetErrorInfo("CarDeliveryTag::OnBeforeRemove", "car location is undefined", NDrive::MakeError("device_location_undefined"));
            return false;
        }
        auto distance = location->GetCoord().GetLengthTo(destination);
        if (distance > Distance) {
            session.SetErrorInfo(
                "CarDeliveryTag::OnBeforeRemove",
                TStringBuilder() << "car location is too distant: " << distance,
                NDrive::MakeError("destination_not_close_enough")
            );
            return false;
        }
    }
    return true;
}

bool TCarDeliveryTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    if (GetPerformer()) {
        const auto api = server ? server->GetDriveAPI() : nullptr;
        const auto& tagsManager = Yensured(api)->GetTagsManager();
        auto d = tagsManager.GetTagsMeta().GetDescriptionByName(GetName());
        auto description = std::dynamic_pointer_cast<const TDescription>(d);
        if (!description) {
            tx.SetErrorInfo("CarDeliveryTag::OnAfterRemove", "cannot acquire description for tag " + GetName());
            return false;
        }

        auto optionalSessions = api->GetSessionManager().GetObjectSessions(self.GetObjectId(), tx);
        if (!optionalSessions) {
            return false;
        }

        const auto& sessions = *optionalSessions;
        for (auto&& session : sessions) {
            if (!session) {
                continue;
            }
            if (session->GetClosed() || session->GetUserId() == GetPerformer()) {
                continue;
            }
            const auto& deliveree = session->GetUserId();
            for (auto&& tagName : description->GetDelivereeTags()) {
                auto tag = tagsManager.GetTagsMeta().CreateTag(tagName);
                if (!tag) {
                    tx.SetErrorInfo("CarDeliveryTag::OnAfterRemove", "cannot create tag " + tagName);
                    return false;
                }
                auto addedTag = tagsManager.GetUserTags().AddTag(tag, userId, deliveree, server, tx);
                if (!addedTag) {
                    return false;
                }
            }
        }
    }
    return true;
}

bool TCarDeliveryTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return
        TBase::DoSpecialDataFromJson(value, errors) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

void TCarDeliveryTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}

ITag::TFactory::TRegistrator<TCarDeliveryTag> TCarDeliveryTag::Registrator(TCarDeliveryTag::Type());

TTagDescription::TFactory::TRegistrator<TCarDeliveryTag::TDescription> CarDeliveryTagDescriptionRegistrator(TCarDeliveryTag::Type());
