#include "device_tags.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/proto/tags.pb.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/string/cast.h>

const TString TDeviceTagRecord::TypeName = "simple_car_tag";
ITag::TFactory::TRegistrator<TDeviceTagRecord> TDeviceTagRecord::Registrator(TDeviceTagRecord::TypeName);

const TString TDeviceAdditionalFeature::TypeName = "additional_feature_tag";
ITag::TFactory::TRegistrator<TDeviceAdditionalFeature> TDeviceAdditionalFeature::Registrator(TDeviceAdditionalFeature::TypeName);
TDeviceAdditionalFeature::TDescription::TFactory::TRegistrator<TDeviceAdditionalFeature::TDescription> TDeviceAdditionalFeature::TDescription::Registrator(TDeviceAdditionalFeature::TypeName);

const TString TConfirmableTag::TypeName = "confirmable_tag";
ITag::TFactory::TRegistrator<TConfirmableTag> TConfirmableTag::Registrator(TConfirmableTag::TypeName);
TConfirmableTag::TDescription::TFactory::TRegistrator<TConfirmableTag::TDescription> TConfirmableTag::TDescription::Registrator(TConfirmableTag::TypeName);

NDrive::TScheme TDeviceAdditionalFeature::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSString>("action_name", "Имя связанного со свойством действия");
    result.Add<TFSString>("action_icon", "Иконка связанного со свойством действия");
    result.Add<TFSString>("action_progress_name", "Описание связанного со свойством действия во включенном состоянии");
    result.Add<TFSString>("action_style", "Стиль отрисовки связанного со свойством действия");
    result.Add<TFSString>("action_color", "Цвет отрисовки свойства");
    result.Add<TFSString>("description", "Детальное описание свойства");
    result.Add<TFSString>("public_icon", "Иконка для отображения свойства");
    result.Add<TFSBoolean>("is_important", "Отображать как важное").SetDefault(false);
    result.Add<TFSBoolean>("visibility", "Видимость для пользователя").SetDefault(true);
    result.Add<TFSBoolean>("show_during_session", "Видимость во время бронирования").SetDefault(false);
    result.Add<TFSBoolean>("show_fullscreen", "Вывести полноэкранное уведомление в начале бронирования").SetDefault(false);
    return result;
}

NJson::TJsonValue TDeviceAdditionalFeature::TDescription::DoBuildJsonReport(ELocalization locale) const {
    NJson::TJsonValue result;
    if (ActionName) {
        result["action_name"] = ActionName;
    }
    if (ActionIcon) {
        result["action_icon"] = ActionIcon;
    }
    if (ActionProgressName) {
        result["action_progress_name"] = ActionProgressName;
    }
    if (ActionStyle) {
        result["action_style"] = ActionStyle;
    }
    if (Color) {
        result["action_color"] = Color;
    }
    if (Description) {
        auto localization = NDrive::HasServer() ? NDrive::GetServer().GetLocalization() : nullptr;
        result["description"] = localization ? localization->ApplyResources(Description, locale) : Description;
    }
    if (PublicIcon) {
        result["public_icon"] = PublicIcon;
    }
    if (ShowDuringSession) {
        result["show_during_session"] = ShowDuringSession;
    }
    if (ShowFullscreen) {
        result["show_fullscreen"] = ShowFullscreen;
    }
    result["is_important"] = IsImportant;
    return result;
}

NJson::TJsonValue TDeviceAdditionalFeature::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
    JWRITE_DEF(jsonMeta, "action_name", ActionName, "");
    JWRITE_DEF(jsonMeta, "action_icon", ActionIcon, "");
    JWRITE_DEF(jsonMeta, "action_progress_name", ActionProgressName, "");
    JWRITE_DEF(jsonMeta, "action_style", ActionStyle, "");
    JWRITE_DEF(jsonMeta, "action_color", Color, "");
    JWRITE_DEF(jsonMeta, "description", Description, "");
    JWRITE_DEF(jsonMeta, "public_icon", PublicIcon, "");
    JWRITE(jsonMeta, "is_important", IsImportant);
    JWRITE(jsonMeta, "visibility", Visibility);
    JWRITE(jsonMeta, "show_during_session", ShowDuringSession);
    JWRITE(jsonMeta, "show_fullscreen", ShowFullscreen);
    return jsonMeta;
}

bool TDeviceAdditionalFeature::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_STRING_OPT(jsonMeta, "action_name", ActionName);
    JREAD_STRING_OPT(jsonMeta, "action_icon", ActionIcon);
    JREAD_STRING_OPT(jsonMeta, "action_progress_name", ActionProgressName);
    JREAD_STRING_OPT(jsonMeta, "action_style", ActionStyle);
    JREAD_STRING_OPT(jsonMeta, "action_color", Color);
    JREAD_STRING_OPT(jsonMeta, "description", Description);
    JREAD_STRING_OPT(jsonMeta, "public_icon", PublicIcon);
    JREAD_BOOL_OPT(jsonMeta, "is_important", IsImportant);
    JREAD_BOOL_OPT(jsonMeta, "visibility", Visibility);
    JREAD_BOOL_OPT(jsonMeta, "show_during_session", ShowDuringSession);
    JREAD_BOOL_OPT(jsonMeta, "show_fullscreen", ShowFullscreen);
    return true;
}

const TString TServiceTagRecord::TypeName = "car_service_tag";
ITag::TFactory::TRegistrator<TServiceTagRecord> TServiceTagRecord::Registrator(TServiceTagRecord::TypeName);
TTagDescription::TFactory::TRegistrator<TServiceTagRecord::TDescription> TServiceTagRecord::TDescription::Registrator(TServiceTagRecord::TypeName);

NDrive::TScheme TServiceTagRecord::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    {
        const TString workTypes = server->GetSettings().GetValueDef<TString>("administration.tags.work_types", "");
        TSet<TString> variants;
        StringSplitter(workTypes).SplitByString(",").SkipEmpty().Collect(&variants);
        result.Add<TFSVariants>("report_work_type", "Тип работ").SetVariants(variants);
    }
    {
        result.Add<TFSVariants>("sensors", "Отображаемые сенсоры").SetVariants(NDrive::NVega::GetSensorNames()).SetMultiSelect(true);
    }
    return result;
}

NJson::TJsonValue TServiceTagRecord::TDescription::DoBuildJsonReport(ELocalization locale) const {
    NJson::TJsonValue result = TBase::DoBuildJsonReport(locale);
    NJson::InsertField(result, "actions", GetAvailableCarActions());
    return result;
}

NJson::TJsonValue TServiceTagRecord::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
    NJson::InsertNonNull(result, "report_work_type", WorkType);
    NJson::TJsonValue sensors = NJson::JSON_ARRAY;
    for (auto&& sensor : Sensors) {
        sensors.AppendValue(ToString(sensor));
    }
    NJson::InsertField(result, "sensors", sensors);
    return result;
}

bool TServiceTagRecord::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return TBase::DoDeserializeMetaFromJson(value) &&
        NJson::ParseField(value["report_work_type"], WorkType) &&
        NJson::ParseField(value["sensors"], Sensors);
}

TTagDescription::TPtr TServiceTagRecord::GetMetaDescription(const TString& /*type*/) const {
    return nullptr;
}

NDrive::IObjectSnapshot::TFactory::TRegistrator<TImagesSnapshot> TImagesSnapshot::Registrator("images_snapshot");

const TString TPhotoRequestTagRecord::TypeName = "photo_request";
ITag::TFactory::TRegistrator<TPhotoRequestTagRecord> TPhotoRequestTagRecord::Registrator(TPhotoRequestTagRecord::TypeName);
TTagDescription::TFactory::TRegistrator<TPhotoRequestTagRecord::TDescription> TPhotoRequestTagRecord::TDescription::Registrator(TPhotoRequestTagRecord::TypeName);

NDrive::TScheme TPhotoRequestTagRecord::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("short_image", "Ссылка на картинку с описанием");
    result.Add<TFSString>("short_description", "Краткое описание");
    return result;
}

const TString TReplaceCarTag::TypeName = "car_replacing";
ITag::TFactory::TRegistrator<TReplaceCarTag> TReplaceCarTag::Registrator(TReplaceCarTag::TypeName);

void TReplaceCarTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE(json, "tmp_offer_id", TmpOfferId);
}

bool TReplaceCarTag::DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) {
    JREAD_STRING_OPT(jsonValue, "tmp_offer_id", TmpOfferId);
    return TBase::DoSpecialDataFromJson(jsonValue, errors);
}

TReplaceCarTag::TProto TReplaceCarTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetTmpOfferId(TmpOfferId);
    return proto;
}

bool TReplaceCarTag::DoDeserializeSpecialDataFromProto(const TReplaceCarTag::TProto& proto) {
    TmpOfferId = proto.GetTmpOfferId();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TRepairTagRecord::TypeName = "car_repair_tag";
ITag::TFactory::TRegistrator<TRepairTagRecord> TRepairTagRecord::Registrator(TRepairTagRecord::TypeName);
TTagDescription::TFactory::TRegistrator<TRepairTagRecord::TDescription> TRepairTagRecord::TDescription::Registrator(TRepairTagRecord::TypeName);

NDrive::TScheme TRepairTagRecord::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSVariants>("work_type", "Выполняемая работа").InitVariants<NMajorClient::EWorkType>().SetDefault(ToString(NMajorClient::EWorkType::None));
    return result;
}

NJson::TJsonValue TRepairTagRecord::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta(NJson::JSON_MAP);
    JWRITE(jsonMeta, "work_type", ToString(WorkType));
    return jsonMeta;
}

bool TRepairTagRecord::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_FROM_STRING_OPT(jsonMeta, "work_type", WorkType);
    return true;
}

void TRepairTagRecord::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE(json, "query_id", QueryId);
    JWRITE(json, "mileage", Mileage);
    JWRITE_INSTANT(json, "date", Date);
    JWRITE_DEF(json, "insurance_number", InsuranceNumber, "");
    JWRITE_DEF(json, "ticket_number", TicketNumber, "");
}

bool TRepairTagRecord::DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) {
    JREAD_STRING_OPT(jsonValue, "query_id", QueryId);
    JREAD_UINT_OPT(jsonValue, "mileage", Mileage);
    JREAD_INSTANT_OPT(jsonValue, "date", Date);
    JREAD_STRING_OPT(jsonValue, "insurance_number", InsuranceNumber);
    JREAD_STRING_OPT(jsonValue, "ticket_number", TicketNumber);
    return TBase::DoSpecialDataFromJson(jsonValue, errors);
}

TRepairTagRecord::TProto TRepairTagRecord::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetQueryId(QueryId);
    proto.SetMileage(Mileage);
    proto.SetTimestamp(Date.Seconds());
    proto.SetInsuranceNumber(InsuranceNumber);
    proto.SetTicketNumber(TicketNumber);
    return proto;
}

bool TRepairTagRecord::DoDeserializeSpecialDataFromProto(const TRepairTagRecord::TProto& proto) {
    QueryId = proto.GetQueryId();
    Mileage = proto.GetMileage();
    Date = TInstant::Seconds(proto.GetTimestamp());
    InsuranceNumber = proto.GetInsuranceNumber();
    TicketNumber = proto.GetTicketNumber();
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NDrive::TScheme TRepairTagRecord::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("date").SetVisual(TFSNumeric::EVisualType::DateTime).SetMin(0);
    result.Add<TFSNumeric>("mileage").SetMin(0);
    result.Add<TFSString>("query_id");
    result.Add<TFSString>("insurance_number");
    result.Add<TFSString>("ticket_number");
    return result;
}

NDrive::TScheme TConfirmableTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSNumeric>("confirmations_count", "Требуется подтверждений до применения").SetMin(1).SetMax(100);
    result.Add<TFSVariants>("double_confirmation_policy", "Политика повторного подтверждения одним пользователем").InitVariants<EDoubleConfirmationPolicy>();
    result.Add<TFSVariants>("self_confirmation_policy", "Политика подтверждения автором").InitVariants<ESelfConfirmationPolicy>();
    result.Add<TFSDuration>("proposition_livetime", "Время жизни предложения").SetDefault(TDuration::Zero());
    return result;
}

NJson::TJsonValue TConfirmableTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    JWRITE(jsonMeta, "confirmations_count", ConfirmationsCount);
    JWRITE(jsonMeta, "self_confirmable", SelfConfirmationPolicy == ESelfConfirmationPolicy::Accept);
    JWRITE(jsonMeta, "user_double_confirmable", DoubleConfirmationPolicy == EDoubleConfirmationPolicy::Accept);
    JWRITE(jsonMeta, "double_confirmation_policy", ToString(DoubleConfirmationPolicy));
    JWRITE(jsonMeta, "self_confirmation_policy", ToString(SelfConfirmationPolicy));
    JWRITE_DURATION(jsonMeta, "proposition_livetime", Livetime);
    return jsonMeta;
}

bool TConfirmableTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_INT_OPT(jsonMeta, "confirmations_count", ConfirmationsCount);
    if (jsonMeta.Has("double_confirmation_policy")) {
        if (!TryFromString(jsonMeta["double_confirmation_policy"].GetStringRobust(), DoubleConfirmationPolicy)) {
            return false;
        }
    } else {
        bool userDoubleConfirmable = false;
        JREAD_BOOL_OPT(jsonMeta, "user_double_confirmable", userDoubleConfirmable);
        DoubleConfirmationPolicy = userDoubleConfirmable ? EDoubleConfirmationPolicy::Accept : EDoubleConfirmationPolicy::Error;
    }
    if (jsonMeta.Has("self_confirmation_policy")) {
        if (!TryFromString(jsonMeta["self_confirmation_policy"].GetStringRobust(), SelfConfirmationPolicy)) {
            return false;
        }
    } else {
        bool selfConfirmable = false;
        JREAD_BOOL_OPT(jsonMeta, "self_confirmable", selfConfirmable);
        SelfConfirmationPolicy = selfConfirmable ? ESelfConfirmationPolicy::Accept : ESelfConfirmationPolicy::Error;
    }
    JREAD_DURATION_OPT(jsonMeta, "proposition_livetime", Livetime);
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}


const TString TMaintenanceTag::TypeName = "maintenance_tag";
ITag::TFactory::TRegistrator<TMaintenanceTag> TMaintenanceTag::Registrator(TMaintenanceTag::TypeName);

EUniquePolicy TMaintenanceTag::GetUniquePolicy() const {
    return EUniquePolicy::Rewrite;
}

TSet<NEntityTagsManager::EEntityType> TMaintenanceTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::Car };
}

void TMaintenanceTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "reason", ToString(Reason));
    NJson::InsertField(json, "current_mileage", CurrentMileage);
    NJson::InsertField(json, "required_mileage", RequiredMileage);
    NJson::InsertField(json, "critical_mileage", CriticalMileage);
    NJson::InsertField(json, "required_timestamp", RequiredTimestamp);
    NJson::InsertField(json, "critical_timestamp", CriticalTimestamp);
}

TMaintenanceTag::TProto TMaintenanceTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetReason((ui32)Reason);
    if (CurrentMileage) {
        proto.SetCurrentMileage(*CurrentMileage);
    }
    if (RequiredMileage) {
        proto.SetRequiredMileage(*RequiredMileage);
    }
    if (CriticalMileage) {
        proto.SetCriticalMileage(*CriticalMileage);
    }
    if (RequiredTimestamp) {
        proto.SetRequiredTimestamp(RequiredTimestamp->Seconds());
    }
    if (CriticalTimestamp) {
        proto.SetCriticalTimestamp(CriticalTimestamp->Seconds());
    }
    return proto;
}

bool TMaintenanceTag::DoDeserializeSpecialDataFromProto(const TMaintenanceTag::TProto& proto) {
    if (proto.HasReason()) {
        auto reason = static_cast<EReason>(proto.GetReason());
        if (GetEnumNames<EReason>().contains(reason)) {
            Reason = reason;
        }
    }
    if (proto.HasCurrentMileage()) {
        CurrentMileage = proto.GetCurrentMileage();
    }
    if (proto.HasRequiredMileage()) {
        RequiredMileage = proto.GetRequiredMileage();
    }
    if (proto.HasCriticalMileage()) {
        CriticalMileage = proto.GetCriticalMileage();
    }
    if (proto.HasRequiredTimestamp()) {
        RequiredTimestamp = TInstant::Seconds(proto.GetRequiredTimestamp());
    }
    if (proto.HasCriticalTimestamp()) {
        CriticalTimestamp = TInstant::Seconds(proto.GetCriticalTimestamp());
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}


const TString TNeedInsuranceUpdateTag::TypeName = "need_insurance_update";
ITag::TFactory::TRegistrator<TNeedInsuranceUpdateTag> TNeedInsuranceUpdateTag::Registrator(TNeedInsuranceUpdateTag::TypeName);

EUniquePolicy TNeedInsuranceUpdateTag::GetUniquePolicy() const {
    return EUniquePolicy::SkipIfExists;
}

TSet<NEntityTagsManager::EEntityType> TNeedInsuranceUpdateTag::GetObjectType() const {
    return {NEntityTagsManager::EEntityType::Car};
}

void TNeedInsuranceUpdateTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertNonNull(json, "provider", Provider);
    NJson::InsertNonNull(json, "update_info", UpdateInfo);
    NJson::InsertNonNull(json, "error_info", ErrorInfo);
    NJson::InsertField(json, "approved", Approved);
}

bool TNeedInsuranceUpdateTag::DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) {
    return NJson::ParseField(jsonValue["approved"], Approved)
        && NJson::ParseField(jsonValue["provider"], Provider)
        && NJson::ParseField(jsonValue["update_info"], UpdateInfo)
        && NJson::ParseField(jsonValue["error_info"], ErrorInfo)
        && TBase::DoSpecialDataFromJson(jsonValue, errors);
}

NDrive::TScheme TNeedInsuranceUpdateTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSVariants>("provider", "Идентификатор страховой").SetVariants(GetEnumAllValues<EProvider>()).SetMultiSelect(false).SetRequired(true);
    result.Add<TFSBoolean>("approved", "Обновить страховку");
    result.Add<TFSString>("update_info", "Информация об обновлении");
    return result;
}

bool TNeedInsuranceUpdateTag::ProvideDataOnEvolve(const TDBTag& fromTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    auto fromTagImpl = fromTag.GetTagAs<TNeedInsuranceUpdateTag>();
    if (fromTagImpl) {
        SetProvider(fromTagImpl->OptionalProvider());
        SetApproved(fromTagImpl->GetApproved());
        SetUpdateInfo(fromTagImpl->GetUpdateInfo());
    } else {
        session.SetErrorInfo("need_insurance_update", "ProvideDataOnEvolve", EDriveSessionResult::IncorrectCarTags);
        return false;
    }
    return TBase::ProvideDataOnEvolve(fromTag, permissions, server, session);
}

bool TNeedInsuranceUpdateTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeAdd(objectId, userId, server, session)) {
        return false;
    }
    TVector<TDBTag> tags;
    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({objectId}, {}, tags, session)) {
        return false;
    }
    for (const auto& tag : tags) {
        if (tag.Is<TNeedInsuranceUpdateTag>()) {
            session.SetErrorInfo("need_insurance_update", "OnBeforeAdd: tag already exists", EDriveSessionResult::IncorrectCarTags);
            return false;
        }
    }
    return true;
}

template <>
NJson::TJsonValue NJson::ToJson(const TNeedInsuranceUpdateTag::EProvider& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TMaybe<TNeedInsuranceUpdateTag::EProvider>& result) {
    TNeedInsuranceUpdateTag::EProvider provider;
    if (NJson::TryFromJson(value, NJson::Stringify(provider))) {
        result = provider;
        return true;
    }
    return false;
}
