#pragma once

#include "organization_tag.h"

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/history/propositions.h>
#include <drive/backend/major/entities.h>
#include <drive/backend/proto/tags.pb.h>

class TConfirmableTag: public ISerializableTag<NDrive::NProto::TCommonTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TCommonTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TConfirmableTag> Registrator;

public:
    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;
        R_FIELD(ui32, ConfirmationsCount, 1);
        R_FIELD(EDoubleConfirmationPolicy, DoubleConfirmationPolicy, EDoubleConfirmationPolicy::Error);
        R_FIELD(ESelfConfirmationPolicy, SelfConfirmationPolicy, ESelfConfirmationPolicy::Error);
        R_FIELD(TDuration, Livetime, TDuration::Zero());

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Car, NEntityTagsManager::EEntityType::User};
    }
};

class TDeviceTagRecord: public ISerializableTag<NDrive::NProto::TCommonTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TCommonTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TDeviceTagRecord> Registrator;

public:
    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Car};
    }
};

class TDeviceAdditionalFeature: public ISerializableTag<NDrive::NProto::TCommonTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TCommonTagData>;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TDeviceAdditionalFeature> Registrator;

public:
    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        R_FIELD(TString, ActionName);
        R_FIELD(TString, ActionIcon);
        R_FIELD(TString, ActionProgressName);
        R_FIELD(TString, ActionStyle);
        R_FIELD(TString, Color);
        R_FIELD(TString, Description);
        R_FIELD(TString, PublicIcon);
        R_FIELD(bool, IsImportant, false);
        R_FIELD(bool, Visibility, true);
        R_FIELD(bool, ShowDuringSession, false);
        R_FIELD(bool, ShowFullscreen, false);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoBuildJsonReport(ELocalization locale) const override;

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

public:
    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Car};
    }
};

class TServiceTagRecord: public TDeviceTagRecord {
private:
    using TBase = TDeviceTagRecord;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TServiceTagRecord> Registrator;

public:
    using TDeviceTagRecord::TDeviceTagRecord;

    virtual TTagDescription::TPtr GetMetaDescription(const TString& type) const override;

    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

    public:
        R_FIELD(TSet<NDrive::TSensorId>, Sensors);
        R_FIELD(TString, WorkType);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    protected:
        virtual NJson::TJsonValue DoBuildJsonReport(ELocalization locale) const override;
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };
};

class TPhotoRequestTagRecord: public TDeviceTagRecord {
private:
    using TBase = TDeviceTagRecord;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TPhotoRequestTagRecord> Registrator;

public:
    using TDeviceTagRecord::TDeviceTagRecord;

    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;
        R_FIELD(TString, ShortImage);
        R_FIELD(TString, ShortDescription);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
            JWRITE(jsonMeta, "short_image", ShortImage);
            JWRITE(jsonMeta, "short_description", ShortDescription);
            return jsonMeta;
        }

        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            JREAD_FROM_STRING_OPT(jsonMeta, "short_image", ShortImage);
            JREAD_FROM_STRING_OPT(jsonMeta, "short_description", ShortDescription);
            return TBase::DoDeserializeMetaFromJson(jsonMeta);
        }
    };
};

class TReplaceCarTag: public ISerializableTag<NDrive::NProto::TReplaceCarTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TReplaceCarTag>;

private:
    R_FIELD(TString, TmpOfferId);
public:
    static const TString TypeName;
    static TFactory::TRegistrator<TReplaceCarTag> Registrator;

public:
    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Car};
    }

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) override;

protected:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
};

class TRepairTagRecord: public ISerializableTag<NDrive::NProto::TMajorTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TMajorTagData>;

private:
    R_FIELD(TString, QueryId);
    R_FIELD(TInstant, Date);
    R_FIELD(ui32, Mileage, 0);
    R_FIELD(TString, InsuranceNumber);
    R_FIELD(TString, TicketNumber);

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TRepairTagRecord> Registrator;

public:
    using TBase::TBase;

    class TDescription: public TTagDescription {
    private:
        R_FIELD(NMajorClient::EWorkType, WorkType, NMajorClient::EWorkType::None);

    private:
        static TFactory::TRegistrator<TDescription> Registrator;

    public:
        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    };

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return {NEntityTagsManager::EEntityType::Car};
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) override;

protected:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
};

class TMaintenanceTag: public ISerializableTag<NDrive::NProto::TMaintenanceTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TMaintenanceTagData>;

public:
    enum class EReason {
        Unknown = 0 /* "unknown" */,
        First = 1 /* "first" */,
        Intermediate = 2 /* "intermediate" */,
        Plan = 3 /* "plan" */,
        Timeout = 4 /* "timeout" */,
    };

private:
    R_FIELD(EReason, Reason, EReason::Unknown);
    R_OPTIONAL(ui64, CurrentMileage);
    R_OPTIONAL(ui64, RequiredMileage);
    R_OPTIONAL(ui64, CriticalMileage);
    R_OPTIONAL(TInstant, RequiredTimestamp);
    R_OPTIONAL(TInstant, CriticalTimestamp);

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TMaintenanceTag> Registrator;

public:
    using TBase::TBase;
    virtual EUniquePolicy GetUniquePolicy() const override;
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;

private:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;

protected:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;
};

class TNeedInsuranceUpdateTag: public IJsonSerializableTag {
public:
    enum class EProvider {
        Major,
    };

private:
    using TBase = IJsonSerializableTag;
    R_OPTIONAL(EProvider, Provider);
    R_FIELD(bool, Approved, false);
    R_FIELD(TString, UpdateInfo);
    R_FIELD(TString, ErrorInfo);

public:
    using TBase::TBase;
    static const TString TypeName;
    static TFactory::TRegistrator<TNeedInsuranceUpdateTag> Registrator;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;
    virtual EUniquePolicy GetUniquePolicy() const override;
    virtual bool ProvideDataOnEvolve(const TDBTag& fromTag, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& session) override;
    virtual bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

protected:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& jsonValue, TMessagesCollector* errors) override;
};
