#include "dictionary_tags.h"

#include <drive/backend/context_fetcher/json.h>
#include <drive/backend/processor/params_processor.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/http/misc/httpcodes.h>

NDrive::TScheme TDictionaryTagDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    NDrive::TScheme& fields = result.Add<TFSArray>("fields", "Поля для ввода").SetElement<NDrive::TScheme>();
    fields.Add<TFSString>("name", "Имя для отображени в форме");
    fields.Add<TFSString>("default", "Значение по умолчанию");
    fields.Add<TFSString>("id", "ID для отправки на сервер");
    fields.Add<TFSString>("description", "Описание");
    fields.Add<TFSVariants>("type", "Тип данных").InitVariants<NDrive::EElementType>();
    fields.Add<TFSBoolean>("materialize_default_value", "Явно проставлять значение по умолчанию");
    return result;
}

NJson::TJsonValue TDictionaryTagDescription::DoSerializeMetaToJson() const{
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    NJson::TJsonValue& fieldsJson = jsonMeta.InsertValue("fields", NJson::JSON_ARRAY);
    for (auto&& i : Fields) {
        NJson::TJsonValue pair;
        pair["name"] = i.Name;
        pair["default"] = i.DefaultValue;
        pair["type"] = ::ToString(i.Type);
        pair["id"] = i.Id;
        pair["materialize_default_value"] = i.MaterializeDefaultValue;
        pair["description"] = i.Description;
        fieldsJson.AppendValue(pair);
    }
    return jsonMeta;
}

bool TDictionaryTagDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    const auto& fields = jsonMeta["fields"];
    if (fields.IsDefined() && !fields.IsArray()) {
        return false;
    }
    for (auto&& item : fields.GetArray()) {
        TField field;
        if (!NJson::ParseField(item["name"], field.Name, false)
            || !NJson::ParseField(item["id"], field.Id, false)
            || !NJson::ParseField(item["default"], field.DefaultValue, false)
            || !NJson::ParseField(item["materialize_default_value"], field.MaterializeDefaultValue, false)
            || !NJson::ParseField(item["type"], NJson::Stringify(field.Type), false)
            || !NJson::ParseField(item["description"], field.Description, false)) {
            return false;
        }
        Fields.insert(field);
    }
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}

TMaybe<TString> TDictionaryTagDescription::GetDefaults(const TString& fieldId) const {
    for (auto&& item : Fields) {
        if (item.Id == fieldId) {
            return item.DefaultValue;
        }
    }
    return {};
}

const TString TUserDictionaryTag::TypeName = "user_dictionary_tag";
ITag::TFactory::TRegistrator<TUserDictionaryTag> TUserDictionaryTag::Registrator(TUserDictionaryTag::TypeName);
TTagDescription::TFactory::TRegistrator<TDictionaryTagDescription> TUserDictionaryTag::RegistratorDescription(TUserDictionaryTag::TypeName);

const TString TCarDictionaryTag::TypeName = "car_dictionary_tag";
ITag::TFactory::TRegistrator<TCarDictionaryTag> TCarDictionaryTag::Registrator(TCarDictionaryTag::TypeName);
TTagDescription::TFactory::TRegistrator<TDictionaryTagDescription> TCarDictionaryTag::RegistratorDescription(TCarDictionaryTag::TypeName);

const TString TAccountDictionaryTag::TypeName = "account_dictionary_tag";
ITag::TFactory::TRegistrator<TAccountDictionaryTag> TAccountDictionaryTag::Registrator(TAccountDictionaryTag::TypeName);
TTagDescription::TFactory::TRegistrator<TDictionaryTagDescription> TAccountDictionaryTag::RegistratorDescription(TAccountDictionaryTag::TypeName);

TString IDictionaryTag::GetServiceReport(const ITagsMeta& tagsManager) const {
    auto description = tagsManager.GetDescriptionByName(GetName());
    if (!description) {
        return "";
    }
    TMap<TString, TString> names;
    const TDictionaryTagDescription* dictDescription = dynamic_cast<const TDictionaryTagDescription*>(description.Get());
    if (dictDescription) {
        for (auto&& field : dictDescription->GetFields()) {
            names[field.Id] = field.Name;
        }
    }
    TString report;
    for (auto& field : Fields) {
        if (names.contains(field.Key)) {
            report += names[field.Key];
        } else {
            report += field.Key;
        }
        report += ": " + field.Value + "\n";
    }
    report += "Комментарий: " + GetComment();
    return report;
}

bool IDictionaryTag::PrepareFields(const NDrive::IServer* server, NDrive::TInfoEntitySession& session) {
    auto description = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    auto dictionaryTagDescription = std::dynamic_pointer_cast<const TDictionaryTagDescription>(description);
    if (!dictionaryTagDescription) {
        session.SetErrorInfo("DictionaryTag::OnBeforeAdd", "cannot get DictionaryTagDescription");
        return false;
    }

    TSet<TString> unique;
    for (auto&& field : Fields) {
        if (!unique.insert(field.Key).second) {
            session.SetErrorInfo(GetName(), "Duplicated field " + field.Key, EDriveSessionResult::IncorrectRequest);
            return false;
        }
    }
    for (auto&& field : dictionaryTagDescription->GetFields()) {
        if (unique.contains(field.Id)) {
            continue;
        }
        if (field.DefaultValue && field.MaterializeDefaultValue) {
            TField f;
            f.Key = field.Id;
            f.Value = field.DefaultValue;
            Fields.push_back(std::move(f));
        }
    }
    return true;
}

bool IDictionaryTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(objectId);
    Y_UNUSED(userId);
    if (!PrepareFields(server, session)) {
        return false;
    }
    return true;
}

bool IDictionaryTag::CopyOnEvolve(const ITag& source, const TTagEvolutionAction* evolution, const NDrive::IServer& server) {
    if (!TBase::CopyOnEvolve(source, evolution, server)) {
        return false;
    }
    NDrive::TInfoEntitySession session;
    if (!PrepareFields(&server, session)) {
        return false;
    }
    return true;
}

NDrive::TScheme IDictionaryTag::GetScheme(const NDrive::IServer* server) const {
    const TDriveAPI* driveApi = server->GetDriveAPI();
    CHECK_WITH_LOG(!!driveApi);

    NDrive::TScheme scheme = TBase::GetScheme(server);
    auto description = driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    if (!description) {
        ERROR_LOG << "Can't read description for " << GetName() << Endl;
        return scheme;
    }

    auto descriptionImpl = std::dynamic_pointer_cast<const TDictionaryTagDescription>(description);

    NDrive::TScheme& fields = scheme.Add<TFSArray>("fields", "Форма").SetElement<NDrive::TScheme>();
    TVector<TString> keys;
    if (descriptionImpl) {
        for (auto&& field : descriptionImpl->GetFields()) {
            keys.push_back(field.Id);
        }
    }
    fields.Add<TFSVariants>("key", "Ключ").SetVariants(keys).SetMultiSelect(false).SetEditable(true);
    fields.Add<TFSString>("value", "Значение");
    return scheme;
}

NJson::TJsonValue IDictionaryTag::GetPublicReport(ELocalization locale, const NDrive::IServer* server, TUserPermissions::TPtr permissions, const TSet<TString>& fields) const {
    NJson::TJsonValue result = NJson::JSON_ARRAY;

    TMaybe<TJsonFetchContext> context;
    TSet<TString> specialUserOptions;
    if (permissions) {
        for (auto&& i : permissions->GetUserOptions()) {
            specialUserOptions.emplace(i->GetOptionId());
            if (i->GetVisible()) {
                result.AppendValue(i->GetPublicReport(locale));
            }
        }
        context.ConstructInPlace(server, permissions->GetPublicReport());
    }

    Y_ENSURE_BT(server);
    auto description = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    if (!description) {
        return result;
    }
    const TDictionaryTagDescription* descriptionImpl = dynamic_cast<const TDictionaryTagDescription*>(description.Get());
    if (!descriptionImpl) {
        return result;
    }

    auto localization = server->GetLocalization();
    Y_ENSURE_BT(localization);
    for (auto&& field : descriptionImpl->GetFields()) {
        if (specialUserOptions.contains(field.Id)) {
            continue;
        }
        if (!fields.empty() && !fields.contains(field.Id)) {
            continue;
        }
        NJson::TJsonValue item;
        item["name"] = localization->GetLocalString(locale, field.Name);
        item["type"] = ::ToString(field.Type);
        item["id"] = field.Id;
        if (field.Description) {
            item["description"] = localization->GetLocalString(locale, field.Description);
        }

        TString value = field.DefaultValue;
        for (auto&& f : Fields) {
            if (f.Key == field.Id) {
                value = f.Value;
            }
        }
        value = localization->ApplyResources(value, locale);
        if (context) {
            TMessagesCollector errors;
            const TString text = IJsonContextFetcher::ProcessText(value, *context, errors);
            if (!errors.HasMessages()) {
                value = text;
            }
        }
        item["value"] = value;
        result.AppendValue(item);
    }
    return result;
}

void IDictionaryTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::TJsonValue& fieldsJson = json.InsertValue("fields", NJson::JSON_ARRAY);
    for (auto&& field : Fields) {
        NJson::TJsonValue pair;
        pair["key"] = field.Key;
        pair["value"] = field.Value;
        fieldsJson.AppendValue(pair);
    }
}

bool IDictionaryTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    const auto& fields = json["fields"];
    if (fields.IsDefined() && !fields.IsArray()) {
        return false;
    }
    for (auto&& item : fields.GetArray()) {
        TField field;
        field.Key = item["key"].GetString();
        field.Value = item["value"].GetString();
        Fields.push_back(field);
    }
    return TBase::DoSpecialDataFromJson(json, errors);
}

IDictionaryTag::TProto IDictionaryTag::DoSerializeSpecialDataToProto() const {
    IDictionaryTag::TProto proto = TBase::DoSerializeSpecialDataToProto();
    for (auto&& field : Fields) {
        NDrive::NProto::TTagField* fieldProto = proto.AddFields();
        fieldProto->SetKey(field.Key);
        fieldProto->SetValue(field.Value);
    }
    return proto;
}

bool IDictionaryTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    for (ui32 i = 0; i < proto.FieldsSize(); ++i) {
        TField field;
        field.Key = proto.GetFields(i).GetKey();
        field.Value = proto.GetFields(i).GetValue();
        Fields.push_back(field);
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

TMaybe<TString> IDictionaryTag::GetField(TStringBuf key) const {
    for (auto&& field : Fields) {
        if (field.Key == key) {
            return field.Value;
        }
    }
    return Nothing();
}

TMaybe<TString> IDictionaryTag::RemoveField(TStringBuf key) {
    auto field = Fields.begin();
    for (; field != Fields.end(); ++field) {
        if (field->Key == key) {
            break;
        }
    }
    if (field == Fields.end()) {
        return {};
    }
    auto result = std::move(field->Value);
    Fields.erase(field);
    return result;
}

void IDictionaryTag::SetField(const TString& key, const TString& value) {
    for (auto&& field : Fields) {
        if (field.Key == key) {
            field.Value = value;
            return;
        }
    }
    TField newItem;
    newItem.Key = key;
    newItem.Value = value;
    Fields.push_back(newItem);
}

void TUserDictionaryTag::SetSettings(const TString& tagId,
                                    const TVector<std::pair<TString, TString>>& settings,
                                    TUserPermissions::TConstPtr permissions,
                                    const NDrive::IServer& server,
                                    NDrive::TEntitySession& session) {
    const auto& userTagsManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
    TDBTag settingsTag;
    auto expectedTag = userTagsManager.RestoreTag(tagId, session);
    R_ENSURE(expectedTag, HTTP_INTERNAL_SERVER_ERROR, "cannot restore tag " << tagId, session);
    settingsTag = std::move(*expectedTag);
    R_ENSURE(permissions->GetTagNamesByAction(TTagAction::ETagAction::Update).contains(settingsTag->GetName()), HTTP_FORBIDDEN, "No permissions for update this type of tag", session);
    SetSettings(settingsTag, settings, permissions, server, session);
}

void TUserDictionaryTag::SetSettings(const TString& tagNameSetting,
                                    const TVector<std::pair<TString, TString>>& settings,
                                    TUserPermissions::TConstPtr permissions,
                                    const TString& userId,
                                    const NDrive::IServer& server,
                                    NDrive::TEntitySession& session) {
    TDBTag settingsTag = server.GetDriveAPI()->GetUserDictionaryTag(userId ? userId : permissions->GetUserId(), tagNameSetting, session);
    if (userId) {
        R_ENSURE(permissions->GetTagNamesByAction(TTagAction::ETagAction::Update).contains(settingsTag->GetName()), 403, "No permissions for update this type of tag");
    }
    SetSettings(settingsTag, settings, permissions, server, session);
}


void TUserDictionaryTag::SetSettings(TDBTag& settingsTag,
                                    const TVector<std::pair<TString, TString>>& settings,
                                    TUserPermissions::TConstPtr permissions,
                                    const NDrive::IServer& server,
                                    NDrive::TEntitySession& session) {
    R_ENSURE(settingsTag, HTTP_INTERNAL_SERVER_ERROR, "incorrect settings configuration", session);
    const auto& userTagsManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
    {
        const TString lockKey = settingsTag.GetTagId() ? settingsTag.GetTagId() : settingsTag->GetName() + "_" + settingsTag.GetObjectId();
        auto optionalTagIdLocked = session.TryLock(lockKey);
        R_ENSURE(optionalTagIdLocked, HTTP_INTERNAL_SERVER_ERROR, "cannot lock key " << lockKey, session);
        bool tagIdLocked = *optionalTagIdLocked;
        R_ENSURE(tagIdLocked, HTTP_CONFLICT, "cannot lock key " << lockKey);
        if (settingsTag.GetTagId()) {
            auto expectedTag = userTagsManager.RestoreTag(settingsTag.GetTagId(), session);
            R_ENSURE(expectedTag, HTTP_INTERNAL_SERVER_ERROR, "cannot restore tag " << settingsTag.GetTagId(), session);
            settingsTag = std::move(*expectedTag);
        } else {
            auto expectedTags = userTagsManager.RestoreEntityTags(settingsTag.GetObjectId(), {settingsTag->GetName()}, session);
            R_ENSURE(expectedTags, HTTP_INTERNAL_SERVER_ERROR, "cannot restore tags " << settingsTag->GetName(), session);
            if (!expectedTags->empty()) {
                settingsTag = std::move(expectedTags->front());
            }
        }
    }
    TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
    R_ENSURE(settingsData, HTTP_INTERNAL_SERVER_ERROR, "incorrect settings configuration");
    TSet<TString> removeEmpty = StringSplitter(server.GetSettings().GetValue<TString>("user.settings.remove_empty").GetOrElse({})).Split(',').SkipEmpty();
    for (auto&&[id, value] : settings) {
        if (value.empty() && removeEmpty.contains(id)) {
            settingsData->RemoveField(id);
        } else {
            settingsData->SetField(id, value);
        }
    }
    if (settingsTag.GetTagId()) {
        R_ENSURE(userTagsManager.UpdateTagData(settingsTag, permissions->GetUserId(), session), HTTP_INTERNAL_SERVER_ERROR, "can not update tag", session);
    } else {
        R_ENSURE(userTagsManager.AddTag(settingsTag.GetData(), permissions->GetUserId(), settingsTag.GetObjectId(), &server, session),
                        HTTP_INTERNAL_SERVER_ERROR, "can not add tag", session);
    }
}
