#pragma once

#include "user_tags.h"

#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/database/drive_api.h>

namespace NDrive {
    const TString ServiceSettingsTagNameSetting = "service_settings.tag_name";
    const TString UserSettingsTagNameSetting = "user_settings.tag_name";
    const TString AdminSettingsTagNameSetting = "admin_settings.tag_name";
    const TString UserFlagsTagNameSetting = "user_flags.tag_name";
    const TString UserCashbackDataTagNameSetting = "user_settings.cashback.tag_name";
    const TString B2BApplicationTagNameSetting = "b2b.application_form.tag_name";
}

class TDictionaryTagDescription: public TTagDescription {
public:
    struct TField {
        TString Name;
        NDrive::EElementType Type;
        TString Id;
        TString DefaultValue;
        TString Description;
        bool MaterializeDefaultValue = false;

        bool operator<(const TField& other) const {
            return Id < other.Id;
        }
    };

private:
    R_READONLY(TSet<TField>, Fields, {});

private:
    using TBase = TTagDescription;

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
    virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    TMaybe<TString> GetDefaults(const TString& fieldId) const;
};

class IDictionaryTag: public ISerializableTag<NDrive::NProto::TDictionaryTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TDictionaryTagData>;

    struct TField {
        TString Key;
        TString Value;

        bool operator<(const TField& other) const {
            return Key < other.Key;
        }
    };

private:
    R_READONLY(TVector<TField>, Fields, {});

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    TMaybe<TString> GetField(TStringBuf key) const;
    TMaybe<TString> RemoveField(TStringBuf key);
    void SetField(const TString& key, const TString& value);

    bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;
    virtual bool CopyOnEvolve(const ITag& source, const TTagEvolutionAction* evolution, const NDrive::IServer& server) override;
    NJson::TJsonValue GetPublicReport(ELocalization locale, const NDrive::IServer* server, TUserPermissions::TPtr permissions, const TSet<TString>& fields = {}) const;

    TString GetServiceReport(const ITagsMeta& tagsManager) const override;
private:
    bool PrepareFields(const NDrive::IServer* server, NDrive::TInfoEntitySession& session);
};

class TUserDictionaryTag : public IDictionaryTag {
public:
    using IDictionaryTag::IDictionaryTag;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return{ NEntityTagsManager::EEntityType::User };
    }

    static const TString TypeName;

public:
    static void SetSettings(const TString& tagId,
                            const TVector<std::pair<TString, TString>>& settings,
                            TUserPermissions::TConstPtr permissions,
                            const NDrive::IServer& server,
                            NDrive::TEntitySession& session);

    static void SetSettings(const TString& tagNameSetting,
                            const TVector<std::pair<TString, TString>>& settings,
                            TUserPermissions::TConstPtr permissions,
                            const TString& userId,
                            const NDrive::IServer& server,
                            NDrive::TEntitySession& session);

private:
    static void SetSettings(TDBTag& settingsTag,
                            const TVector<std::pair<TString, TString>>& settings,
                            TUserPermissions::TConstPtr permissions,
                            const NDrive::IServer& server,
                            NDrive::TEntitySession& session);

private:
    static TFactory::TRegistrator<TUserDictionaryTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TDictionaryTagDescription> RegistratorDescription;
};

class TCarDictionaryTag : public IDictionaryTag {
public:
    using IDictionaryTag::IDictionaryTag;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return{ NEntityTagsManager::EEntityType::Car };
    }

    static const TString TypeName;

private:
    static TFactory::TRegistrator<TCarDictionaryTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TDictionaryTagDescription> RegistratorDescription;
};

class TAccountDictionaryTag : public IDictionaryTag {
public:
    using IDictionaryTag::IDictionaryTag;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return{ NEntityTagsManager::EEntityType::Account };
    }

    static const TString TypeName;

private:
    static TFactory::TRegistrator<TAccountDictionaryTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TDictionaryTagDescription> RegistratorDescription;
};
