
#include "event_tag.h"

const TString TUserEventTag::TypeName = "user_event_tag";
ITag::TFactory::TRegistrator<TUserEventTag> TUserEventTag::Registrator(TUserEventTag::TypeName);

const TString TCarEventTag::TypeName = "car_event_tag";
ITag::TFactory::TRegistrator<TCarEventTag> TCarEventTag::Registrator(TCarEventTag::TypeName);


template<>
NJson::TJsonValue NJson::ToJson(const IEventTag::TEvent& event) {
    NJson::TJsonValue result;
    result["name"] = event.GetName();
    result["timestamp"] = event.GetTimestamp().Seconds();
    if (event.GetMeta().IsDefined()) {
        result["meta"] = event.GetMeta();
    }
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, IEventTag::TEvent& event) {
    return NJson::ParseField(value, "name", event.MutableName(), true)
        && NJson::ParseField(value, "timestamp", event.MutableTimestamp(), true)
        && NJson::ParseField(value, "meta", event.MutableMeta(), false);
}

NDrive::TScheme IEventTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    NDrive::TScheme& events = result.Add<TFSArray>("events", "События").SetElement<NDrive::TScheme>();
    events.Add<TFSString>("name", "Название события");
    events.Add<TFSNumeric>("timestamp", "Время срабатывания").SetVisual(NDrive::TFSNumeric::EVisualType::DateTime);
    result.Add<TFSJson>("meta", "Дополнительная информация").SetReadOnly(true);
    return result;
}

void IEventTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "events", Events);
}

bool IEventTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!NJson::ParseField(json, "events", Events, false)) {
        return false;
    }
    std::sort(Events.begin(), Events.end());
    return TBase::DoSpecialDataFromJson(json, errors);
}

IEventTag::TProto IEventTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TEventTag proto = TBase::DoSerializeSpecialDataToProto();
    for (auto&& event : Events) {
        NDrive::NProto::TEventTag::TEvent* protoEvent = proto.AddEvent();
        protoEvent->SetName(event.GetName());
        protoEvent->SetTimestamp(event.GetTimestamp().Seconds());
        if (event.GetMeta().IsDefined()) {
            protoEvent->SetMeta(event.GetMeta().GetStringRobust());
        }
    }
    return proto;
}

bool IEventTag::DoDeserializeSpecialDataFromProto(const IEventTag::TProto& proto) {
    for (auto&& protoEvent : proto.GetEvent()) {
        IEventTag::TEvent event;
        event.SetName(protoEvent.GetName());
        event.SetTimestamp(TInstant::Seconds(protoEvent.GetTimestamp()));
        event.SetMeta(NJson::ToJson(NJson::JsonString(protoEvent.GetMeta())));
        AddEvent(std::move(event));
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

ui32 IEventTag::GetEventCount(const TSet<TString>& eventNames, const TMaybe<TInstant> since /*= {}*/, const TMaybe<TInstant> until /*= {}*/) const {
    int result = 0;
    for (auto&& event : Events) {
        if ((!since || event.GetTimestamp() >= *since) && (!until || event.GetTimestamp() < *until) && eventNames.contains(event.GetName())) {
            ++result;
        }
    }
    return result;
}

bool IEventTag::HasEvents(const TSet<TString>& requestedEventNames) const {
    TSet<TString> actualEventNames;
    for (auto&& event : Events) {
        actualEventNames.emplace(event.GetName());
    }
    for (auto&& event : requestedEventNames) {
        if (!actualEventNames.contains(event)) {
            return false;
        }
    }
    return true;
}

bool IEventTag::HasEventsOrdered(const TVector<TString>& eventNames) const {
    auto nameIt = eventNames.begin();
    for (auto eventIt = Events.begin(); eventIt != Events.end() && nameIt != eventNames.end(); ++eventIt) {
        if (eventIt->GetName() == *nameIt) {
            ++nameIt;
        }
    }
    return nameIt == eventNames.end();
}

void IEventTag::AddEvent(IEventTag::TEvent&& event) {
    Events.emplace(std::upper_bound(Events.begin(), Events.end(), event), event);
}

void IEventTag::AddEvent(const TString& eventName, const TInstant eventTimestamp, NJson::TJsonValue meta /*= NJson::JSON_NULL*/) {
    AddEvent(IEventTag::TEvent(eventName, eventTimestamp, meta));
}

TMaybe<IEventTag::TEvent> IEventTag::GetLastEvent(const TString& eventName) const {
    for (auto revIt = Events.rbegin(); revIt != Events.rend(); ++revIt) {
        if (revIt->GetName() == eventName) {
            return *revIt;
        }
    }
    return Nothing();
}
