#pragma once

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/proto/tags.pb.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

class IEventTag: public ISerializableTag<NDrive::NProto::TEventTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TEventTag>;
    using TDescription = TTagDescription;

public:
    using TBase::TBase;

    class TEvent {
    private:
        R_FIELD(TString, Name);
        R_FIELD(TInstant, Timestamp);
        R_FIELD(NJson::TJsonValue, Meta, NJson::JSON_NULL);

    public:
        TEvent() = default;

        TEvent(TString name, TInstant timestamp, NJson::TJsonValue meta = NJson::JSON_NULL)
            : Name(name)
            , Timestamp(timestamp)
            , Meta(meta)
        {
        }

        bool operator<(const TEvent& event) const {
            if (Timestamp == event.GetTimestamp()) {
                return Name < event.GetName();
            }
            return Timestamp < event.GetTimestamp();
        }
    };

private:
    R_READONLY(TVector<TEvent>, Events);

public:
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override = 0;

    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    ui32 GetEventCount(const TSet<TString>& eventNames, const TMaybe<TInstant> since = {}, const TMaybe<TInstant> until = {}) const;
    bool HasEventsOrdered(const TVector<TString>& eventNames) const;
    bool HasEvents(const TSet<TString>& requestedEventNames) const;
    void AddEvent(TEvent&& event);
    void AddEvent(const TString& eventName, const TInstant eventTimestamp, NJson::TJsonValue meta = NJson::JSON_NULL);
    TMaybe<TEvent> GetLastEvent(const TString& eventName) const;
};

class TUserEventTag : public IEventTag {
public:
    using IEventTag::IEventTag;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return{ NEntityTagsManager::EEntityType::User };
    }

    static const TString TypeName;

private:
    static TFactory::TRegistrator<TUserEventTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TUserEventTag> RegistratorDescription;
};

class TCarEventTag : public IEventTag {
public:
    using IEventTag::IEventTag;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return{ NEntityTagsManager::EEntityType::Car };
    }

    static const TString TypeName;

private:
    static TFactory::TRegistrator<TCarEventTag> Registrator;
    static TTagDescription::TFactory::TRegistrator<TCarEventTag> RegistratorDescription;
};
