#include "events.h"

#include <drive/backend/proto/snapshot.pb.h>

#include <util/memory/blob.h>

TBlob TEventsSnapshot::DoSerializeToBlob() const {
    NDrive::NProto::TEventsSnapshot proto;
    for (auto&& info : PushInfos) {
        auto push = proto.AddPush();
        push->SetContent(info.Content);
        push->SetType(info.Type);
        push->SetTimestamp(info.Timestamp.MicroSeconds());
    }
    return TBlob::FromString(proto.SerializeAsString());
}

bool TEventsSnapshot::DoDeserializeFromBlob(const TBlob& data) {
    NDrive::NProto::TEventsSnapshot proto;
    if (!proto.ParseFromArray(data.Data(), data.Size())) {
        return false;
    }
    for (auto&& push : proto.GetPush()) {
        TPushInfo info;
        info.Content = push.GetContent();
        info.Type = push.GetType();
        info.Timestamp = TInstant::MicroSeconds(push.GetTimestamp());
        AddPush(std::move(info));
    }
    return true;
}

NJson::TJsonValue TEventsSnapshot::DoSerializeToJson() const {
    NJson::TJsonValue result;
    NJson::TJsonValue& pushes = result.InsertValue("pushes", NJson::JSON_ARRAY);
    for (auto&& info : PushInfos) {
        NJson::TJsonValue push;
        push["content"] = info.Content;
        push["type"] = info.Type;
        push["timestamp"] = info.Timestamp.MicroSeconds();
        pushes.AppendValue(std::move(push));
    }
    return result;
}

bool TEventsSnapshot::DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    const NJson::TJsonValue& pushes = jsonValue["pushes"];
    if (pushes.IsDefined() && !pushes.IsArray()) {
        return false;
    }
    for (auto&& push : pushes.GetArray()) {
        TPushInfo info;
        info.Content = push["content"].GetString();
        info.Type = push["type"].GetString();
        info.Timestamp = TInstant::MicroSeconds(push["timestamp"].GetUInteger());
        AddPush(std::move(info));
    }
    return true;
}

NDrive::IObjectSnapshot::TFactory::TRegistrator<TEventsSnapshot> PushSnapshotRegistrator("push");
