#include "evolution_policy.h"

#include "area_tags.h"
#include "chargable.h"
#include "user_tags.h"

#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/helpers.h>
#include <drive/backend/sessions/manager/billing.h>

bool TCarDelegationPolicy::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_FROM_STRING(json, "type", Type);
    if (Type == ECarDelegationType::P2P || Type == ECarDelegationType::P2PPassOffer) {
        JREAD_STRING_OPT(json, "user_id", TargetUserId);
    }
    DelegationRequest = true;
    return true;
}

NJson::TJsonValue TCarDelegationPolicy::SerializeToJson() const {
    NJson::TJsonValue result;
    result["type"] = ToString(Type);
    if (TargetUserId) {
        result["target_user_id"] = TargetUserId;
    }
    if (TagName) {
        result["tag_name"] = TagName;
    }
    return result;
}

ITag::TPtr TCarDelegationPolicy::CreateTag(const NDrive::IServer* server) const {
    ITag::TPtr result;
    switch (Type) {
        case ECarDelegationType::Free:
            {
                TAtomicSharedPtr<TFreeDelegationTag> dTag = new TFreeDelegationTag(TFreeDelegationTag::TypeName);
                dTag->SetBaseUserId(UserId);
                result = dTag;
            }
            break;
        case ECarDelegationType::P2PPassOffer:
        case ECarDelegationType::P2P:
            {
                TAtomicSharedPtr<TP2PDelegationTag> dTag = new TP2PDelegationTag(TP2PDelegationTag::TypeName);
                dTag->SetBaseUserId(UserId);
                dTag->SetP2PUserId(TargetUserId);
                dTag->SetPreserveOffer(Type == ECarDelegationType::P2PPassOffer);
                auto maybeTargetUser = server->GetDriveAPI()->GetUsersData()->GetCachedObject(TargetUserId);
                if (maybeTargetUser) {
                    dTag->SetP2PUserName(maybeTargetUser->GetShortName());
                    dTag->SetP2PUserPhone(maybeTargetUser->GetPhone());
                }
                dTag->SetSLAInstant(Now() + P2PDelegationDuration);
                result = dTag;
            }
            break;
    }
    return result;
}

bool TCarDelegationPolicy::NeedToStoreDelegation() const {
    return Type == ECarDelegationType::P2P || Type == ECarDelegationType::P2PPassOffer;
}

TString TCarDelegationPolicy::GetConfirmationLandingId(const TConstDBTag& fromTag, const NDrive::IServer* server) const {
    Y_UNUSED(fromTag);
    Y_UNUSED(server);
    if (Type == ECarDelegationType::Free) {
        return FreeDelegationConfirmationLanding;
    }
    return "";
}

bool TCarDelegationPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* server, const TUserPermissions& permissions, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& session) const {
    // block delegation in case of debt
    if (Type == ECarDelegationType::P2PPassOffer) {
        auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetTagSession(fromTag.GetTagId(), session);
        R_ENSURE(optionalSession, {}, "cannot GetTagSession " << fromTag.GetTagId(), session);
        const auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(*optionalSession);
        auto currentOffer = billingSession ? billingSession->GetCurrentOffer() : nullptr;
        if (currentOffer && currentOffer->CheckSession(permissions, session, server, false) != EDriveSessionResult::Success) {
            return false;
        }
    }

    const TString confirmationLandingId = GetConfirmationLandingId(fromTag, server);
    if (!!confirmationLandingId) {
        if (!!confirmationLandingId && confirmationLandingId != "NO_ALERT") {
            if (session.GetRequestContext().GetUserChoice() == "") {
                if (!server->GetDriveAPI()->GetLandingsDB()->InitLandingUserRequest({confirmationLandingId}, "evolution", session)) {
                    ALERT_LOG << "No error landing configured: " << confirmationLandingId << Endl;
                    session.SetErrorInfo("evolution", "no error landing", EDriveSessionResult::InternalError);
                }
                return false;
            } else if (session.GetRequestContext().GetUserChoice() != "accept") {
                session.SetErrorInfo("evolution", "problem", EDriveSessionResult::IncorrectUserReply);
                return false;
            }
        }
        if (!DelegationRequest) {
            session.SetErrorInfo("evolution", "problem", EDriveSessionResult::InconsistencyUser);
            return false;
        }
    }

    if (NeedToStoreDelegation()) {
        auto fromTagCopy = fromTag.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
        auto fromTagImpl = fromTagCopy.MutableTagAs<TChargableTag>();
        if (!fromTagImpl) {
            session.SetErrorInfo("evolution", "problem", EDriveSessionResult::InconsistencySystem);
            return false;
        }
        fromTagImpl->SetDelegationType(Type);
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagData(fromTagCopy, permissions.GetUserId(), session)) {
            return false;
        }
    }

    return true;
}

bool TCarDelegationPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TCarDelegationPolicy::ExecuteAfterEvolveCommit(const TString& objectId, const NDrive::ITag* tag, const TUserPermissions& permissions, const NDrive::IServer* server) const {
    if (tag->GetName() != TChargableTag::Parking) {
        NDrive::TEventLog::Log("CarDelegationPolicyError", NJson::TMapBuilder
            ("type", "BadTagName")
            ("tag", tag->GetName())
        );
        return false;
    }

    auto delegationTag = CreateTag(server);
    if (!delegationTag) {
        NDrive::TEventLog::Log("CarDelegationPolicyError", NJson::TMapBuilder
            ("type", "CreateTag")
        );
        return false;
    }
    auto delegationTagImpl = dynamic_cast<IDelegationTag*>(delegationTag.Get());
    if (delegationTagImpl) {
        auto sBuilder = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
        auto userSession = sBuilder->GetLastObjectSession(objectId);
        delegationTagImpl->SetSessionId(userSession->GetSessionId());
    }

    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(delegationTag, permissions.GetUserId(), objectId, server, session) || !session.Commit()) {
        NDrive::TEventLog::Log("CarDelegationPolicyError", NJson::TMapBuilder
            ("type", "AddTag")
            ("errors", session.GetReport())
        );
        return false;
    }
    return true;
}

bool TCarDelegationPolicy::CheckDelegationPossibleFromTag(const TConstDBTag& fromTag, ITag::TAggregateEvolutionPolicy& policyWithError) const {
    if ((Type == ECarDelegationType::P2P) && fromTag->GetName() != TChargableTag::Parking) {
        policyWithError
            .SetErrorMessage("You can only P2P delegate from parking")
            .SetUIErrorTitle("delegation.p2p.not_from_parking.title")
            .SetUIErrorMessage("delegation.p2p.not_from_parking.message");
        return false;
    }
    return true;
}

bool TRideFinishPolicy::ExecuteAfterEvolveCommit(const TString& objectId, const NDrive::ITag* tag, const TUserPermissions& permissions, const NDrive::IServer* server) const {
    Y_UNUSED(objectId);
    if (Yensured(tag)->GetName() != TChargableTag::Reservation) {
        NDrive::TEventLog::Log("RideFinishPolicyError", NJson::TMapBuilder
            ("type", "BadTagName")
            ("tag", tag->GetName())
        );
        return false;
    }
    auto impl = dynamic_cast<const TChargableTag*>(tag);
    if (!impl) {
        NDrive::TEventLog::Log("RideFinishPolicyError", NJson::TMapBuilder
            ("type", "BadTagType")
            ("tag_type", TypeName(*tag))
        );
        return false;
    }
    if (!!impl->GetAlertLandingId()) {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        if (!server->GetDriveAPI()->IncrementUserLandingAlertViewsInfo(permissions.GetUserId(), impl->GetAlertLandingId(), TDuration::Minutes(2), server, session) || !session.Commit()) {
            NDrive::TEventLog::Log("RideFinishPolicyError", NJson::TMapBuilder
                ("type", "IncrementUserLandingAlertViewsInfo")
                ("errors", session.GetReport())
            );
            return false;
        }
    }
    return true;
}

bool TRideFinishPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr toTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NDrive::ITag::TEvolutionContext* eContext, NDrive::TEntitySession& session) const {
    if (fromTag->GetName() == TChargableTag::Reservation || fromTag->GetName() == TChargableTag::Acceptance) {
        return true;
    }
    if (fromTag.GetTagId() != TagId) {
        session.SetErrorInfo("RideFinishPolicy::ExecuteBeforeEvolution", "TagId mismatch: " + fromTag.GetTagId() + "/" + TagId);
        return false;
    }
    TDropPolicyProvider* dpProvider = dynamic_cast<TDropPolicyProvider*>(toTag.Get());
    if (!dpProvider) {
        session.SetErrorInfo("TRideFinishPolicy::ExecuteBeforeEvolution", "incorrect provider", EDriveSessionResult::InternalError);
        return false;
    }

    double currentPrice = Max<ui32>();
    auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetTagSession(TagId, session);
    if (!optionalSession) {
        return false;
    } else {
        auto sessionTrace = *optionalSession;
        if (sessionTrace) {
            const TBillingSession* bSession = dynamic_cast<const TBillingSession*>(sessionTrace.Get());
            if (bSession) {
                TBillingSession::TBillingCompilation compilation;
                if (bSession->FillCompilation(compilation)) {
                    if (!bSession->GetCurrentOffer() || bSession->GetCurrentOffer()->CheckNeedFeesOnFinish(server)) {
                        currentPrice = compilation.GetBillingSumPrice();
                    }
                }
            }
        }
    }
    const auto allowDropDef = CarLocationFeatures.GetAllowDropDef(EDropAbility::NotAllow);
    if (allowDropDef == EDropAbility::Allow) {
        if (!CarLocationFeatures.HasFeeInfo() || (currentPrice >= CarLocationFeatures.GetFeeInfoUnsafe().GetFee() && CarLocationFeatures.GetFeeInfoUnsafe().GetPolicy() == EDropOfferPolicy::FeesMax)) {
            return true;
        }
        const bool forceMode = eContext && eContext->GetMode() != EEvolutionMode::Default;
        const auto viewsInfo = server->GetDriveAPI()->GetUserLandingAlertViewsInfo(permissions.GetUserId(), CarLocationFeatures.GetFeeInfoUnsafe().GetAlertType(), session);
        const ui32 viewsCount = viewsInfo ? viewsInfo->GetViews() : 0;

        const TString landingId = CarLocationFeatures.GetFeeInfoUnsafe().GetNextLanding(viewsCount);
        if (landingId != "NO_ALERT" && landingId != "" && landingId != "NO_LANDING" && !forceMode) {
            if (session.GetRequestContext().GetUserChoice() == "") {
                if (!server->GetDriveAPI()->GetLandingsDB()->InitLandingUserRequest({landingId}, "evolution", session)) {
                    ALERT_LOG << "No error landing configured: " << landingId << Endl;
                    session.SetErrorInfo("evolution", "no error landing", EDriveSessionResult::InternalError);
                }
                return false;
            } else if (session.GetRequestContext().GetUserChoice() != "accept") {
                session.SetErrorInfo("evolution", "problem", EDriveSessionResult::IncorrectUserReply);
                return false;
            }
            dpProvider->SetAlertLandingId(CarLocationFeatures.GetFeeInfoUnsafe().GetAlertType());
        }
        if (CarLocationFeatures.GetFeeInfoUnsafe().GetAlertsCount() <= viewsCount || forceMode) {
            dpProvider->SetFinishOfferFee(CarLocationFeatures.GetFeeInfoUnsafe().GetFee());
            dpProvider->SetFinishOfferFeePolicy(CarLocationFeatures.GetFeeInfoUnsafe().GetPolicy());
        }
    } else {
        auto allowIncorrectPosition = permissions.GetSetting<bool>("scooters.allow_incorrect_position").GetOrElse(false);
        auto forcedCompletion = session.GetRequestContext().GetUserChoice() == "accept";
        // scooters: outside allow_drop_car is ok (if allowIncorrectPosition is true), inside deny_drop_car is not
        if (allowIncorrectPosition && forcedCompletion && allowDropDef != EDropAbility::Deny) {
            auto evlog = NDrive::GetThreadEventLogger();
            if (evlog) {
                evlog->AddEvent(NJson::TMapBuilder("event", "AllowIncorrectPosition"));
            }
        } else if (permissions.GetSetting<TString>("scooters.finish_ride.incorrect_position_message", "") == "incorrect_scooter_position"
                    && (allowDropDef != EDropAbility::NotAllow || !allowIncorrectPosition)) {
            // scooters: if allowIncorrectPosition is true, we want to allow forcing outside of allow_drop_car
            // clients expect incorrect_car_position for it; incorrect_scooter_position will cause error to be shown otherwise
            session.SetErrorInfo("evolution", "problem", NDrive::MakeError("incorrect_scooter_position"));
            return false;
        } else {
            session.SetError(NDrive::MakeError("incorrect_car_position"));
            session.SetErrorInfo("evolution", "problem", EDriveSessionResult::IncorrectCarPosition);
            return false;
        }
    }
    return true;
}

bool TRideFinishPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TAskForSelfiePolicy::ExecuteAfterEvolveCommit(const TString& /*objectId*/, const NDrive::ITag* /*tag*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/) const {
    return true;
}

bool TAskForSelfiePolicy::ExecuteBeforeEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& session) const {
    if (VerificationLandingId != "NO_ALERT" && VerificationLandingId != "" && VerificationLandingId != "NO_LANDING") {
        if (!server->GetDriveAPI()->GetLandingsDB()->InitLandingUserRequest({VerificationLandingId}, "evolution", session)) {
            ALERT_LOG << "No selfie verification landing configured: " << VerificationLandingId << Endl;
            session.SetErrorInfo("evolution", "no selfie verification landing", EDriveSessionResult::InternalError);
        }
        return false;
    }
    return true;
}

bool TAskForSelfiePolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TPermitEvolutionPolicy::ExecuteAfterEvolveCommit(const TString& objectId, const NDrive::ITag* tag, const TUserPermissions& permissions, const NDrive::IServer* server) const {
    Y_UNUSED(objectId);
    Y_UNUSED(tag);
    TVector<TDBTag> tags;
    const auto& userTagManager = server->GetDriveAPI()->GetTagsManager().GetUserTags();
    auto session = userTagManager.BuildSession();
    if (!userTagManager.RestoreTags({permissions.GetUserId()}, {PermitTagName}, tags, session) || tags.empty()) {
        NDrive::TEventLog::Log("PermitEvolutionPolicyError", NJson::TMapBuilder
            ("type", "RestoreTags")
            ("errors", session.GetReport())
        );
        return false;
    }
    if (!userTagManager.RemoveTags(tags, permissions.GetUserId(), server, session)) {
        NDrive::TEventLog::Log("PermitEvolutionPolicyError", NJson::TMapBuilder
            ("type", "RemoveTags")
            ("errors", session.GetReport())
        );
        return false;
    }
    if (!session.Commit()) {
        NDrive::TEventLog::Log("PermitEvolutionPolicyError", NJson::TMapBuilder
            ("type", "Commit")
            ("errors", session.GetReport())
        );
        return false;
    }
    return true;
}

bool TPermitEvolutionPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr toTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NDrive::ITag::TEvolutionContext* eContext, NDrive::TEntitySession& session) const {
    Y_UNUSED(fromTag);
    Y_UNUSED(toTag);
    Y_UNUSED(eContext);
    auto optionalTaggedUser = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(permissions.GetUserId(), session);
    if (!optionalTaggedUser) {
        return false;
    }
    auto optionalTag = optionalTaggedUser->GetTag(PermitTagName);
    bool hasPermit = optionalTag.Defined();
    if (!hasPermit && PermitRequiredLandingId) {
        if (!server->GetDriveAPI()->GetLandingsDB()->InitLandingUserRequest({PermitRequiredLandingId}, "evolution", session)) {
            ALERT_LOG << "No permit requirement landing configured: " << PermitRequiredLandingId << Endl;
            session.SetErrorInfo("evolution", "no permit requirement verification landing", EDriveSessionResult::InternalError);
        }
        return false;
    }
    return true;
}

bool TPermitEvolutionPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TPassCheckEvolutionPolicy::ExecuteAfterEvolveCommit(const TString& /*objectId*/, const NDrive::ITag* /*tag*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/) const {
    return true;
}

bool TPassCheckEvolutionPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* server, const TUserPermissions& permissions, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& session) const {
    TVector<TTaggedUser> cachedUserTags;
    TSet<TString> tagNames = {PassTagName};
    if (IgnoreTagName) {
        tagNames.insert(IgnoreTagName);
    }
    auto optionalTaggedUser = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(permissions.GetUserId(), session);
    if (!optionalTaggedUser) {
        return false;
    }
    auto optionalPassTag = optionalTaggedUser->GetTag(PassTagName);
    auto optionalIgnoreTag = optionalTaggedUser->GetTag(IgnoreTagName);
    bool hasPass = optionalPassTag || optionalIgnoreTag;
    bool passActive = false;
    if (hasPass) {
        for (auto&& cachedTag : cachedUserTags.front().GetTags()) {
            if (IgnoreTagName && cachedTag.GetTypeId(cachedTag) == IgnoreTagName) {
                passActive = true;
                break;
            }
            const TUserIsolationPassTag* passTag = cachedTag.GetTagAs<TUserIsolationPassTag>();
            if (passTag) {
                if (passTag->IsValid() && fromTag.GetObjectId().equal(passTag->GetCarId())) {
                    passActive = true;
                    break;
                }
            }
        }
    }

    if (!hasPass || !passActive) {
        if (PassRequiredLandingId && !server->GetDriveAPI()->GetLandingsDB()->InitLandingUserRequest({PassRequiredLandingId}, "evolution", session)) {
            ALERT_LOG << "No pass requirement landing configured: " << PassRequiredLandingId << Endl;
            session.SetErrorInfo("evolution", "no pass requirement verification landing", EDriveSessionResult::InternalError);
        }

        auto chatRobot = server->GetChatRobot(PassChatName);
        if (!chatRobot) {
            return false;
        }

        auto currentOffer = server->GetDriveAPI()->GetCurrentCarOffer(fromTag.GetObjectId());
        auto topic = currentOffer ? currentOffer->GetOfferId() : "";

        chatRobot->EnsureChat(permissions.GetUserId(), topic, session, true);

        return false;
    }
    return true;
}

bool TPassCheckEvolutionPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TCloseChatEvolutionPolicy::ExecuteAfterEvolveCommit(const TString& objectId, const NDrive::ITag* tag, const TUserPermissions& permissions, const NDrive::IServer* server) const {
    Y_UNUSED(objectId);
    Y_UNUSED(tag);
    Y_UNUSED(permissions);
    Y_UNUSED(server);
    return true;
}

bool TCloseChatEvolutionPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr toTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NDrive::ITag::TEvolutionContext* eContext, NDrive::TEntitySession& session) const {
    Y_UNUSED(fromTag);
    Y_UNUSED(toTag);
    Y_UNUSED(eContext);
    auto optionalTags = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreEntityTags(permissions.GetUserId(), ChatTagNames, session);
    if (!optionalTags) {
        return false;
    }
    for (auto tag : *optionalTags) {
        auto deferrableTag = tag.GetTagAs<IDeferrableTag>();
        if (deferrableTag && !deferrableTag->OnClose(tag, Messages, permissions, session, server)) {
            return false;
        }
    }
    return true;
}

bool TCloseChatEvolutionPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TScooterFixEvolutionPolicy::ExecuteAfterEvolveCommit(const TString& /*objectId*/, const NDrive::ITag* /*tag*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/) const {
    return true;
}

bool TScooterFixEvolutionPolicy::ExecuteBeforeEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* eContext, NDrive::TEntitySession& session) const {
    if (fromTag->GetName() == TChargableTag::Reservation) {
        return true;
    }

    const auto offer = server->GetDriveAPI()->GetCurrentCarOffer(fromTag.GetObjectId());
    if (!offer) {
        return true;
    }

    const auto fixPointOffer = std::dynamic_pointer_cast<TFixPointOffer>(offer);
    if (!fixPointOffer) {
        return true;
    }

    if (CarLocationFeatures.GetAllowDropDef(EDropAbility::NotAllow) != EDropAbility::Allow) {
        return true;
    }

    const auto location = CarLocationFeatures.OptionalLocation();
    if (!location) {
        return true;
    }

    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder("event", "CheckFinishArea"));
    }

    const bool isInFinishArea = IsInFinishArea(location->GetCoord(), server, fixPointOffer->GetFinishArea(), fixPointOffer->GetTypeName()).GetOrElse(false);
    const bool forceMode = eContext && eContext->GetMode() != EEvolutionMode::Default;
    const bool forcedCompletion = session.GetRequestContext().GetUserChoice() == "accept";

    if (isInFinishArea || forcedCompletion || forceMode) {
        return true;
    }

    session.SetCode(HTTP_BAD_REQUEST);
    session.SetErrorInfo("evolution", "problem", NDrive::MakeError("scooter_is_not_in_fix_finish_area"));
    return false;
}

bool TScooterFixEvolutionPolicy::ExecuteAfterEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TEmailAfterRideEvolutionPolicy::ExecuteAfterEvolveCommit(const TString& /*objectId*/, const NDrive::ITag* /*tag*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/) const {
    return true;
}

bool TEmailAfterRideEvolutionPolicy::ExecuteBeforeEvolution(const TConstDBTag& /*fromTag*/, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* /*server*/, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& /*session*/) const {
    return true;
}

bool TEmailAfterRideEvolutionPolicy::ExecuteAfterEvolution(const TConstDBTag& fromTag, NDrive::ITag::TPtr /*toTag*/, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NDrive::ITag::TEvolutionContext* /*eContext*/, NDrive::TEntitySession& session) const {
    auto offer = server->GetDriveAPI()->GetCurrentCarOffer(fromTag.GetObjectId());
    R_ENSURE(offer, HTTP_INTERNAL_SERVER_ERROR, "offer is nullptr", session);

    auto driveApi = server->GetDriveAPI();
    const auto emailData = offer->GetEmailData(server, session, EmailType);
    if (emailData && !emailData->EmailTagName.Empty()) {
        auto tag = driveApi->GetTagsManager().GetTagsMeta().CreateTag(emailData->EmailTagName);
        if (auto emailTag = std::dynamic_pointer_cast<TUserMailTag>(tag); emailTag) {
            emailTag->SetTemplateArgs(std::move(emailData->Args));
            auto& emailArgs = emailTag->MutableTemplateArgs();
            emailArgs["json_data"] = ToString(emailData->JsonData);
            R_ENSURE(driveApi->GetTagsManager().GetUserTags().AddTag(tag, offer->GetUserId(), offer->GetUserId(), server, session), HTTP_INTERNAL_SERVER_ERROR, "cannot add tag", session);
        }
    }
    return true;
}
