#include "feedback.h"

#include "billing_tags.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

NDrive::TScheme TFeedbackTraceTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSString>("button_localization", "Локализация текста кнопки");
    result.Add<TFSString>("message_localization", "Локализация текста сообщения");
    result.Add<TFSString>("submessage_localization", "Локализация второго уровня текста сообщения");
    result.Add<TFSString>("bonus_button_localization", "Локализация текста кнопки при начислении бонусов");
    result.Add<TFSString>("bonus_message_localization", "Локализация текста сообщения при начислении бонусов");
    result.Add<TFSString>("bonus_submessage_localization", "Локализация второго уровня текста сообщения при начислении бонусов");
    result.Add<TFSArray>("bonus_required_user_tags", "Пользовательские теги, необходимые для начисления бонуса").SetElement<TFSString>();
    result.Add<TFSString>("bonus_tag", "Тег, начисляющий бонусы");
    result.Add<TFSString>("bonus_amount", "Количество бонусов (в копейках)");
    result.Add<TFSNumeric>("priority", "Приоритет");
    result.Add<TFSBoolean>("show", "Show the notification").SetDefault(Show);
    return result;
}

void TFeedbackTraceTag::TDescription::SetBonusInfo(const TString& tag, ui32 amount) {
    BonusTag = tag;
    BonusAmount = amount;
}

void TFeedbackTraceTag::TDescription::SetBonusLocalization(const TString& button, const TString& message, const TString& submessage) {
    BonusButtonLocalization = button;
    BonusMessageLocalization = message;
    BonusSubmessageLocalization = submessage;
}

void TFeedbackTraceTag::TDescription::SetLocalization(const TString& button, const TString& message, const TString& submessage) {
    ButtonLocalization = button;
    MessageLocalization = message;
    SubmessageLocalization = submessage;
}

NJson::TJsonValue TFeedbackTraceTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::InsertField(result, "show", Show);
    if (ButtonLocalization) {
        result["button_localization"] = ButtonLocalization;
    }
    if (MessageLocalization) {
        result["message_localization"] = MessageLocalization;
    }
    if (SubmessageLocalization) {
        result["submessage_localization"] = SubmessageLocalization;
    }
    if (BonusButtonLocalization) {
        result["bonus_button_localization"] = BonusButtonLocalization;
    }
    if (BonusMessageLocalization) {
        result["bonus_message_localization"] = BonusMessageLocalization;
    }
    if (BonusSubmessageLocalization) {
        result["bonus_submessage_localization"] = BonusSubmessageLocalization;
    }
    if (BonusRequiredUserTags) {
        result["bonus_required_user_tags"] = NJson::ToJson(BonusRequiredUserTags);
    }
    if (BonusTag) {
        result["bonus_tag"] = BonusTag;
    }
    if (BonusAmount) {
        result["bonus_amount"] = BonusAmount;
    }
    if (Priority) {
        result["priority"] = Priority;
    }
    return result;
}

bool TFeedbackTraceTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    if (!TTagDescription::DoDeserializeMetaFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["button_localization"], ButtonLocalization) &&
        NJson::ParseField(value["message_localization"], MessageLocalization) &&
        NJson::ParseField(value["submessage_localization"], SubmessageLocalization) &&
        NJson::ParseField(value["bonus_button_localization"], BonusButtonLocalization) &&
        NJson::ParseField(value["bonus_message_localization"], BonusMessageLocalization) &&
        NJson::ParseField(value["bonus_submessage_localization"], BonusSubmessageLocalization) &&
        NJson::ParseField(value["bonus_required_user_tags"], BonusRequiredUserTags) &&
        NJson::ParseField(value["bonus_tag"], BonusTag) &&
        NJson::ParseField(value["bonus_amount"], BonusAmount) &&
        NJson::ParseField(value["show"], Show) &&
        NJson::ParseField(value["priority"], Priority);
}

TFeedbackTraceTag::TResponse TFeedbackTraceTag::GetResponse(const NDrive::IServer* server) const {
    if (!server) {
        return {};
    }
    auto description = std::dynamic_pointer_cast<const TDescription>(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName()));
    if (!description) {
        return {};
    }
    return GetResponse(*description);
}

TFeedbackTraceTag::TResponse TFeedbackTraceTag::GetResponse(const TDescription& description) const {
    TResponse result;
    result.Priority = description.GetPriority();
    result.Show = description.ShouldShow();
    if (BonusAmount) {
        result.Bonuses = BonusAmount;
        result.BonusTag = BonusTag;
        result.ButtonLocalization = description.GetBonusButtonLocalization();
        result.MessageLocalization = description.GetBonusMessageLocalization();
        result.SubmessageLocalization = description.GetBonusSubmessageLocalization();
    } else {
        result.ButtonLocalization = description.GetButtonLocalization();
        result.MessageLocalization = description.GetMessageLocalization();
        result.SubmessageLocalization = description.GetSubmessageLocalization();
    }
    return result;
}

TFeedbackTraceTag::TResponse TFeedbackTraceTag::MergeResponses(const TResponse* base, const TResponse& delta) {
    if (base) {
        return MergeResponses(*base, delta);
    } else {
        return delta;
    }
}

TFeedbackTraceTag::TResponse TFeedbackTraceTag::MergeResponses(const TResponse& base, const TResponse& delta) {
    auto result = base;
    if (delta) {
        if (result < delta) {
            result = delta;
        }
    }
    return result;
}

NJson::TJsonValue TFeedbackTraceTag::FormatResponse(const TResponse& response, ELocalization locale, const NDrive::IServer* server) {
    if (!server) {
        return {};
    }
    const ILocalization* localization = server->GetLocalization();
    if (!localization) {
        return {};
    }

    TString buttonLocalization = response ? response.ButtonLocalization : "session.feedback.default_button";
    TString messageLocalization = response ? response.MessageLocalization : "session.feedback.default_message";
    TString submessageLocalization = response ? response.SubmessageLocalization : "";

    TString button = localization->GetLocalString(locale, buttonLocalization);
    TString message = localization->GetLocalString(locale, messageLocalization);
    TString submessage = localization->GetLocalString(locale, submessageLocalization);

    if (response.Bonuses) {
        constexpr TStringBuf placeholder = "_Bonus_";
        auto bonuses = localization->FormatPrice(locale, response.Bonuses);
        SubstGlobal(button, placeholder, bonuses);
        SubstGlobal(message, placeholder, bonuses);
        SubstGlobal(submessage, placeholder, bonuses);
    }

    NJson::TJsonValue result;
    result["button"] = button;
    result["message"] = message;
    if (submessage) {
        result["submessage"] = submessage;
    }
    return result;
}

bool TFeedbackTraceTag::Set(const TString& deviceId, const TString& userId, TInstant timestamp, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    DeviceId = deviceId;
    UserId = userId;
    Timestamp = timestamp;
    auto optionalBonus = CalculateBonus(timestamp, server, session);
    if (!optionalBonus) {
        return false;
    }
    std::tie(BonusTag, BonusAmount) = *optionalBonus;
    return true;
}

void TFeedbackTraceTag::CancelBonus() {
    BonusTag.clear();
    BonusAmount = 0;
}

bool TFeedbackTraceTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto& tagsManager = server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
    if (BonusTag) {
        auto tag = tagsManager.CreateTag(BonusTag);
        if (!tag) {
            session.SetErrorInfo("FeedbackTraceTag::OnAfterAdd", "cannot create tag " + BonusTag);
            return false;
        }
        auto operationTag = std::dynamic_pointer_cast<TOperationTag>(tag);
        if (operationTag) {
            operationTag->SetAmount(BonusAmount);
            operationTag->SetSessionId(self.GetObjectId());
        }
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, UserId, server, session)) {
            return false;
        }
    }
    return true;
}

TMaybe<std::pair<TString, ui32>> TFeedbackTraceTag::CalculateBonus(TInstant timestamp, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const std::pair<TString, ui32> nothing = { TString(), 0 };
    if (!server) {
        return nothing;
    }

    const ITagsMeta& tagsMeta = server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
    auto description = std::dynamic_pointer_cast<const TDescription>(tagsMeta.GetDescriptionByName(GetName()));
    if (!description) {
        description = MakeAtomicShared<TDescription>();
    }

    const ui32 bonusAmount = description->GetBonusAmount();
    const TString& bonusTag = description->GetBonusTag();
    if (!bonusTag) {
        return nothing;
    }

    TSet<TString> bonusTags;
    auto descriptions = tagsMeta.GetTagsByType(Type());
    for (auto&& d : descriptions) {
        auto dd = std::dynamic_pointer_cast<const TDescription>(d);
        if (dd) {
            bonusTags.insert(dd->GetBonusTag());
        }
    }

    const TString& userId = UserId;
    const TUserTagsManager& userTags = server->GetDriveAPI()->GetTagsManager().GetUserTags();
    if (!description->GetBonusRequiredUserTags().empty()) {
        auto optionalTaggedUser = userTags.RestoreObject(userId, session);
        if (!optionalTaggedUser) {
            session.AddErrorMessage("FeedbackTraceTag::CalculateBonus", "cannot RestoreObject");
            return {};
        }
        bool found = false;
        for (auto&& name : description->GetBonusRequiredUserTags()) {
            auto optionalTag = optionalTaggedUser->GetTag(name);
            if (optionalTag) {
                found = true;
                break;
            }
        }
        if (!found) {
            return nothing;
        }
    }
    if (!IsFeedbackEnabled(server, *description, session)) {
        NDrive::TEventLog::Log("DisableFeedbackBonus", NJson::JSON_MAP);
        return nothing;
    }

    auto optionalEvents = userTags.GetEventsByObject(userId, session, /*sinceId=*/0, timestamp - description->GetHistoryDepth());
    if (!optionalEvents) {
        session.AddErrorMessage("FeedbackTraceTag::CalculateBonus", "cannot GetEventsByObject");
        return {};
    }

    ui32 selfCount = 0;
    ui32 totalCount = 0;
    for (auto&& ev : *optionalEvents) {
        if (!ev) {
            continue;
        }
        if (ev.GetHistoryAction() != EObjectHistoryAction::Add) {
            continue;
        }
        auto tag = ev.GetData();
        if (!tag) {
            continue;
        }
        if (bonusTag == tag->GetName()) {
            ++selfCount;
        }
        if (bonusTags.contains(tag->GetName())) {
            ++totalCount;
        }
    }
    if (selfCount >= description->GetMaxSelfCount()) {
        return nothing;
    }
    if (totalCount >= description->GetMaxTotalCount()) {
        return nothing;
    }
    return std::make_pair(bonusTag, bonusAmount);
}

bool TFeedbackTraceTag::IsFeedbackEnabled(const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const ITagsMeta& tagsMeta = server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
    auto description = std::dynamic_pointer_cast<const TDescription>(tagsMeta.GetDescriptionByName(GetName()));
    if (!description) {
        description = MakeAtomicShared<TDescription>();
    }
    return IsFeedbackEnabled(server, *description, session);
}

bool TFeedbackTraceTag::IsFeedbackEnabled(const NDrive::IServer* server, const TDescription& description, NDrive::TEntitySession& session) const {
    const TUserTagsManager& userTags = server->GetDriveAPI()->GetTagsManager().GetUserTags();
    if (!description.GetBonusDisableUserTags().empty()) {
        auto optionalTaggedUser = userTags.RestoreObject(UserId, session);
        if (!optionalTaggedUser) {
            return false;
        }
        for (auto&& name : description.GetBonusDisableUserTags()) {
            auto optionalTag = optionalTaggedUser->GetTag(name);
            if (optionalTag) {
                return false;
            }
        }
    }
    return true;
}

TFeedbackTraceTag::TProto TFeedbackTraceTag::DoSerializeSpecialDataToProto() const {
    TProto result = TBase::DoSerializeSpecialDataToProto();
    result.SetDeviceId(DeviceId);
    result.SetUserId(UserId);
    result.SetTimestamp(Timestamp.Seconds());
    result.SetBonusTag(BonusTag);
    result.SetBonusAmount(BonusAmount);
    return result;
}

bool TFeedbackTraceTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    DeviceId = proto.GetDeviceId();
    UserId = proto.GetUserId();
    Timestamp = TInstant::Seconds(proto.GetTimestamp());
    BonusTag = proto.GetBonusTag();
    BonusAmount = proto.GetBonusAmount();
    return true;
}

ITag::TFactory::TRegistrator<TFeedbackTraceTag> FeedbackTraceTagRegistrator(TFeedbackTraceTag::Type());
TTagDescription::TFactory::TRegistrator<TFeedbackTraceTag::TDescription> FeedbackTraceTagDescriptionRegistrator(TFeedbackTraceTag::Type());
