#pragma once

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/proto/tags.pb.h>

class TFeedbackTraceTag : public INativeSerializationTag<NDrive::NProto::TFeedbackTraceTagData> {
public:
    class TDescription: public TTagDescription {
    public:
        using TTagDescription::TTagDescription;

        const TString& GetButtonLocalization() const {
            return ButtonLocalization;
        }
        const TString& GetMessageLocalization() const {
            return MessageLocalization;
        }
        const TString& GetSubmessageLocalization() const {
            return SubmessageLocalization;
        }

        const TString& GetBonusButtonLocalization() const {
            return BonusButtonLocalization;
        }
        const TString& GetBonusMessageLocalization() const {
            return BonusMessageLocalization;
        }
        const TString& GetBonusSubmessageLocalization() const {
            return BonusSubmessageLocalization;
        }

        const TSet<TString>& GetBonusDisableUserTags() const {
            return BonusDisableUserTags;
        }
        const TSet<TString>& GetBonusRequiredUserTags() const {
            return BonusRequiredUserTags;
        }
        const TString& GetBonusTag() const {
            return BonusTag;
        }
        ui32 GetBonusAmount() const {
            return BonusAmount;
        }

        ui32 GetMaxSelfCount() const {
            return MaxSelfCount;
        }
        ui32 GetMaxTotalCount() const {
            return MaxTotalCount;
        }
        ui32 GetPriority() const {
            return Priority;
        }
        TDuration GetHistoryDepth() const {
            return HistoryDepth;
        }
        bool HasLocalization() const {
            return ButtonLocalization || MessageLocalization;
        }
        bool ShouldShow() const {
            return Show;
        }

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        void SetBonusInfo(const TString& tag, ui32 amount);
        void SetBonusLocalization(const TString& button, const TString& message, const TString& submessage = {});
        void SetLocalization(const TString& button, const TString& message, const TString& submessage = {});

    protected:
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

    private:
        TString ButtonLocalization;
        TString MessageLocalization;
        TString SubmessageLocalization;

        TString BonusButtonLocalization;
        TString BonusMessageLocalization;
        TString BonusSubmessageLocalization;
        TSet<TString> BonusDisableUserTags = { "user_character_feedback_ban" };
        TSet<TString> BonusRequiredUserTags;
        TString BonusTag;
        ui32 BonusAmount;

        ui32 MaxSelfCount = 1;
        ui32 MaxTotalCount = 2;
        ui32 Priority = 0;
        bool Show = true;
        TDuration HistoryDepth = TDuration::Days(7);
    };

    struct TResponse {
        TString BonusTag;
        TString ButtonLocalization;
        TString MessageLocalization;
        TString SubmessageLocalization;
        ui32 Priority = 0;
        ui32 Bonuses = 0;
        bool Show = true;

        explicit operator bool() const {
            return ButtonLocalization || MessageLocalization;
        }
        inline bool operator<(const TResponse& other) const {
            return std::tie(Bonuses, Priority) < std::tie(other.Bonuses, other.Priority);
        }
    };

private:
    using TBase = INativeSerializationTag<NDrive::NProto::TFeedbackTraceTagData>;

public:
    using TBase::TBase;

    const TString& GetBonusTag() const {
        return BonusTag;
    }
    const TString& GetDeviceId() const {
        return DeviceId;
    }
    const TString& GetUserId() const {
        return UserId;
    }
    TInstant GetTimestamp() const {
        return Timestamp;
    }
    ui32 GetEffectiveBonusAmount() const {
        return BonusAmount;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Trace };
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    TResponse GetResponse(const NDrive::IServer* server) const;
    TResponse GetResponse(const TDescription& description) const;
    static TResponse MergeResponses(const TResponse* base, const TResponse& delta);
    static TResponse MergeResponses(const TResponse& base, const TResponse& delta);
    static NJson::TJsonValue FormatResponse(const TResponse& response, ELocalization locale, const NDrive::IServer* server);

    bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

    bool IsFeedbackEnabled(const NDrive::IServer* server, NDrive::TEntitySession& session) const;
    bool IsFeedbackEnabled(const NDrive::IServer* server, const TDescription& description, NDrive::TEntitySession& session) const;
    bool Set(const TString& deviceId, const TString& userId, TInstant timestamp, const NDrive::IServer* server, NDrive::TEntitySession& session);
    void CancelBonus();

    static TString Type() {
        return "feedback_trace_tag";
    }

protected:
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

private:
    TMaybe<std::pair<TString, ui32>> CalculateBonus(TInstant timestamp, const NDrive::IServer* server, NDrive::TEntitySession& session) const;

private:
    TString DeviceId;
    TString UserId;
    TInstant Timestamp;

    TString BonusTag;
    ui32 BonusAmount = 0;
};
