#include "fueling.h"

#include "chargable.h"
#include "device_tags.h"
#include "telematics.h"

#include <drive/backend/database/drive_api.h>

#include <drive/telematics/api/sensor/interface.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <util/generic/serialized_enum.h>

class TSimpleFuelingTag: public ISerializableTag<NDrive::NProto::TFuelingTag> {
private:
    using TProto = NDrive::NProto::TFuelingTag;
    using TBase = ISerializableTag<TProto>;

public:
    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;
    public:
        TDuration GetWarmingDuration() const {
            return WarmingDuration;
        }

        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
            NDrive::TScheme result = TTagDescription::GetScheme(server);
            result.Add<TFSDuration>("warming_duration", "Duration of WARMING command");
            return result;
        }

    protected:
        NJson::TJsonValue DoSerializeMetaToJson() const override {
            NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
            result["warming_duration"] = NJson::ToJson(WarmingDuration);
            return result;
        }

        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override {
            if (!TBase::DoDeserializeMetaFromJson(jsonMeta)) {
                return false;
            }
            SetEnableSessions(true);
            return NJson::ParseField(jsonMeta["warming_duration"], WarmingDuration);
        }

    private:
        TDuration WarmingDuration = TDuration::Minutes(1);
    };

public:
    bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override {
        CHECK_WITH_LOG(server);
        const TDriveAPI* drive = server->GetDriveAPI();
        CHECK_WITH_LOG(drive);

        auto d = drive->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
        auto description = d ? dynamic_cast<const TDescription*>(d.Get()) : Singleton<TDescription>();

        auto expectedPerformedTagIds = TChargableTag::GetActiveSessionTagIds(self.GetObjectId(), drive->GetTagsManager().GetTagsMeta(), Max<ui64>(), session);
        if (!expectedPerformedTagIds) {
            return false;
        }
        if (!expectedPerformedTagIds->IsEmpty()) {
            return true;
        }

        const NDrive::ISensorApi* sensors = server->GetSensorApi();
        const NDrive::TPusherPtr pusher = sensors ? sensors->GetPusher() : nullptr;
        const TString& imei = drive->GetIMEI(self.GetObjectId());
        const auto now = Now();
        if (pusher && imei) {
            NDrive::TSensor sensor(NDrive::DerivativeFuelSensor);
            sensor.Value = ui64(100);
            sensor.Since = now;
            sensor.Timestamp = now;
            pusher->Push(imei, sensor);
        }
        if (!imei) {
            return true;
        }

        TUserPermissionsFeatures upf;
        TUserPermissions::TPtr permissions = server->GetDriveAPI()->GetUserPermissions(userId, upf);
        if (!permissions) {
            session.SetErrorInfo("fueling_remove", "cannot build permissions for " + userId, EDriveSessionResult::InternalError);
            return false;
        }
        NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::YADRIVE_WARMING);
        NDrive::NVega::TCommandRequest::TWarming warming;
        warming.Time = description->GetWarmingDuration().Minutes();
        command.Argument.Set(warming);
        auto response = TTelematicsCommandTag::Command(self.GetObjectId(), command, TDuration::Minutes(1), *permissions, server, session);
        if (!response.Initialized()) {
            return false;
        }
        return true;
    }

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSNumeric>("tank_filled", "Залито топлива");
        return result;
    }

    TProto DoSerializeSpecialDataToProto() const override {
        TProto proto = TBase::DoSerializeSpecialDataToProto();
        proto.MutableFuelingTagData()->SetTankFilled(TankFilled);
        return proto;
    }

    bool DoDeserializeSpecialDataFromProto(const TProto& proto) override {
        TankFilled = proto.GetFuelingTagData().GetTankFilled();
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        JWRITE_DEF(json, "tank_filled", TankFilled, 0);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        JREAD_UINT_OPT(json, "tank_filled", TankFilled);
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

private:
    ui32 TankFilled = 0;
};

class TServiceFuelingTag: public TSimpleFuelingTag {
public:
    TTagDescription::TPtr GetMetaDescription(const TString& type) const override {
        TTagDescription::TPtr result = TSimpleFuelingTag::GetMetaDescription(type);
        if (result) {
            result->SetDefaultPriority(1000);
        }
        return result;
    }
};

ITag::TFactory::TRegistrator<TSimpleFuelingTag> SimpleFuelingTagRegistrator("simple_fueling_tag");
ITag::TFactory::TRegistrator<TServiceFuelingTag> ServiceFuelingTagRegistrator("service_fueling_tag");

TString TUserFuelingTag::TypeName = "user_fueling_tag";

TUserFuelingTag::TUserFuelingTag(const TString& stationId, const TString& columnId)
    : TBase(TypeName)
    , StationId(stationId)
    , ColumnId(columnId)
{
}

TSet<NEntityTagsManager::EEntityType> TUserFuelingTag::GetObjectType() const {
    return {NEntityTagsManager::EEntityType::Car, NEntityTagsManager::EEntityType::User};
}

TUserFuelingTag::TProto TUserFuelingTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetPreOrderId(PreOrderId);
    proto.SetOrderId(OrderId);
    proto.SetObjectId(ObjectId);
    proto.SetSessionId(SessionId);
    proto.SetCurrentState((ui32)CurrentState);
    proto.SetLiters(Liters);
    proto.SetStationId(StationId);
    proto.SetColumnId(ColumnId);
    if (HasFuelType()) {
        proto.SetFuelType((ui32)*FuelType);
    }
    if (HasActualFuelType()) {
        proto.SetActualFuelType((ui32)*ActualFuelType);
    }
    if (HasActualLiters()) {
        proto.SetActualLiters(*ActualLiters);
    }
    if (HasFuelClientType()) {
        proto.SetFuelClientType((ui32)*FuelClientType);
    }
    return proto;
}

bool TUserFuelingTag::DoDeserializeSpecialDataFromProto(const TUserFuelingTag::TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (!proto.HasCurrentState()) {
        CurrentState = EFuelingStatus::Unknown;
    } else {
        auto state = static_cast<EFuelingStatus>(proto.GetCurrentState());
        if (GetEnumNames<EFuelingStatus>().contains(state)) {
            CurrentState = state;
        } else {
            CurrentState = EFuelingStatus::Unknown;
        }
    }
    Liters = proto.GetLiters();
    if (!proto.HasObjectId()) {
        return false;
    }
    SessionId = proto.GetSessionId();
    ObjectId = proto.GetObjectId();
    StationId = proto.GetStationId();
    ColumnId = proto.GetColumnId();

    auto castFuelType = [](const auto protoType) {
        auto state = static_cast<EFuelType>(protoType);
        if (GetEnumNames<EFuelType>().contains(state)) {
            return state;
        }
        return EFuelType::Undefined;
    };
    FuelType = proto.HasFuelType()
        ? castFuelType(proto.GetFuelType())
        : EFuelType::Undefined;
    if (proto.HasActualFuelType()) {
        ActualFuelType = castFuelType(proto.GetActualFuelType());
    }

    if (proto.HasActualLiters()) {
        ActualLiters = proto.GetActualLiters();
    }
    if (proto.HasFuelClientType()) {
        FuelClientType = EFuelClientType(proto.GetFuelClientType());
    }
    OrderId = proto.GetOrderId();
    if (proto.HasPreOrderId()) {
        PreOrderId = proto.GetPreOrderId();
    } else {
        PreOrderId = OrderId;
    }
    return true;
}

TString TUserFuelingTag::GetTypeName() {
    return TypeName;
}

ui64 TUserFuelingTag::GetFuelingPercentLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server) {
    TMaybe<ui64> result;
    if (!result) {
        result = TUserPermissions::GetSetting<ui64>("fueling." + modelId + ".perc_limit", server.GetSettings(), permissions);
    }
    if (!result) {
        result = TUserPermissions::GetSetting<ui64>("fueling.perc_limit", server.GetSettings(), permissions);
    }
    return result.GetOrElse(90);
}

double TUserFuelingTag::GetFuelingVolumeLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server) {
    TMaybe<double> result;
    if (!result) {
        result = TUserPermissions::GetSetting<double>("fueling." + modelId + ".liters_limit", server.GetSettings(), permissions);
    }
    if (!result) {
        result = TUserPermissions::GetSetting<double>("fueling.liters_limit", server.GetSettings(), permissions);
    }
    return result.GetOrElse(1);
}

ui64 TUserFuelingTag::GetTankerFuelingPercentLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server) {
    auto result = TUserPermissions::GetSetting<ui64>("fueling.tanker." + modelId + ".perc_limit", server.GetSettings(), permissions);
    if (!result) {
        result = TUserPermissions::GetSetting<ui64>("fueling.tanker.perc_limit", server.GetSettings(), permissions);
    }
    return result.GetOrElse(90);
}

double TUserFuelingTag::GetTankerFuelingVolumeLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server) {
    auto result = TUserPermissions::GetSetting<double>("fueling.tanker." + modelId + ".liters_limit", server.GetSettings(), permissions);
    if (!result) {
        result = TUserPermissions::GetSetting<double>("fueling.tanker.liters_limit", server.GetSettings(), permissions);
    }
    return result.GetOrElse(1);
}

bool TUserFuelingTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto copy = self.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
    if (!copy) {
        return false;
    } else {
        auto optionalAddedTags = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(copy.GetData(), userId, ObjectId, server, session);
        return optionalAddedTags.Defined();
    }
}

bool TUserFuelingTag::OnAfterRemove(const TDBTag& /*self*/, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    TVector<TDBTag> tags;
    if (server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreEntityTags(ObjectId, {TypeName}, tags, session)) {
        for (auto&& i : tags) {
            TUserFuelingTag* tag = i.MutableTagAs<TUserFuelingTag>();
            if (tag) {
                tag->SetCurrentState(GetCurrentState()).SetComment(GetComment());
            }
        }
        return
            server->GetDriveAPI()->GetTagsManager().GetDeviceTags().UpdateTagsData(tags, userId, session) &&
            server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTags(tags, userId, server, session, true);
    } else {
        return false;
    }
}

NDrive::TScheme TUserFuelingTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("preorder_id", "Предварительный идентификатор заказа");
    result.Add<TFSString>("order_id", "Идентификатор заказа");
    result.Add<TFSString>("station_id", "Идентификатор станции");
    result.Add<TFSString>("object_id", "Идентификатор объекта");
    result.Add<TFSString>("current_state", "Текущее состояние");
    result.Add<TFSString>("fuel_type", "Запрошенный тип топлива");
    result.Add<TFSNumeric>("liters", "Количество запрошенных литров").SetPrecision(2);
    return result;
}

void TUserFuelingTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json.InsertValue("preorder_id", PreOrderId);
    json.InsertValue("order_id", OrderId);
    json.InsertValue("station_id", StationId);
    json.InsertValue("session_id", SessionId);
    json.InsertValue("column_id", ColumnId);
    json.InsertValue("object_id", ObjectId);
    json.InsertValue("current_state", ::ToString(CurrentState));
    json.InsertValue("liters", Liters);
    json.InsertValue("fuel_type", ::ToString(FuelType.GetOrElse(EFuelType::Undefined)));
    if (ActualFuelType) {
        NJson::InsertNonNull(json, "actual_fuel_type", ::ToString(ActualFuelType));
    }
    NJson::InsertNonNull(json, "actual_liters", ActualLiters);
    json.InsertValue("client_type", ::ToString(FuelClientType.GetOrElse(EFuelClientType::Default)));
}

NJson::TJsonValue TUserFuelingTag::GetPublicReport(const ILocalization* localization, const TFuelingManager* manager, const TSet<EFuelingStatus>& allowedPostPayRetry /* = {} */) const {
    NJson::TJsonValue result;
    TStation station;
    if (manager && HasFuelType()) {
        const TSet<EFuelType> fType = {GetFuelTypeUnsafe()};
        if (StationId && manager->GetStationInfo(StationId, station)) {
            result.InsertValue("station", station.GetJsonMapReport(&fType));
        }
        if (OrderId.empty()) {
            result.InsertValue("fueling_message", manager->GetFuelingTypeMessage(GetFuelTypeUnsafe(), Liters));
        } else {
            result.InsertValue("fueling_message", manager->GetFuelingTypeMessage(GetFuelTypeUnsafe()));
        }
    }
    result.InsertValue("column", ColumnId);
    result.InsertValue("status", ::ToString(CurrentState));
    result.InsertValue("session_id", SessionId);
    result.InsertValue("liters", Liters);
    result.InsertValue("status_info", GetFuelingStatusInfo(localization, CurrentState));
    TString commentPhrase = GetFuelingStatusDescription(localization, CurrentState, GetComment(), OrderId.empty());
    if (!commentPhrase.empty()) {
        result.InsertValue("description", commentPhrase);
    }
    if (manager && HasActualFuelType()) {
        result.InsertValue("actual_fuel", manager->GetFuelingTypeMessage(GetActualFuelTypeUnsafe()));
    }
    if (HasActualLiters()) {
        result.InsertValue("actual_liters", *ActualLiters);
    }
    result.InsertValue("client_type", ::ToString(FuelClientType.GetOrElse(EFuelClientType::Default)));
    result.InsertValue("can_retry", HasActualFuelType() && HasActualLiters() && allowedPostPayRetry.contains(CurrentState));
    return result;
}

bool TUserFuelingTag::DoSpecialDataFromJson(const NJson::TJsonValue& /*json*/, TMessagesCollector* /*errors*/) {
    return false;
}

EUniquePolicy TUserFuelingTag::GetUniquePolicy() const {
    return EUniquePolicy::Rewrite;
}


ITag::TFactory::TRegistrator<TUserFuelingTag> TUserFuelingTag::Registrator(TUserFuelingTag::TypeName);

TTagDescription::TFactory::TRegistrator<TSimpleFuelingTag::TDescription> SimpleFuelingTagDescription("simple_fueling_tag");
TTagDescription::TFactory::TRegistrator<TSimpleFuelingTag::TDescription> ServiceFuelingTagDescription("service_fueling_tag");


TString TPatchFuelingTag::GetTypeName() {
    return "patch_fueling_tag";
}

EUniquePolicy TPatchFuelingTag::GetUniquePolicy() const {
    return EUniquePolicy::NoUnique;
}

TSet<NEntityTagsManager::EEntityType> TPatchFuelingTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::Car };
}

ITag::TFactory::TRegistrator<TPatchFuelingTag> TPatchFuelingTag::Registrator;


NDrive::TScheme TPatchFuelingTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSArray>("fuel_types", "Типы топлива").SetElement<TFSVariants>().InitVariants<EFuelType>();
    scheme.Add<TFSNumeric>("sensor_id", "Идентификатор сенсора (если 0, то используется основной сенсор бака)").SetDefault(0);
    scheme.Add<TFSBoolean>("overrive_model", "Не использовать данные модели").SetDefault(OverrideModel);
    return scheme;
}

NJson::TJsonValue TPatchFuelingTag::TDescription::DoSerializeMetaToJson() const {
    auto json = TBase::DoSerializeMetaToJson();
    NJson::InsertField(json, "fuel_types", PatchTypes);
    NJson::InsertField(json, "sensor_id", Sensor);
    NJson::InsertField(json, "overrive_model", OverrideModel);
    return json;
}

bool TPatchFuelingTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    return TBase::DoDeserializeMetaFromJson(json)
        && NJson::ParseField(json, "fuel_types", PatchTypes, /* required = */ true)
        && NJson::ParseField(json, "sensor_id", Sensor, /* required = */ true)
        && NJson::ParseField(json, "overrive_model", OverrideModel, /* required = */ true);
}

TPatchFuelingTag::TDescription::TFactory::TRegistrator<TPatchFuelingTag::TDescription> TPatchFuelingTag::TDescription::Registrator(TPatchFuelingTag::GetTypeName());
