#pragma once

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/proto/tags.pb.h>
#include <drive/backend/proto/tags.pb.h>

#include <drive/backend/common/localization.h>
#include <drive/backend/fueling_manager/fueling_manager.h>
#include <drive/backend/roles/permissions.h>

namespace NDrive {
    const TSensorId FuelSensor = { CAN_FUEL_LEVEL_P };
    const TSensorId DerivativeFuelSensor = { CAN_FUEL_LEVEL_P, CAN_ENGINE_IS_ON };
}

class TUserFuelingTag: public ISerializableTag<NDrive::NProto::TUserFuelingTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TUserFuelingTag>;

private:
    R_FIELD(TString, StationId);
    R_FIELD(TString, ColumnId);
    R_OPTIONAL(EFuelType, FuelType);
    R_FIELD(TString, PreOrderId);
    R_FIELD(TString, OrderId);
    R_FIELD(TString, ObjectId);
    R_FIELD(TString, SessionId);
    R_FIELD(EFuelingStatus, CurrentState, EFuelingStatus::Unknown);
    R_FIELD(double, Liters, 0);
    R_OPTIONAL(EFuelType, ActualFuelType);
    R_OPTIONAL(double, ActualLiters);
    R_OPTIONAL(EFuelClientType, FuelClientType);

private:
    static TFactory::TRegistrator<TUserFuelingTag> Registrator;

protected:
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

protected:
    static TString TypeName;

public:
    static TString GetTypeName();
    static ui64 GetFuelingPercentLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server);
    static double GetFuelingVolumeLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server);
    static ui64 GetTankerFuelingPercentLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server);
    static double GetTankerFuelingVolumeLimit(const TString& modelId, TUserPermissions::TPtr permissions, const NDrive::IServer& server);

public:
    using TBase::TBase;
    TUserFuelingTag() = default;
    TUserFuelingTag(const TString& stationId, const TString& columnId);

    virtual bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    NJson::TJsonValue GetPublicReport(const ILocalization* localization, const TFuelingManager* manager, const TSet<EFuelingStatus>& allowedPostPayRetry = {}) const;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& /*json*/, TMessagesCollector* /*errors*/) override;
    virtual EUniquePolicy GetUniquePolicy() const override;
};

class TPatchFuelingTag: public ISerializableTag<NDrive::NProto::TCommonTagData> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TCommonTagData>;

public:
    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

    private:
        R_FIELD(TVector<EFuelType>, PatchTypes);
        R_FIELD(ui16, Sensor, 0);
        R_FIELD(bool, OverrideModel, false);

    public:
        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;

    private:
        static TFactory::TRegistrator<TDescription> Registrator;
    };

private:
    static TFactory::TRegistrator<TPatchFuelingTag> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName();

    virtual EUniquePolicy GetUniquePolicy() const override;
    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;
};
