#include "acl.h"

#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/drivematics/signals/signal_configurations.h>
#include <drive/backend/drivematics/zone/zone.h>
#include <drive/backend/drivematics/zone/config/zone_config.h>
#include <drive/backend/database/config.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/notifications/manager.h>
#include <drive/backend/notifications/telegram/telegram.h>
#include <drive/backend/roles/manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>

const TString NDrivematics::TACLTag::TypeName = "acl_tag";
ITag::TFactory::TRegistrator<NDrivematics::TACLTag> NDrivematics::TACLTag::Registrator(NDrivematics::TACLTag::TypeName);
TTagDescription::TFactory::TRegistrator<NDrivematics::TACLTag::TDescription> NDrivematics::TACLTag::TDescription::Registrator(NDrivematics::TACLTag::TypeName);

namespace {
    inline bool IsAllIntersection(const TSet<TString>& ids, const TSet<TString>& idsInTag) {
        auto first1 = ids.begin();
        auto last1 = ids.end();
        auto first2 = idsInTag.begin();
        auto last2 = idsInTag.end();
        size_t countIdsFound = 0;
        while (first1 != last1 && first2 != last2) {
            if (*first1 < *first2) {
                ++first1;
            } else if (*first2 < *first1) {
                ++first2;
            } else {
                ++countIdsFound;
                ++first1; ++first2;
            }
        }
        return countIdsFound == ids.size();
    }

    void SetSchemeFromSettings(NDrive::TScheme& scheme, const NDrive::IServer& server) {
        NJson::TJsonValue aclObjects = server.GetSettings().GetJsonValue("leasing.acl.structure");
        if (!aclObjects.IsArray()) {
            return;
        }
        TAdministrativeAction::EEntity entity;
        for (auto&& entitySetting : aclObjects.GetArray()) {
            if (TryFromString(entitySetting.GetString(), entity)) {
                switch (entity) {
                    case TAdministrativeAction::EEntity::Zone:
                    {
                        if (server.GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetConfig().GetZoneConfig()->GetEnable()) {
                            NDrive::TFSVariants::TCompoundVariants compoundVariant;
                            auto action = [&compoundVariant](const TZone& zone) -> void {
                                compoundVariant.emplace(
                                    NDrive::TFSVariants::TCompoundVariant{zone.GetInternalId(), zone.GetName(), zone.GetOwnerDef("")}
                                );
                            };
                            Y_UNUSED(server.GetDriveAPI()->GetZoneDB()->ForObjectsList(action));
                            scheme.Add<TFSVariable>(entitySetting.GetString(), "Зоны").MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
                        } else {
                            scheme.Add<TFSArray>(ToString(entity)).SetElement<TFSString>();
                        }
                        break;
                    }
                    case TAdministrativeAction::EEntity::NamedFilter:
                    {
                        NDrive::TFSVariants::TCompoundVariants compoundVariant;
                        auto action = [&compoundVariant](const TNamedFilter& namedFilter) -> void {
                            compoundVariant.emplace(
                                NDrive::TFSVariants::TCompoundVariant{namedFilter.GetInternalId(), namedFilter.GetDisplayName(), namedFilter.GetFilter().ToString()}
                            );
                        };
                        Y_UNUSED(Yensured(server.GetDriveAPI()->GetNamedFiltersDB())->ForObjectsList(action, TInstant::Now()));
                        scheme.Add<TFSVariable>(entitySetting.GetString(), "Динамические группы").MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
                        break;
                    }
                    case TAdministrativeAction::EEntity::SignalConfiguration:
                    {
                        NDrive::TFSVariants::TCompoundVariants compoundVariant;

                        auto tx = server.GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
                        auto signalsConf = Yensured(server.GetDriveAPI()->GetSignalsConfigurationsDB())->GetObjects(tx);
                        if (signalsConf) {
                            ForEach(signalsConf->begin(), signalsConf->end(), [&compoundVariant](const TSignalConfigurationDB& signalConf) -> void {
                                compoundVariant.emplace(
                                    NDrive::TFSVariants::TCompoundVariant{signalConf.GetInternalId(), signalConf->GetDisplayName(), signalConf->GetOwnerName()}
                                );
                            });
                        }

                        scheme.Add<TFSVariable>(entitySetting.GetString(), "Сигналы").MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
                        break;
                    }
                    case TAdministrativeAction::EEntity::Notifier:
                    {
                        const auto notifiersManager = server.GetNotifiersManager();
                        const auto notifiersManagerImpl = static_cast<const TNotifiersManager*>(notifiersManager);

                        NDrive::TFSVariants::TCompoundVariants compoundVariant;
                        auto action = [&] (const TNotifierContainer& object) -> void {
                                TUserOrganizationAffiliationTag::ESupportNotifierType type;
                            if (TryFromString(object.GetNotifierConfigPtr()->GetTypeName(), type) && type != TUserOrganizationAffiliationTag::ESupportNotifierType::Unsupport) {
                                compoundVariant.emplace(
                                    NDrive::TFSVariants::TCompoundVariant{object.GetInternalId(), object.GetDisplayName()}
                                );
                            }
                        };
                        Y_UNUSED(notifiersManagerImpl->ForObjectsList(action, Now()));
                        scheme.Add<TFSVariable>(entitySetting.GetString(), "Нотификаторы").MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
                        break;
                    }
                    case TAdministrativeAction::EEntity::Models:
                    {
                        NDrive::TFSVariants::TCompoundVariants compoundVariant;
                        auto models = server.GetDriveAPI()->GetModelsData()->GetCached().GetResult();
                        ForEach(models.begin(), models.end(), [&compoundVariant](const std::pair<TString, TDriveModelData>& modelPair) -> void {
                            compoundVariant.emplace(
                                NDrive::TFSVariants::TCompoundVariant{modelPair.first, modelPair.second.GetName()}
                            );
                        });
                        scheme.Add<TFSVariable>(entitySetting.GetString(), "Модели объектов").MutableCondition().SetMultiSelect(true).SetCompoundVariants(compoundVariant);
                        break;
                    }
                    default:
                        scheme.Add<TFSArray>(ToString(entity)).SetElement<TFSString>();
                        break;
                }
            }
        }
    }
}

namespace NDrivematics {
    TACLSaver::TACLSaver(class TDBTag&& aclTag):
        TagImpl(std::dynamic_pointer_cast<TACLTag>(aclTag.GetData())),
        TdbTag(std::move(aclTag))
    {}

    TACLTag* TACLSaver::operator->() {
        Y_ENSURE_BT(TagImpl);
        return TagImpl.Get();
    }

    TACLSaver::operator class TDBTag&() {
        return TdbTag;
    }

    NDrive::TScheme TACLTag::TDescription::GetScheme(const NDrive::IServer* server) const {
        NDrive::TScheme result;
        NDrive::TScheme& objects = result.Add<TFSStructure>("acl", "Объекты компании").SetStructure<NDrive::TScheme>();
        SetSchemeFromSettings(objects, *server);
        return result;
    }

    NJson::TJsonValue TACLTag::TDescription::DoSerializeMetaToJson() const {
        NJson::TJsonValue jsonMeta = NJson::JSON_MAP;
        for (const auto& el : CompanyObjects) {
            jsonMeta["acl"].InsertValue(ToString(el.first), NJson::ToJson(el.second));
        }
        return jsonMeta;
    }

    bool TACLTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
        if (jsonMeta.Has("acl") && jsonMeta["acl"].IsMap()) {
            TAdministrativeAction::EEntity entity;
            for (const auto& element : jsonMeta["acl"].GetMap()) {
                if (!NJson::ParseField(element.first, NJson::Stringify(entity), true)
                    || !NJson::ParseField(element.second, CompanyObjects[entity], true)) {
                    return false;
                }
            }
        }
        return true;
    }

    bool TACLTag::TDescription::CheckPermission(const TAdministrativeAction::EEntity& entity, const TAdministrativeAction::EAction& action, TUserPermissions::TPtr permissions) const {
        return permissions->CheckAdministrativeActions(action, entity);
    }

    const TACLTag::TObjectsIds& TACLTag::TDescription::GetEntityObjects(const TAdministrativeAction::EEntity& entity, TUserPermissions::TPtr permissions, bool force) const {
        R_ENSURE(force || CheckPermission(entity, TAdministrativeAction::EAction::Observe, permissions), HTTP_FORBIDDEN, "no permission " << ToString(entity) << ":" << TAdministrativeAction::EAction::Observe, NDrive::MakeError("acl.permission"));
        auto p = CompanyObjects.find(entity);
        if (p != CompanyObjects.end()) {
            return p->second;
        } else {
            return Default<TObjectsIds>();
        }
    }

    [[nodiscard]] bool TACLTag::TDescription::AddEntityObject(const TAdministrativeAction::EEntity& entity, const TString& objectId, TUserPermissions::TPtr permissions, TMessagesCollector& errors) {
        R_ENSURE(CheckPermission(entity, TAdministrativeAction::EAction::Add, permissions), HTTP_FORBIDDEN, "no permission " << ToString(entity) << ":" << TAdministrativeAction::EAction::Add, NDrive::MakeError("acl.permission"));
        if (!CompanyObjects[entity].insert(objectId).second) {
            errors.AddMessage("::AddEntityObject", TStringBuilder() << "object already exist: " << objectId);
            return false;
        }
        return true;
    }

    [[nodiscard]] bool TACLTag::TDescription::RemoveEntityObjects(const TAdministrativeAction::EEntity& entity, const TACLTag::TObjectsIds& ids, TUserPermissions::TPtr permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, ERemoveEntityPolicy policy) {
        R_ENSURE(CheckPermission(entity, TAdministrativeAction::EAction::Remove, permissions), HTTP_FORBIDDEN, "no permission " << ToString(entity) << ":" << TAdministrativeAction::EAction::Remove, NDrive::MakeError("acl.permission"), tx);
        auto& entityCompanyObjects = CompanyObjects[entity];
        if (!IsAllIntersection(ids, entityCompanyObjects)) {
            tx.SetErrorInfo("::RemoveEntityObjects", TStringBuilder() << "not all object fround to delete");
            return false;
        }
        if (policy != ERemoveEntityPolicy::Reference) {
            auto optionalTaggedUser = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(permissions->GetUserId(), tx);
            if (!optionalTaggedUser) {
                return false;
            }
            if (!IChain::DoRemove<TString>(optionalTaggedUser, ids, entity, permissions->GetUserId(), *server, tx)) {
                return false;
            }
        }

        if (policy != ERemoveEntityPolicy::Object) {
            for (const auto& id : ids) {
                entityCompanyObjects.erase(id);
            }
        }
        return true;
    }

    void TACLTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
        TBase::SerializeSpecialDataToJson(json);
        for (const auto& el : UserObjects) {
            json["acl"].InsertValue(ToString(el.first), NJson::ToJson(el.second));
        }
    }

    bool TACLTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector*  errors) {
        if (json.Has("acl") && json["acl"].IsMap()) {
            TAdministrativeAction::EEntity entity;
            for (const auto& element : json["acl"].GetMap()) {
                if (!NJson::ParseField(element.first, NJson::Stringify(entity), true)
                    || !NJson::ParseField(element.second, UserObjects[entity], true)) {
                    return false;
                }
            }
        }
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    NDrive::TScheme TACLTag::GetScheme(const NDrive::IServer* server) const {
        NDrive::TScheme result = TBase::GetScheme(server);
        NDrive::TScheme& objects = result.Add<TFSStructure>("acl", "Объекты пользователя").SetStructure<NDrive::TScheme>();
        SetSchemeFromSettings(objects, *server);
        return result;
    }

    TString TACLTag::GetOrganizationAffiliationCompanyName(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        auto organizationAffiliationTagPtr = TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(userId, server, tx);

        return Yensured(organizationAffiliationTagPtr)->GetCompanyName();
    }

    TACLTag::TDescription::TPtrImpl TACLTag::GetOrCreateACLTagDescription(const TString& userId, const NDrive::IServer& server) {
        const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
        auto tx = server.GetDriveAPI()->template BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

        TString companyName = GetOrganizationAffiliationCompanyName(userId, server, tx);
        R_ENSURE(companyName, {}, "cannot get company name", NDrive::MakeError("acl.create"), tx);
        TString tagName = TypeName + "_" + ToString(GetUid(companyName));
        auto tagDescriptionsPtr = tagsMeta.GetDescriptionByName(tagName);

        if (!tagDescriptionsPtr) {
            auto tagDescription = MakeAtomicShared<TDescription>();
            tagDescription->SetName(tagName);
            tagDescription->SetType(TypeName);
            tagDescription->SetDisplayName("ACL tag for company: " + companyName);
            R_ENSURE(tagsMeta.RegisterTag(tagDescription, userId, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot register tag ", NDrive::MakeError("acl.create"), tx);
            NDrive::TEventLog::Log(
                "GetOrCreateACLTagDescription",
                NJson::TMapBuilder
                    ("event", "AutonomousTagCompanyCreate")
                    ("tag_name", tagName)
            );
            R_ENSURE(tx.Commit(), {}, "cannot commit", NDrive::MakeError("tx.save"), tx);

            return tagDescription;
        }
        return std::dynamic_pointer_cast<TDescription>(tagDescriptionsPtr->Clone());
    }

    TACLTag::TDescription::TPtrImpl TACLTag::GetCompanyTagDescription(const TString& userId, const NDrive::IServer& server) {
        auto tagDescriptionPtrImpl = GetOrCreateACLTagDescription(userId, server);
        R_ENSURE(tagDescriptionPtrImpl, HTTP_INTERNAL_SERVER_ERROR, "cannot get base Description", NDrive::MakeError("acl.restore"));
        return tagDescriptionPtrImpl;
    }

    TDBTag TACLTag::CreateACLTag(TACLTag::TDescription::TConstPtrImpl tagDescriptionPtrImpl, const TString& userId, const NDrive::IServer& server) {
        auto tx = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
        auto tagData = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagDescriptionPtrImpl->GetName(), "", Now());
        R_ENSURE(tagData, HTTP_INTERNAL_SERVER_ERROR, "cannot CreateTag", NDrive::MakeError("acl.create"), tx);
        auto optionalObjectContainer = server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tagData, userId, userId, &server, tx);
        R_ENSURE(optionalObjectContainer && optionalObjectContainer->size() && tx.Commit(), HTTP_INTERNAL_SERVER_ERROR, "cannot add tag", NDrive::MakeError("acl.add"), tx);
        return *optionalObjectContainer->begin();
    }

    TACLSaver TACLTag::GetACLTag(TUserPermissions::TPtr permissions, NDrive::TEntitySession& tx, const NDrive::IServer& server) {
        auto optionalObject = server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(permissions->GetUserId(), tx);
        R_ENSURE(optionalObject, HTTP_BAD_REQUEST, "user had not tags", NDrive::MakeError("acl.restore"), tx);
        auto aclTagContainer = optionalObject->GetTagsByClass<TACLTag>();
        R_ENSURE(aclTagContainer.size() <= 1, HTTP_INTERNAL_SERVER_ERROR, "inconsistent user ACLTag", NDrive::MakeError("acl.internal_error"), tx);

        auto tagDescriptionPtrImpl = GetCompanyTagDescription(permissions->GetUserId(), server);

        if (aclTagContainer.size() == 0) {
            return CreateACLTag(tagDescriptionPtrImpl, permissions->GetUserId(), server);
        }
        return std::move(*aclTagContainer.begin());
    }

    bool TACLTag::CheckPermission(const TAdministrativeAction::EEntity&  /*entity*/, const TAdministrativeAction::EAction&  /*action*/, TUserPermissions::TPtr  /*permissions*/) const {
        return true;
    }

    [[nodiscard]] const TACLTag::TObjectsIds& TACLTag::GetEntityObjects(const TAdministrativeAction::EEntity& entity, TUserPermissions::TPtr permissions, bool force) const {
        R_ENSURE(force || CheckPermission(entity, TAdministrativeAction::EAction::Observe, permissions), HTTP_FORBIDDEN, "no permission " << ToString(entity) << ":" << TAdministrativeAction::EAction::Observe, NDrive::MakeError("acl.permission"));
        auto p = UserObjects.find(entity);
        if (p != UserObjects.end()) {
            return p->second;
        } else {
            return Default<TObjectsIds>();
        }
    }

    [[nodiscard]] bool TACLTag::AddEntityObject(const TAdministrativeAction::EEntity& entity, const TString& objectId, TMessagesCollector& errors, TUserPermissions::TPtr permissions) {
        R_ENSURE(CheckPermission(entity, TAdministrativeAction::EAction::Add, permissions), HTTP_FORBIDDEN, "no permission " << ToString(entity) << ":" << TAdministrativeAction::EAction::Add, NDrive::MakeError("acl.permission"));
        if (!UserObjects[entity].insert(objectId).second) {
            errors.AddMessage("::AddEntityObject", TStringBuilder() << "object already exist: " << objectId);
            return false;
        }
        return true;
    }

    [[nodiscard]] bool TACLTag::RemoveEntityObjects(const TAdministrativeAction::EEntity& entity, const TACLTag::TObjectsIds& ids, TUserPermissions::TPtr permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, bool force) {
        R_ENSURE(CheckPermission(entity, TAdministrativeAction::EAction::Remove, permissions), HTTP_FORBIDDEN, "no permission " << ToString(entity) << ":" << TAdministrativeAction::EAction::Remove, NDrive::MakeError("acl.permission"), tx);
        auto& entityUserObjects = UserObjects[entity];
        if (!IsAllIntersection(ids, entityUserObjects)) {
            tx.SetErrorInfo("::RemoveEntityObjects", TStringBuilder() << "not all object found to delete");
            return false;
        }
        if (!force) {
            auto optionalTaggedUser = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(permissions->GetUserId(), tx);
            if (!optionalTaggedUser) {
                return false;
            }
            if (!IChain::DoRemove<TString>(optionalTaggedUser, ids, entity, permissions->GetUserId(), *server, tx)) {
                return false;
            }
        }
        for (const auto& id : ids) {
            entityUserObjects.erase(id);
        }
        return true;
    }

    bool TACLTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
        auto optionalTaggedUser = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(self.GetObjectId(), tx);
        if (!optionalTaggedUser) {
            return false;
        }
        for (const auto& [entity, ids] : UserObjects) {
            if (!ids.empty() && !IChain::DoRemove<TString>(optionalTaggedUser, ids, entity, userId, *server, tx)) {
                return false;
            }
        }
        return true;
    }

    template<class TObjectid>
    bool IChain::DoRemove(TMaybe<TTaggedObject> taggedUser, const TSet<TObjectid>& ids, TAdministrativeAction::EEntity entity, const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx){
        const auto* api = server.GetDriveAPI();
        switch (entity) {
            case TAdministrativeAction::EEntity::Zone:
                {
                    const auto& entitiesManager = api->GetZoneDB();
                    const auto& subEntitiesManager = api->GetAreasDB();

                    auto zoneToRemove = entitiesManager->GetObjectsByZoneIds(ids);
                    if (!zoneToRemove) {
                        break;
                    }
                    NDrive::TAreaIds areaIdsToRemove;
                    ForEach(zoneToRemove->begin(), zoneToRemove->end(),
                        [&areaIdsToRemove](const NDrivematics::TZone& zone) {
                            if (!zone.IsEmpty()) {
                                for (auto& areaId : zone.GetAreaIdsRef()) {
                                    areaIdsToRemove.insert(areaId);
                                }
                            }
                        }
                    );
                    if (areaIdsToRemove.size() != ids.size()) {
                        const auto companyUid = ToString(TACLTag::GetUid(TACLTag::GetOrganizationAffiliationCompanyName(taggedUser->GetId(), server, tx)));

                        TSet<TString> areaIdsToRemove;
                        const auto action = [&ids, &areaIdsToRemove, &companyUid](const TString& key, const TArea& entity) {
                            if (ids.contains(entity.GetZoneName())) {
                                const TVector<TString> parts = SplitString(entity.GetIdentifier(), "_");
                                if (parts.size() == 2 && companyUid == parts[0]) {
                                    areaIdsToRemove.insert(key);
                                }
                            }
                        };
                        if (!subEntitiesManager->ForObjectsMap(action)) {
                            break;
                        }
                    }

                    if (subEntitiesManager &&
                        subEntitiesManager->RemoveObject(areaIdsToRemove, userId, tx) &&
                        entitiesManager &&
                        entitiesManager->RemoveObjects(*zoneToRemove, userId, tx)) {
                        return true;
                    }
                    break;
                }
            case TAdministrativeAction::EEntity::NamedFilter:
                {
                    const auto& entitiesManager = api->GetNamedFiltersDB();
                    if (entitiesManager && entitiesManager->RemoveObject(ids, userId, tx)) {
                        return true;
                    }
                    break;
                }
            case TAdministrativeAction::EEntity::SignalConfiguration:
                {
                    const auto& entitiesManager = api->GetSignalsConfigurationsDB();
                    if (entitiesManager && entitiesManager->RemoveObjects(ids, userId, tx)) {
                        return true;
                    }
                    break;
                }
            case TAdministrativeAction::EEntity::Notifier:
                {
                    const auto notifiersManager = server.GetNotifiersManager();
                    if (notifiersManager && notifiersManager->RemoveObject(ids, userId, tx)) {
                        return true;
                    }
                    break;
                }
            case TAdministrativeAction::EEntity::Models:
                {
                    auto& modelsDb = api->GetModelsDB();
                    bool upsertFailed = false;

                    auto modelInfo = modelsDb.FetchInfo(ids, tx);
                    if (!modelInfo) {
                        break;
                    }

                    for (auto [key, model] : modelInfo.GetResult()) {
                        model.SetDeprecated(true);
                        if (!modelsDb.Upsert(model, tx)) {
                            upsertFailed = true;
                            break;
                        }
                    }

                    if (!upsertFailed) {
                        return true;
                    }
                    break;
                }
            default:
                break;
        }
        NDrive::TEventLog::Log("CannotProcessIChain::DoRemove", NJson::TMapBuilder
            ("userId", userId)
            ("entity", ToString(entity))
            ("tagsOwnerUserId", taggedUser->GetId())
            ("idsToManualRemove", NJson::ToJson(ids))
        );
        return false;
    }
}
