#include "company.h"

#include <drive/backend/data/leasing/acl/acl.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/roles/manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>

ITag::TFactory::TRegistrator<NDrivematics::TUserOrganizationAffiliationTag> NDrivematics::TUserOrganizationAffiliationTag::Registrator(NDrivematics::TUserOrganizationAffiliationTag::TypeName);
const TTagDescription::TFactory::TRegistrator<NDrivematics::TUserOrganizationAffiliationTag::TDescription> NDrivematics::TUserOrganizationAffiliationTag::TDescription::Registrator{NDrivematics::TUserOrganizationAffiliationTag::TypeName};

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrivematics::TUserOrganizationAffiliationTag::TNotifierIdsReference& companyNotifiers) {
    return companyNotifiers.DeserializeFromJson(value);
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::TUserOrganizationAffiliationTag::TNotifierIdsReference& companyNotifiers) {
    return companyNotifiers.SerializeToJson();
}

namespace {
    const TString DMRolePrefix = "wl_driver_role_";

    THolder<TUserAction> CreateObserveAction(const NDrivematics::TUserOrganizationAffiliationTag::TDescription& tagDescription) {
        auto action = MakeHolder<TTagAction>();
        action->SetDescription("Доступ на просмотр машин для водителя: " + tagDescription.GetCompanyName());
        action->AddTagAction(TTagAction::ETagAction::ObserveObject);
        action->SetTagName(tagDescription.GetOwningCarTagName());
        return action;
    }

    TSet<TString> CreateActions(const TString& authorId, const NDrivematics::TUserOrganizationAffiliationTag::TDescription& tagDescription, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        const auto& api = *Yensured(server.GetDriveAPI());
        TSet<TString> result;
        {
            const auto observeObjectCarsActionId = TString{"wl_observe_object_cars_" + ToString(tagDescription.GetUid())};
            result.insert(observeObjectCarsActionId);

            auto actionRestored = api.GetRolesManager()->GetAction(observeObjectCarsActionId, Now());
            if (!actionRestored) {
                auto action = CreateObserveAction(tagDescription);
                action->SetName(observeObjectCarsActionId);
                R_ENSURE(api.GetRolesManager()->GetActionsDB().ForceUpsert(std::move(action), authorId, tx), {}, "can't upsert action", tx);
            }
        }
        return result;
    }

    void LinkActions(const TString& authorId, const TString& roleId, const TSet<TString>& actionsIds, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        const auto& api = *Yensured(server.GetDriveAPI());
        for (const auto& actionId : actionsIds) {
            auto actionRoleHeader = TLinkedRoleActionHeader{};
            actionRoleHeader.SetSlaveObjectId(actionId).SetRoleId(roleId);
            R_ENSURE(api.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionRoleHeader, authorId, tx), {}, "cant link action with role", tx);
        }
    }
}

namespace NDrivematics {

    void TUserOrganizationAffiliationTag::TNotifierIdsReference::RefreshReference(const TChatId& chatId) {
        if (chatId && ChatIdToBotId.contains(chatId)) {
            const auto& botId = ChatIdToBotId[chatId];

            BotIdsToPairRef[botId].push_back(std::ref(*ChatIdToBotId.find(chatId)));
        }
    }
    void TUserOrganizationAffiliationTag::TNotifierIdsReference::RefreshReference(const TVector<TChatId>* chatIds) {
        if (chatIds) {
            for (const auto& chatId : *chatIds) {
                RefreshReference(chatId);
            }
        } else {
            BotIdsToPairRef.clear();
            for (auto& el : ChatIdToBotId) {
                BotIdsToPairRef[el.second].push_back(std::ref(el));
            }
        }
    }

    NJson::TJsonValue TUserOrganizationAffiliationTag::TNotifierIdsReference::SerializeToJson() const {
        NJson::TJsonValue result;
        for (const auto& [botId, pairRefs] : BotIdsToPairRef) {
            NJson::TJsonValue map;
            map["bot_id"] = botId;
            for (const auto& pair : pairRefs) {
                map["chat_ids"].AppendValue(pair.get().first);
            }
            result.AppendValue(map);
        }
        for (const auto& botId : OrphanBotId) {
            result.AppendValue(
                NJson::TMapBuilder
                    ("bot_id", botId)
                    ("chat_ids", NJson::JSON_ARRAY)
            );
        }
        return result;
    }

    bool TUserOrganizationAffiliationTag::TNotifierIdsReference::DeserializeFromJson(const NJson::TJsonValue& value) {
        if (!value.IsArray()) {
            return false;
        }
        for (const auto& obj : value.GetArray()) {
            if (!obj.IsMap()) {
                return false;
            }
            auto map = obj.GetMap();
            TSet<TChatId> chatIds;
            TBotId botId = map["bot_id"].GetString();
            if (!TJsonProcessor::ReadContainer(obj, "chat_ids", chatIds) || !botId) {
                return false;
            }
            if (chatIds.empty()) {
                OrphanBotId.insert(botId);
            }
            for (auto&& chatId : chatIds) {
                ChatIdToBotId.insert({chatId, botId});
            }
        }
        RefreshReference();
        return true;
    }

    bool TUserOrganizationAffiliationTag::TNotifierIdsReference::ChangeGlobalBotId(const TBotId& oldBotId, const TBotId& newBotId) {
        TVector<TChatId> updatedChatId;
        if (auto findIt = BotIdsToPairRef.find(oldBotId); findIt != BotIdsToPairRef.end()) {
            for (auto& pairRef : findIt->second) {
                pairRef.get().second = newBotId;
                updatedChatId.push_back(pairRef.get().first);
            }
            if (OrphanBotId.contains(newBotId)) {
                OrphanBotId.erase(newBotId);
            }
            BotIdsToPairRef.erase(oldBotId);
            RefreshReference(&updatedChatId);
            return true;
        } else if (OrphanBotId.contains(oldBotId)) {
            OrphanBotId.erase(oldBotId);
            OrphanBotId.insert(newBotId);
            return true;
        }
        return false;
    }

    bool TUserOrganizationAffiliationTag::TNotifierIdsReference::ChangeLocalBotId(const TChatId& chatId, const TBotId& newBotId) {
        if (auto findIt = ChatIdToBotId.find(chatId); findIt != ChatIdToBotId.end()){
            TBotId oldBotId = findIt->second;
            if (BotIdsToPairRef[oldBotId].size() == 1) {
                OrphanBotId.insert(oldBotId);
            }
            ChatIdToBotId[chatId] = newBotId;
            if (OrphanBotId.contains(newBotId)) {
                OrphanBotId.erase(newBotId);
            }
            RefreshReference();
            return true;
        }
        return false;
    }

    bool TUserOrganizationAffiliationTag::TNotifierIdsReference::Add(const TBotId& botId, const TChatId* chatId) {
        if (!chatId) {
            if (!BotIdsToPairRef.contains(botId)) {
                OrphanBotId.insert(botId);
            } else {
                return false;
            }
        } else {
            if (OrphanBotId.contains(botId)) {
                OrphanBotId.erase(botId);
            }
            ChatIdToBotId[*chatId] = botId;
            RefreshReference(*chatId);
        }
        return true;
    }

    TMaybe<TUserOrganizationAffiliationTag::TNotifierIdsReference::TBotId> TUserOrganizationAffiliationTag::TNotifierIdsReference::GetBotId(const TChatId& chatId) const {
        if (auto findIt = ChatIdToBotId.find(chatId); findIt != ChatIdToBotId.end()){
            return findIt->second;
        }
        return Nothing();
    }

    TSet<TUserOrganizationAffiliationTag::TNotifierIdsReference::TBotId> TUserOrganizationAffiliationTag::TNotifierIdsReference::GetBotIds() const {
        return MakeSet(NContainer::Keys(BotIdsToPairRef));
    }

    void TUserOrganizationAffiliationTag::TNotifierIdsReference::RemoveChatId(const TChatId& chatId) {
        if (ChatIdToBotId.contains(chatId)) {
            TBotId botId = ChatIdToBotId[chatId];
            auto& pairRef = BotIdsToPairRef[botId];
            for (auto it = pairRef.begin(); it != pairRef.end();) {
                if (it->get().first == chatId) {
                    it = pairRef.erase(it);
                    break;
                } else {
                    ++it;
                }
            }
            if (pairRef.size() == 0) {
                OrphanBotId.insert(botId);
                BotIdsToPairRef.erase(botId);
            }
            ChatIdToBotId.erase(chatId);
        }
    }


    NDrive::TScheme TUserOrganizationAffiliationTag::TDescription::GetScheme(const NDrive::IServer* server) const {
        NDrive::TScheme result;
        result.Add<TFSString>("company_name", "Имя компании").SetRequired(true);
        result.Add<TFSString>("owning_car_tag_name", "Тег владения машинами").SetRequired(true);
        result.Add<TFSString>("phone", "Номер телефона").SetRequired(false);
        result.Add<TFSString>("email", "Email").SetRequired(false);
        result.Add<TFSString>("booking_confirmed_sms_tag_name", "Тег смс информирования о подтверждении бронирования").SetRequired(false);
        result.Add<TFSString>("booking_cancelled_sms_tag_name", "Тег смс информирования об отмене бронирования").SetRequired(false);
        result.Add<TFSString>("booking_confirmed_email_tag_name", "Тег email информирования о подтверждении бронирования").SetRequired(false);
        result.Add<TFSString>("booking_cancelled_email_tag_name", "Тег email информирования об отмене бронирования").SetRequired(false);
        result.Add<TFSString>("after_finish_ride_email_tag_name", "Тег email информирования о завершении аренды").SetRequired(false);
        result.Add<TFSString>("after_acceptance_email_tag_name", "Тег email информирования о завершении приемки").SetRequired(false);
        result.Add<TFSString>("short_description", "Краткое описание компании").SetRequired(false);
        result.Add<TFSString>("icon_mobile_login", "Иконка для экрана входа в аккаунт в мобильном приложении").SetRequired(false);
        {
            NDrive::TScheme& renterOfferCommonInfo = result.Add<TFSArray>("company_information", "Параметры для смс/email", 100000).SetElement<NDrive::TScheme>();
            renterOfferCommonInfo.SetRequired(false);
            renterOfferCommonInfo.Add<TFSString>("key", "Ключ текста");
            renterOfferCommonInfo.Add<TFSString>("value", "Текст");
        }
        {
            auto rolesStorage = server->GetDriveAPI()->GetRolesManager()->GetRoles(TInstant::Zero());
            TSet<TString> rolesSet;
            for (auto&& role : rolesStorage) {
                rolesSet.emplace(role.GetName());
            }
            NDrive::TScheme& role  = result.Add<TFSArray>("company_roles", "Список ролей компании").SetElement<NDrive::TScheme>();
            role.Add<TFSVariants>("name", "Имя роли").SetMultiSelect(false).SetVariants({TUserOrganizationAffiliationTag::AdminRoleName, TUserOrganizationAffiliationTag::DriverRoleName}).SetEditable(true).SetRequired(true);
            role.Add<TFSVariants>("roles", "Список ролей").SetMultiSelect(true).SetVariants(rolesSet);
        }
        {
            NDrive::TScheme& telegram = result.Add<TFSArray>("telegram_notifiers", "Список telegram нотификаторов компании", 100001).SetElement<NDrive::TScheme>();
            telegram.SetRequired(false);
            telegram.Add<TFSString>("bot_id");
            telegram.Add<TFSArray>("chat_ids").SetElement<TFSString>();
        }
        return result;
    }

    ui64 TUserOrganizationAffiliationTag::TDescription::GetUid() const {
        return FnvHash<ui64>(GetCompanyName());
    }

    void TUserOrganizationAffiliationTag::AddRoleAndTagsToUser(
          const TDriveUserData& driverUser
        , const TVector<TString>& tagNames
        , const TMap<TString, TVector<TString>>& dmRoleToCommonActionsIds
        , TUserPermissions::TPtr permissions
        , const NDrive::IServer* server
        , NDrive::TEntitySession& tx
        , const TUserOrganizationAffiliationTag::TRoles& userRoles
    ) {
        const auto& api = *Yensured(server->GetDriveAPI());
        const auto& defaultRole = TUserOrganizationAffiliationTag::DriverRoleName;
        auto isRealtime = server->GetSettings().GetValue<bool>("company.add_role.is_realtime").GetOrElse(false);

        auto affiliatedCompanyTagDesc = TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(permissions, *server, tx);
        auto tagData = api.GetTagsManager().GetTagsMeta().CreateTag(Yensured(affiliatedCompanyTagDesc)->GetName());
        {
            auto tag = dynamic_cast<TUserOrganizationAffiliationTag*>(tagData.Get());
            R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag as user organization affiliation tag", tx);

            auto companyRoles = MakeSet(NContainer::Keys(affiliatedCompanyTagDesc->GetCompanyRoles()));
            TSet<TString> difference;
            SetDifference(userRoles.begin(), userRoles.end(), companyRoles.begin(), companyRoles.end(), std::inserter(difference, difference.begin()));
            R_ENSURE(difference.empty() || (difference.size() == 1 && difference.contains(defaultRole)), HTTP_INTERNAL_SERVER_ERROR, "unknown role: " + JoinSeq(',', difference), tx);
            tag->SetRoles(userRoles);
            tag->SetIsRealtime(isRealtime);
        }
        R_ENSURE(api.GetTagsManager().GetUserTags().AddTag(tagData, permissions->GetUserId(), driverUser.GetUserId(), server, tx), {}, "can't add tag", tx);

        for (auto&& tagName : tagNames) {
            auto tag = api.GetTagsManager().GetTagsMeta().CreateTag(tagName);
            R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "cannot create tag " << tagName, tx);
            R_ENSURE(
                api.GetTagsManager().GetUserTags().AddTag(tag, permissions->GetUserId(), driverUser.GetUserId(), server, tx),
                {},
                "cannot add tag " << tag->GetName(),
                tx
            );
        }

        if (isRealtime) {
            return;
        }

        const auto roleId = TString{DMRolePrefix + ToString(affiliatedCompanyTagDesc->GetUid())};
        TVector<TDriveRoleHeader> roles = api.GetRolesManager()->GetRoles({roleId}, Now());
        if (roles.empty()) {
            if (userRoles.contains(defaultRole)) {
                auto roleHeader = TDriveRoleHeader{roleId};
                roleHeader.SetDescription("Водитель: " + affiliatedCompanyTagDesc->GetCompanyName());
                R_ENSURE(api.GetRolesManager()->GetRolesDB().Upsert(roleHeader, permissions->GetUserId(), tx), {}, "can't upsert role", tx);
            }

            auto actionsIds = CreateActions(permissions->GetUserId(), *affiliatedCompanyTagDesc, *server, tx);
            if (auto dmRoleIt = dmRoleToCommonActionsIds.find(defaultRole); dmRoleToCommonActionsIds.end() != dmRoleIt) {
                for (const auto& actionId : dmRoleIt->second) {
                    actionsIds.insert(actionId);
                }
            }
            LinkActions(permissions->GetUserId(), roleId, actionsIds, *server, tx);
        }

        if (
            !affiliatedCompanyTagDesc->GetCompanyRoles().contains(defaultRole) ||
            !affiliatedCompanyTagDesc->GetCompanyRoles().at(defaultRole).contains(roleId)
        ) {
            auto affiliatedCompanyTagDescUpdate = affiliatedCompanyTagDesc->Clone();
            auto affiliatedCompanyTagDescUpdateImpl = std::dynamic_pointer_cast<TDescription>(affiliatedCompanyTagDescUpdate);
            affiliatedCompanyTagDescUpdateImpl->MutableCompanyRoles()[defaultRole].insert(roleId);
            R_ENSURE(
                  api.GetTagsManager().GetTagsMeta().RegisterTag(affiliatedCompanyTagDescUpdateImpl, permissions->GetUserId(), tx)
                , HTTP_INTERNAL_SERVER_ERROR
                , "cannot update tag " << affiliatedCompanyTagDescUpdateImpl->GetName()
                , tx
            );
        }

    }

    NJson::TJsonValue TUserOrganizationAffiliationTag::TDescription::DoSerializeMetaToJson() const {
        NJson::TJsonValue result;
        NJson::FieldsToJson(result, GetFields());
        NJson::InsertField(result, "company_roles", NJson::KeyValue(CompanyRoles, "name", "roles"));
        NJson::InsertNonNull(result, "company_information", NJson::KeyValue(CompanyInformation));
        return result;
    }

    bool TUserOrganizationAffiliationTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
        bool result = NJson::TryFieldsFromJson(json, GetFields());
        if (json["company_roles"].IsMap()) {
            result &= NJson::TryFromJson(json["company_roles"], NJson::Dictionary(CompanyRoles));
        } else {
            result &= NJson::ParseField(json["company_roles"], NJson::KeyValue(CompanyRoles, "name", "roles"));
        }
        const NJson::TJsonValue::TArray* companyInformationArray;
        if (json["company_information"].GetArrayPointer(&companyInformationArray)) {
            for (auto&& obj : *companyInformationArray) {
                TString key;
                TString value;
                if (!obj["key"].GetString(&key) || !obj["value"].GetString(&value)) {
                    ERROR_LOG << "Incorrect company information format: " << value << Endl;
                    continue;
                }
                CompanyInformation[key] = value;
            }
        }
        return result;
    }

    TDBActions TUserOrganizationAffiliationTag::GetActions(const TConstDBTag& /*self*/, const IDriveTagsManager& tagsManager, const TRolesManager& rolesManager, bool /* getPotential */) const {
        TTagDescription::TConstPtr description = tagsManager.GetTagsMeta().GetDescriptionByName(GetName(), Now());
        auto descriptionImpl = description->GetAs<TDescription>();

        if (!descriptionImpl || descriptionImpl->GetCompanyRoles().empty()) {
            return {};
        }

        TSet<TString> actionIds;
        for (const auto& [dmRole, roles] : descriptionImpl->GetCompanyRoles()) {
            if (Roles.contains(dmRole)) {
                auto restoreActions = rolesManager.GetRolesInfoDB().GetActions(roles);
                actionIds.insert(
                    restoreActions.begin(), restoreActions.end()
                );
            }
        }

        TDBActions result;
        TMap<TString, TDBAction> additionalActions = rolesManager.GetActions(actionIds);
        for (auto&& [_, action] : additionalActions) {
            result.emplace_back(action);
        }

        if (IsRealtime && Roles.contains(DriverRoleName)) {
            auto action = CreateObserveAction(*descriptionImpl);
            if (action) {
                result.emplace_back(std::move(action));
            }
        }

        return result;
    }

    [[nodiscard]] bool TUserOrganizationAffiliationTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
        auto optionalTaggedUser = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(self.GetObjectId(), tx);
        if (!optionalTaggedUser) {
            return false;
        }
        TVector<TDBTag> userTags = optionalTaggedUser->GetTagsByClass<TACLTag>();
        for (auto&& tag : userTags) {
            auto tagImpl = tag.GetTagAs<TACLTag>();
            if (!tagImpl) {
                continue;
            }
            if (tag.GetObjectId() == self.GetObjectId()) {
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(tag, userId, server, tx)) {
                    return false;
                }
            }
        }
        return true;
    }

    void TUserOrganizationAffiliationTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
        TBase::SerializeSpecialDataToJson(json);
        json["roles"] = NJson::ToJson(Roles);
        NJson::InsertField(json, "is_realtime", IsRealtime);
    }

    bool TUserOrganizationAffiliationTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* /* errors */) {
        return NJson::ParseField(json["roles"], Roles) && NJson::ParseField(json["is_realtime"], IsRealtime, false);
    }

    NDrive::TScheme TUserOrganizationAffiliationTag::GetScheme(const NDrive::IServer* server) const {
        NDrive::TScheme result = TBase::GetScheme(server);
        TRoles roles{TUserOrganizationAffiliationTag::AdminRoleName, TUserOrganizationAffiliationTag::DriverRoleName};
        if (server) {
            if (auto desc = GetDescriptionAs<TUserOrganizationAffiliationTag::TDescription>(*server)) {
                roles = MakeSet(NContainer::Keys(desc->GetCompanyRoles()));
            }
        }
        result.Add<TFSVariants>("roles", "Роли пользователя").SetMultiSelect(true).SetVariants(roles);
        return result;
    }

    TUserOrganizationAffiliationTag::TDescriptionConstPtr TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        auto optionalObject = server.GetDriveDatabase().GetTagsManager().GetUserTags().GetCachedOrRestoreObject(userId, tx);
        R_ENSURE(optionalObject, {}, "cannot restore object " << userId, tx);
        const auto& taggedObject = *optionalObject;
        return GetAffiliatedCompanyTagDescription(taggedObject, server);
    }

    TUserOrganizationAffiliationTag::TDescriptionConstPtr TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        return GetAffiliatedCompanyTagDescription(permissions->GetUserId(), server, tx);
    }

    TUserOrganizationAffiliationTag::TDescriptionConstPtr TUserOrganizationAffiliationTag::GetAffiliatedCompanyTagDescription(const TTaggedObject& taggedUser, const NDrive::IServer& server) {
        auto dbTag = taggedUser.GetFirstTagByClass<TUserOrganizationAffiliationTag>();
        R_ENSURE(dbTag, HTTP_BAD_REQUEST, "user has no affiliation tag");
        auto description = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(dbTag->GetName());
        R_ENSURE(description, HTTP_INTERNAL_SERVER_ERROR, "cannot find description " << dbTag->GetName());
        auto impl = std::dynamic_pointer_cast<const TUserOrganizationAffiliationTag::TDescription>(description);
        R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot cast description " << description->GetName() << " to UserOrganizationAffiliationTag::Description");
        return impl;
    }

    ITag::TConstPtr TUserOrganizationAffiliationTag::GetAffiliatedCompanyTag(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        auto optionalObject = server.GetDriveDatabase().GetTagsManager().GetUserTags().GetCachedOrRestoreObject(userId, tx);
        R_ENSURE(optionalObject, {}, "cannot restore object " << userId, tx);
        const auto& taggedUser = *optionalObject;
        auto dbTag = taggedUser.GetFirstTagByClass<TUserOrganizationAffiliationTag>();
        R_ENSURE(dbTag, HTTP_BAD_REQUEST, "user has no affiliation tag");
        return dbTag.GetData();
    }

    const TUserOrganizationAffiliationTag::TNotifierIdsReference& TUserOrganizationAffiliationTag::GetNotifiersCompany(const NDrive::IServer& server) const {
        const auto tdCompany = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
        auto tdCompanyImpl = tdCompany->GetAs<const TDescription>();
        if (tdCompanyImpl) {
            return tdCompanyImpl->GetTelegramNotifier();
        }
        return Default<TNotifierIdsReference>();
    }
}
