#pragma once

#include <drive/backend/actions/administrative.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/common/temporary_tags.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/roles/permissions.h>

#include <rtline/util/types/field.h>

class TTaggedObject;

namespace NDrivematics {

    class TUserOrganizationAffiliationTag : public IJsonSerializableTag, public IUserActionTag {
    public:
        enum class ESupportNotifierType {
            Unsupport = 0 /* "unsupport" */,
            TelegramCompany = 1 /* "telegram_company" */
        };

        struct TNotifierIdsReference {
        public:
            using TChatId = TString;
            using TBotId = TString;
            using TChatIdToBotId = TMap<TChatId, TBotId>;
            using TBotIdToPairRef = TMap<TBotId, TVector<std::reference_wrapper<std::pair<const TChatId, TBotId>>>>;

        private:
            R_FIELD(TChatIdToBotId, ChatIdToBotId);
            R_FIELD(TBotIdToPairRef, BotIdsToPairRef);
            R_FIELD(TSet<TBotId>, OrphanBotId);

        private:
            void RefreshReference(const TChatId& chatId);
            void RefreshReference(const TVector<TChatId>* chatIds = nullptr);

        public:
            NJson::TJsonValue SerializeToJson() const;
            bool DeserializeFromJson(const NJson::TJsonValue& value);

            bool ChangeGlobalBotId(const TBotId& oldBotId, const TBotId& newBotId);
            bool ChangeLocalBotId(const TChatId& chatId, const TBotId& newBotId);
            bool Add(const TBotId& botId, const TChatId* chatId = nullptr);
            TMaybe<TBotId> GetBotId(const TChatId& chatId) const;
            TSet<TBotId> GetBotIds() const;
            void RemoveChatId(const TChatId& chatId);
        };

    public:
        class TDescription : public TTagDescription {
        public:
            virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        public:
            ui64 GetUid() const;
        private:
            virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
            virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

        private:
            using TCompanyInformation = TMap<TString, TString>;
            using TCompanyRoles = TMap<TString, TSet<TString>>;

        private:
            R_FIELD(TString, CompanyName);
            R_FIELD(TString, OwningCarTagName);
            R_FIELD(TString, Phone);
            R_FIELD(TString, Email);
            R_FIELD(TString, BookingConfirmedSmsTagName);
            R_FIELD(TString, BookingCancelledSmsTagName);
            R_FIELD(TString, BookingConfirmedEmailTagName);
            R_FIELD(TString, BookingCancelledEmailTagName);
            R_FIELD(TString, AfterFinishRideEmailTagName);
            R_FIELD(TString, AfterAcceptanceEmailTagName);
            R_FIELD(TString, ShortDescription);
            R_FIELD(TString, IconMobileLogin);
            R_FIELD(TCompanyInformation, CompanyInformation);
            R_FIELD(TCompanyRoles, CompanyRoles);
            R_FIELD(TNotifierIdsReference, TelegramNotifier);

        public:
            DECLARE_FIELDS(
                Field(CompanyName, "company_name"),
                Field(OwningCarTagName, "owning_car_tag_name"),
                Field(Phone, "phone"),
                Field(Email, "email"),
                Field(BookingConfirmedSmsTagName, "booking_confirmed_sms_tag_name"),
                Field(BookingCancelledSmsTagName, "booking_cancelled_sms_tag_name"),
                Field(BookingConfirmedEmailTagName, "booking_confirmed_email_tag_name"),
                Field(BookingCancelledEmailTagName, "booking_cancelled_email_tag_name"),
                Field(AfterFinishRideEmailTagName, "after_finish_ride_email_tag_name"),
                Field(AfterAcceptanceEmailTagName, "after_acceptance_email_tag_name"),
                Field(ShortDescription, "short_description"),
                Field(IconMobileLogin, "icon_mobile_login"),
                Field(TelegramNotifier, "telegram_notifiers")
            );

        private:
            static const TFactory::TRegistrator<TDescription> Registrator;
        };
        using TDescriptionConstPtr = TAtomicSharedPtr<const TDescription>;
        using TRoles = TSet<TString>;

    private:
        using TBase = IJsonSerializableTag;

    public:
        using TBase::TBase;

    public:
        static void AddRoleAndTagsToUser(
              const TDriveUserData& driverUser
            , const TVector<TString>& tagNames
            , const TMap<TString, TVector<TString>>& dmRoleToCommonActionsIds
            , TUserPermissions::TPtr permissions
            , const NDrive::IServer* server
            , NDrive::TEntitySession& tx
            , const TRoles& userRoles = {TUserOrganizationAffiliationTag::DriverRoleName}
        );

    private:
        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { NEntityTagsManager::EEntityType::User };
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::Rewrite;
        }

        [[nodiscard]] virtual bool OnBeforeRemove(const TDBTag& self, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;

        TDBActions GetActions(const TConstDBTag& self, const IDriveTagsManager& tagsManager, const TRolesManager& rolesManager, bool getPotential) const override;

        virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
        virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    public:
        static inline const TString TypeName{"user_organization_affiliation_tag"};
        static TFactory::TRegistrator<TUserOrganizationAffiliationTag> Registrator;
        static inline const TString DriverRoleName{"driver"};
        static inline const TString AdminRoleName{"admin"};

    public:
        static TDescriptionConstPtr GetAffiliatedCompanyTagDescription(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx);
        static TDescriptionConstPtr GetAffiliatedCompanyTagDescription(TUserPermissions::TPtr permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx);
        static TDescriptionConstPtr GetAffiliatedCompanyTagDescription(const TTaggedObject& taggedUser, const NDrive::IServer& server);

        static ITag::TConstPtr GetAffiliatedCompanyTag(const TString& userId, const NDrive::IServer& server, NDrive::TEntitySession& tx);

        const TNotifierIdsReference& GetNotifiersCompany(const NDrive::IServer& server) const;

    private:
        R_FIELD(TRoles, Roles);
        R_FIELD(bool, IsRealtime, false);
    };
}
