#include "leasing.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/notifications/manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>

const TString NDrivematics::TLeasingStatsTag::TypeName = "leasing_stats";
ITag::TFactory::TRegistrator<NDrivematics::TLeasingStatsTag> NDrivematics::TLeasingStatsTag::Registrator(NDrivematics::TLeasingStatsTag::TypeName);

const TString NDrivematics::TEnableSessionTag::TypeName = "enable_session";
ITag::TFactory::TRegistrator<NDrivematics::TEnableSessionTag> NDrivematics::TEnableSessionTag::Registrator(NDrivematics::TEnableSessionTag::TypeName);
TTagDescription::TFactory::TRegistrator<NDrivematics::TEnableSessionTag::TDescription> NDrivematics::TEnableSessionTag::TDescription::Registrator(NDrivematics::TEnableSessionTag::TypeName);

const TString NDrivematics::TTaxiCompanyTag::TypeName = "taxi_company";
ITag::TFactory::TRegistrator<NDrivematics::TTaxiCompanyTag> NDrivematics::TTaxiCompanyTag::Registrator(NDrivematics::TTaxiCompanyTag::TypeName);
TTagDescription::TFactory::TRegistrator<NDrivematics::TTaxiCompanyTag::TDescription> NDrivematics::TTaxiCompanyTag::TDescription::Registrator(NDrivematics::TTaxiCompanyTag::TypeName);

const TString NDrivematics::TLeasingCompanyTag::TypeName = "leasing_company";
ITag::TFactory::TRegistrator<NDrivematics::TLeasingCompanyTag> NDrivematics::TLeasingCompanyTag::Registrator(NDrivematics::TLeasingCompanyTag::TypeName);
TTagDescription::TFactory::TRegistrator<NDrivematics::TLeasingCompanyTag::TDescription> NDrivematics::TLeasingCompanyTag::TDescription::Registrator(NDrivematics::TLeasingCompanyTag::TypeName);

const TString NDrivematics::TBrandTag::TypeName = "brand";
ITag::TFactory::TRegistrator<NDrivematics::TBrandTag> NDrivematics::TBrandTag::Registrator(NDrivematics::TBrandTag::TypeName);
TTagDescription::TFactory::TRegistrator<NDrivematics::TBrandTag::TDescription> NDrivematics::TBrandTag::TDescription::Registrator(NDrivematics::TBrandTag::TypeName);

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrivematics::TDailyLeasingStats& dailyStats) {
    return NJson::ParseField(value["day"], dailyStats.MutableDay(), true)
        && NJson::ParseField(value["sh"], dailyStats.MutableSupplyHours())
        && NJson::ParseField(value["utilization"], dailyStats.MutableUtilization())
        && NJson::ParseField(value["gmv"], dailyStats.MutableGMV())
        && NJson::ParseField(value["park_commission"], dailyStats.MutableParkCommission())
        && NJson::ParseField(value["churn"], dailyStats.MutableChurn());
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::TDailyLeasingStats& dailyStats) {
    NJson::TJsonValue result;
    result["day"] = dailyStats.GetDay();
    result["sh"] = dailyStats.GetSupplyHours();
    result["utilization"] = dailyStats.GetUtilization();
    result["gmv"] = dailyStats.GetGMV();
    result["park_commission"] = dailyStats.GetParkCommission();
    result["churn"] = dailyStats.GetChurn();
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrivematics::TAggregatedLeasingStats& aggregatedStats) {
    return NJson::TryFieldsFromJson(value, aggregatedStats.GetFields());
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::TAggregatedLeasingStats& aggregatedStats) {
    NJson::TJsonValue result;
    NJson::FieldsToJson(result, aggregatedStats.GetFields());
    return result;
}

namespace NDrivematics {

    void TLeasingStatsTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
        TBase::SerializeSpecialDataToJson(json);
        json["stats"] = NJson::ToJson(Stats);
        json["issue_date"] = IssueDate;
        json["weight"] = NJson::ToJson(Weight);
        json["attenuation"] = NJson::ToJson(Attenuation);
    }

    bool TLeasingStatsTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
        return TBase::DoSpecialDataFromJson(json, errors)
            && NJson::ParseField(json["stats"], Stats, true)
            && NJson::ParseField(json["issue_date"], IssueDate)
            && NJson::ParseField(json["weight"], Weight)
            && NJson::ParseField(json["attenuation"], Attenuation);
    }

    TLeasingStatsTag::TProto TLeasingStatsTag::DoSerializeSpecialDataToProto() const {
        NDrive::NProto::TLeasingStatsTag proto = TBase::DoSerializeSpecialDataToProto();
        for (const auto& dailyStats : Stats) {
            NDrive::NProto::TLeasingStatsTag::TDailyStats* protoInfo = proto.AddDailyStats();
            protoInfo->SetDay(dailyStats.GetDay());
            protoInfo->SetSupplyHours(dailyStats.GetSupplyHours());
            protoInfo->SetUtilization(dailyStats.GetUtilization());
            protoInfo->SetGMV(dailyStats.GetGMV());
            protoInfo->SetParkCommission(dailyStats.GetParkCommission());
            protoInfo->SetChurn(dailyStats.GetChurn());
        }
        proto.SetIssueDate(IssueDate);
        if (Weight) {
            proto.SetWeight(*Weight);
        }
        if (Attenuation) {
            proto.SetAttenuation(*Attenuation);
        }
        return proto;
    }

    bool TLeasingStatsTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
        for (const auto& protoStats : proto.GetDailyStats()) {
            TDailyLeasingStats stats;
            stats.SetDay(protoStats.GetDay());
            stats.SetSupplyHours(protoStats.GetSupplyHours());
            stats.SetUtilization(protoStats.GetUtilization());
            stats.SetGMV(protoStats.GetGMV());
            stats.SetParkCommission(protoStats.GetParkCommission());
            stats.SetChurn(protoStats.GetChurn());
            Stats.insert(std::move(stats));
        }
        IssueDate = proto.GetIssueDate();
        if (proto.HasWeight()) {
            Weight = proto.GetWeight();
        }
        if (proto.HasAttenuation()) {
            Attenuation = proto.GetAttenuation();
        }
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }

    bool TLeasingStatsTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
        if (!TBase::OnBeforeAdd(objectId, userId, server, tx)) {
            return false;
        }
        TVector<TDBTag> dbTags;
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({objectId}, {TypeName}, dbTags, tx)) {
            return false;
        }
        for (const auto& dbTag : dbTags) {
            const auto* tag = dbTag.GetTagAs<TLeasingStatsTag>();
            Y_ENSURE(tag);
            MergeFrom(*tag);
        }
        return true;
    }

    void TLeasingStatsTag::MergeFrom(const TLeasingStatsTag& old) {
        for (const auto& dailyStats : old.GetStats()) {
            //we merge from old tag, so we do not copy the entries with existing day
            auto it = Stats.find(dailyStats);
            if (it != Stats.end()) {
                continue;
            }
            Stats.insert(dailyStats);
        }
        if (!IssueDate) {
            IssueDate = old.GetIssueDate();
        }
        if (!Weight) {
            Weight = old.OptionalWeight();
        }
        if (!Attenuation) {
            Attenuation = old.OptionalAttenuation();
        }
    }

    TMaybe<TAggregatedLeasingStats> TLeasingStatsTag::AggregateStats(ui32 dayFrom, ui32 dayUntil) const {
        if (!IssueDate || dayUntil < IssueDate) {
            return Nothing();
        }
        TAggregatedLeasingStats result;
        std::size_t count = 0;
        double utilization = 0;
        for (const auto& dailyStats : Stats) {
            if (IssueDate <= dailyStats.GetDay() && dayFrom <= dailyStats.GetDay() && dailyStats.GetDay() <= dayUntil) {
                ++count;
                result.MutableSupplyHours() += dailyStats.GetSupplyHours();
                utilization += dailyStats.GetUtilization();
                result.MutableGMV() += dailyStats.GetGMV();
                result.MutableParkCommission() += dailyStats.GetParkCommission();
            }
        }
        if (count) {
            result.MutableUtilization() = utilization * 100 / count;
        }
        result.SetCarsCount(1);
        result.SetWeight(Weight.GetOrElse(1));
        result.OptionalAttenuation() = Attenuation;
        return result;
    }

    TMaybe<TAggregatedLeasingStats> TLeasingStatsTag::AggregateStats(ui32 daysCount) const {
        if (!IssueDate) {
            return Nothing();
        }
        auto it = Stats.rbegin();
        if (it == Stats.rend()) {
            return Nothing();
        }
        return AggregateStats(it->GetDay() - daysCount + 1, it->GetDay());
    }

    TAggregatedLeasingStats TLeasingStatsTag::CalculateStatsCommon(const TLeasingStatsTagsRefs& tags, ui32 noSHThreshold, const std::function<TMaybe<TAggregatedLeasingStats>(const TLeasingStatsTag&)>& singleTagStatsAggregator) {
        TAggregatedLeasingStats result;
        for (const auto& tag : tags) {
            auto maybeCurStats = singleTagStatsAggregator(tag.get());
            if (!maybeCurStats) {
                continue;
            }
            const auto& curStats = *maybeCurStats;
            if (curStats.GetSupplyHours() < noSHThreshold) {
                ++result.MutableInsufficientSupplyHoursCarsCount();
            }
            if (curStats.HasAttenuation()) {
                result.MutableSupplyHours()    += curStats.GetAttenuationRef() * curStats.GetWeight() * curStats.GetSupplyHours();
                result.MutableUtilization()    += static_cast<ui32>(curStats.GetAttenuationRef() * curStats.GetWeight() * curStats.GetUtilization());
                result.MutableGMV()            += curStats.GetAttenuationRef() * curStats.GetWeight() * curStats.GetGMV();
                result.MutableParkCommission() += curStats.GetAttenuationRef() * curStats.GetWeight() * curStats.GetParkCommission();
            } else {
                result.MutableSupplyHours()    += curStats.GetWeight() * curStats.GetSupplyHours();
                result.MutableUtilization()    += curStats.GetWeight() * curStats.GetUtilization();
                result.MutableGMV()            += curStats.GetWeight() * curStats.GetGMV();
                result.MutableParkCommission() += curStats.GetWeight() * curStats.GetParkCommission();
            }
            result.MutableCarsCount() += curStats.GetWeight() * curStats.GetCarsCount();
        }
        return result;
    }

    TAggregatedLeasingStats TLeasingStatsTag::CalculateTaxiCompanyStats(const TLeasingStatsTagsRefs& tags, ui32 noSHThreshold, const std::function<TMaybe<TAggregatedLeasingStats>(const TLeasingStatsTag&)>& singleTagStatsAggregator) {
        auto result = CalculateStatsCommon(tags, noSHThreshold, singleTagStatsAggregator);
        if (result.GetCarsCount()) {
            result.MutableSupplyHours() /= result.GetCarsCount();
            result.MutableUtilization() /= result.GetCarsCount();
        }
        return result;
    }

    TAggregatedLeasingStats TLeasingStatsTag::CalculatePortfolioStats(const TLeasingStatsTagsRefs& tags, ui32 noSHThreshold, const std::function<TMaybe<TAggregatedLeasingStats>(const TLeasingStatsTag&)>& singleTagStatsAggregator) {
        auto result = CalculateStatsCommon(tags, noSHThreshold, singleTagStatsAggregator);
        if (result.GetCarsCount()) {
            result.MutableSupplyHours() /= result.GetCarsCount();
            result.MutableUtilization() /= result.GetCarsCount();
            result.MutableParkCommission() /= result.GetCarsCount();
            result.MutableGMV() /= result.GetCarsCount();
        }
        return result;
    }

    NDrive::TScheme TEnableSessionTag::TDescription::GetScheme(const NDrive::IServer* /* server */) const {
        NDrive::TScheme result;
        result.Add<TFSString>("setting_path", "Путь в GVars до .offer_name").SetRequired(true);
        return result;
    }

    NJson::TJsonValue TEnableSessionTag::TDescription::DoSerializeMetaToJson() const {
        return NJson::TMapBuilder("setting_path", SettingPath);
    }
    bool TEnableSessionTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
        return NJson::ParseField(value["setting_path"], SettingPath);
    }

    NDrive::TScheme TEnableSessionTag::GetScheme(const NDrive::IServer* /*server*/) const {
        NDrive::TScheme result;
        return result;
    }

    bool TEnableSessionTag::OnBeforeRemove(const TDBTag& self, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
        const auto& driveApi = server->GetDriveAPI();
        const auto& tagManager = driveApi->GetTagsManager().GetDeviceTags();
        auto restoredTags = tagManager.RestoreTags(TVector<TString>{ self.GetObjectId() }, {
            TChargableTag::Reservation,
            TChargableTag::Riding,
            TChargableTag::Parking,
        }, tx);
        const auto offer = driveApi->GetCurrentCarOffer(self.GetObjectId());
        if (!restoredTags || restoredTags->size() != 1) {
            if (!offer) {
                return true;
            }
            return false;
        }
        const auto& chargableTag = restoredTags->front();
        if (chargableTag->GetName() == TChargableTag::Reservation && chargableTag->GetPerformer().empty()) {
            return true;
        }
        const auto& tagsMeta = driveApi->GetTagsManager().GetTagsMeta();
        auto tagDescriptionsPtr = tagsMeta.GetDescriptionByName(self->GetName());
        if (!tagDescriptionsPtr) {
            return false;
        }
        auto tagDescription = tagDescriptionsPtr->GetAs<TDescription>();
        if (!tagDescription) {
            return false;
        }
        auto tagOfferName = server->GetSettings().GetValue<TString>(tagDescription->GetSettingPath() + ".offer_name");
        if (!tagOfferName) {
            return false;
        }
        if (offer->GetName() == *tagOfferName) {
            const auto performerPermissions = driveApi->GetUserPermissions(chargableTag->GetPerformer());
            if (!performerPermissions) {
                return false;
            }
            return TChargableTag::DirectEvolve(chargableTag, TChargableTag::Reservation, *performerPermissions, *server, tx, nullptr);
        }
        return true;
    }

    bool IUniqueTypeNameTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
        if (!TBase::OnBeforeAdd(objectId, userId, server, tx)) {
            return false;
        }
        const auto& deviceTagManager = server->GetDriveDatabase().GetTagsManager().GetDeviceTags();
        if (!CachedObject) {
            auto object = deviceTagManager.RestoreObject(objectId, tx);
            if (!object) {
                return false;
            }
            CachedObject = std::move(*object);
        }
        auto tagsToRemove = GetTagsToRemove(*CachedObject);
        if (!deviceTagManager.RemoveTagsSimple(tagsToRemove, userId, tx, false)) {
            return false;
        }
        return true;
    }

    NDrive::TScheme TTaxiCompanyTag::TDescription::GetScheme(const NDrive::IServer* /* server */) const {
        NDrive::TScheme result;
        result.Add<TFSString>("name", "Имя таксопарка").SetRequired(true);
        result.Add<TFSString>("city", "Город").SetRequired(true);
        result.Add<TFSNumeric>("tin", "ИНН");
        result.Add<TFSNumeric>("score", "Оценка");
        result.Add<TFSNumeric>("telematics_cars_score", "Оценка машин телематики");
        return result;
    }

    NJson::TJsonValue TTaxiCompanyTag::TDescription::DoSerializeMetaToJson() const {
        return NJson::TMapBuilder("name", TaxiCompanyName)
                                 ("city", City)
                                 ("score", Score)
                                 ("telematics_cars_score", TelematicsCarsScore)
                                 ("tin", Tin);
    }

    bool TTaxiCompanyTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
        return NJson::ParseField(value["name"], TaxiCompanyName, true)
            && NJson::ParseField(value["city"], City)
            && NJson::ParseField(value["score"], Score)
            && NJson::ParseField(value["telematics_cars_score"], TelematicsCarsScore)
            && NJson::ParseField(value["tin"], Tin);
    }

    TVector<TDBTag> TTaxiCompanyTag::GetTagsToRemove(const TTaggedObject& object) const {
        return object.GetTagsByClass<TTaxiCompanyTag>();
    }

    NDrive::TScheme TLeasingCompanyTag::TDescription::GetScheme(const NDrive::IServer* /* server */) const {
        NDrive::TScheme result;
        result.Add<TFSString>("name", "Имя лизинговой компании").SetRequired(true);
        result.Add<TFSNumeric>("score", "Оценка");
        return result;
    }

    NJson::TJsonValue TLeasingCompanyTag::TDescription::DoSerializeMetaToJson() const {
        return NJson::TMapBuilder("name", LeasingCompanyName)
                                 ("score", Score);
    }

    bool TLeasingCompanyTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
        return NJson::ParseField(value["name"], LeasingCompanyName, true)
            && NJson::ParseField(value["score"], Score);
    }

    TVector<TDBTag> TLeasingCompanyTag::GetTagsToRemove(const TTaggedObject& object) const {
        return object.GetTagsByClass<TLeasingCompanyTag>();
    }

    NDrive::TScheme TBrandTag::TDescription::GetScheme(const NDrive::IServer* /* server */) const {
        NDrive::TScheme result;
        result.Add<TFSString>("name", "Имя бренда");
        result.Add<TFSString>("id", "id бренда").SetRequired(true);
        result.Add<TFSArray>("taxi_companies_tins", "ИННы таксопарков, принадлежащих бренду").SetElement<TFSNumeric>();
        return result;
    }

    NJson::TJsonValue TBrandTag::TDescription::DoSerializeMetaToJson() const {
        return NJson::TMapBuilder("name", BrandName)
                                 ("taxi_companies_tins", NJson::ToJson(TaxiCompaniesTins))
                                 ("id", BrandId);
    }

    bool TBrandTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
        auto result = NJson::ParseField(value["name"], BrandName)
                   && NJson::ParseField(value["id"], BrandId);
        if (!result) {
            return false;
        }
        TVector<ui64> taxiCompaniesTinsInts;
        if (NJson::ParseField(value["taxi_companies_tins"], taxiCompaniesTinsInts)) {
            for (ui64 tin : taxiCompaniesTinsInts) {
                TaxiCompaniesTins.push_back(ToString(tin));
            }
        } else {
            if (!NJson::ParseField(value["taxi_companies_tins"], TaxiCompaniesTins)) {
                return false;
            }
        }
        return true;
    }

    TVector<TDBTag> TBrandTag::GetTagsToRemove(const TTaggedObject& object) const {
        return object.GetTagsByClass<TBrandTag>();
    }

    TVector<TDBTag> THasTelematicsTag::GetTagsToRemove(const TTaggedObject& object) const {
        return object.GetTagsByClass<THasTelematicsTag>();
    }

    TVector<TDBTag> THasSignalqTag::GetTagsToRemove(const TTaggedObject& object) const {
        return object.GetTagsByClass<THasSignalqTag>();
    }
}
