#pragma once

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/common/temporary_tags.h>
#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/actions/administrative.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/notifications/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <rtline/util/types/field.h>

#include <util/generic/hash.h>

class TTaggedObject;

namespace NDrivematics {
    const TString InsufficientSupplyHoursTagName = "insufficient_sh_tag";
    const TString FreshIssueDateTagName = "fresh_issue_date_tag";
    enum class ELeasingStatsType {
        Numeric     = 0  /* "numeric" */,
        Percent     = 1  /* "percent" */,
        Money       = 2  /* "money" */,
        Kilometers  = 3  /* "kilometers" */,
        Hours       = 4  /* "hours" */,
    };

    class TDailyLeasingStats {
    public:
        bool operator<(const TDailyLeasingStats& rhs) const {
            return Day < rhs.Day;
        }

    public:
        R_FIELD(ui32, Day, 0);
        R_FIELD(double, SupplyHours, 0);
        R_FIELD(double, Utilization, 0);
        R_FIELD(double, GMV, 0);
        R_FIELD(double, ParkCommission, 0);
        R_FIELD(double, Churn, 0);
    };

    using TLeasingTraits = ui32;
    enum ELeasingTraits : ui32 {
        ReportInPortfolio = 1 << 0,
        ReportInTaxiCompanies = 1 << 1,
        ReportInCarList = 1 << 2,
    };

    template<class T>
    class TLeasingField : public TField<T> {
    public:
        TLeasingField(T&& value, TStringBuf name, TStringBuf displayName, ELeasingStatsType statsType, TLeasingTraits traits)
            : TField<T>(std::forward<T>(value), name, 0)
            , DisplayName(displayName)
            , StatsType(statsType)
            , Traits(traits)
        {
        }

    private:
        R_READONLY(TStringBuf, DisplayName);
        R_READONLY(ELeasingStatsType, StatsType);
        R_READONLY(TLeasingTraits, Traits, 0);
    };

    template <class T>
    auto LeasingField(T&& value, TStringBuf name, TStringBuf displayName, ELeasingStatsType statsType, TLeasingTraits traits) {
        return TLeasingField<T>(std::forward<T>(value), name, displayName, statsType, traits);
    }

    class TAggregatedLeasingStats {
    public:
        R_FIELD(ui32, CarsCount, 0);
        R_FIELD(double, SupplyHours, 0);
        R_FIELD(ui32, Utilization, 0);
        R_FIELD(double, GMV, 0);
        R_FIELD(double, ParkCommission, 0);
        R_FIELD(ui32, InsufficientSupplyHoursCarsCount, 0);
        R_FIELD(ui32, FreshIssueDateCarsCount, 0);
        R_FIELD(ui32, Weight, 0);
        R_OPTIONAL(double, Attenuation);

    public:
        DECLARE_FIELDS(
            LeasingField(Utilization, "utilization", "Сдаваемость", ELeasingStatsType::Percent, ELeasingTraits::ReportInPortfolio | ELeasingTraits::ReportInTaxiCompanies | ELeasingTraits::ReportInCarList),
            LeasingField(SupplyHours, "sh", "Supply Hours", ELeasingStatsType::Numeric, ELeasingTraits::ReportInPortfolio | ELeasingTraits::ReportInTaxiCompanies | ELeasingTraits::ReportInCarList),
            LeasingField(CarsCount, "cars_count", "Машины", ELeasingStatsType::Numeric, ELeasingTraits::ReportInPortfolio | ELeasingTraits::ReportInTaxiCompanies),
            LeasingField(GMV, "gmv", "GMV на 1 машину", ELeasingStatsType::Money, ELeasingTraits::ReportInPortfolio | ELeasingTraits::ReportInTaxiCompanies),
            LeasingField(ParkCommission, "park_commission", "Комиссия на 1 машину", ELeasingStatsType::Money, ELeasingTraits::ReportInPortfolio | ELeasingTraits::ReportInTaxiCompanies),
            LeasingField(InsufficientSupplyHoursCarsCount, "no_sh", "Не выполняют SH", ELeasingStatsType::Numeric, ELeasingTraits::ReportInPortfolio),
            LeasingField(FreshIssueDateCarsCount, "fresh_issue_date", "Имеют дату выдачи менее 60 дней", ELeasingStatsType::Numeric, ELeasingTraits::ReportInPortfolio)
        );
    };

    class TLeasingStatsTag;

    using TLeasingStatsTagsRefs = TVector<std::reference_wrapper<const TLeasingStatsTag>>;

    class TLeasingStatsTag : public ISerializableTag<NDrive::NProto::TLeasingStatsTag> {
    public:
        using TBase = ISerializableTag<NDrive::NProto::TLeasingStatsTag>;

    public:
        using TBase::TBase;

        TMaybe<TAggregatedLeasingStats> AggregateStats(ui32 dayFrom, ui32 dayUntil) const;
        TMaybe<TAggregatedLeasingStats> AggregateStats(ui32 daysCount) const;

    public:
        static TAggregatedLeasingStats CalculateTaxiCompanyStats(const TLeasingStatsTagsRefs& tags, ui32 noSHThreshold, const std::function<TMaybe<TAggregatedLeasingStats>(const TLeasingStatsTag&)>& singleTagStatsAggregator);
        static TAggregatedLeasingStats CalculatePortfolioStats(const TLeasingStatsTagsRefs& tags, ui32 noSHThreshold, const std::function<TMaybe<TAggregatedLeasingStats>(const TLeasingStatsTag&)>& singleTagStatsAggregator);

    private:
        static TAggregatedLeasingStats CalculateStatsCommon(const TLeasingStatsTagsRefs& tags, ui32 noSHThreshold, const std::function<TMaybe<TAggregatedLeasingStats>(const TLeasingStatsTag&)>& singleTagStatsAggregator);

    private:
        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { NEntityTagsManager::EEntityType::Car };
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::Rewrite;
        }

        virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
        virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;

        virtual TProto DoSerializeSpecialDataToProto() const override;
        virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

        [[nodiscard]] virtual bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;

        void MergeFrom(const TLeasingStatsTag& old);

    public:
        static const TString TypeName;
        static TFactory::TRegistrator<TLeasingStatsTag> Registrator;

    private:
        R_FIELD(TSet<TDailyLeasingStats>, Stats);
        R_FIELD(ui32, IssueDate, 0);
        R_OPTIONAL(ui32, Weight);
        R_OPTIONAL(double, Attenuation);
    };

    class IUniqueTypeNameTag : public IJsonSerializableTag {
    private:
        using TBase = IJsonSerializableTag;

    public:
        using TBase::TBase;

        void SetCachedObject(TTaggedObject&& value) {
            CachedObject = std::move(value);
        }

    private:
        [[nodiscard]] virtual bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;
        [[nodiscard]] virtual TVector<TDBTag> GetTagsToRemove(const TTaggedObject& object) const = 0;

    public:
        TMaybe<TTaggedObject> CachedObject;
    };

    class TEnableSessionTag: public IJsonSerializableTag {
    public:
        using TBase = IJsonSerializableTag;

    public:
        static const TString TypeName;

    public:
        using TBase::TBase;

    public:
        class TDescription : public TTagDescription {
        public:
            virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        private:
            virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
            virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

        private:
            static TFactory::TRegistrator<TDescription> Registrator;

            R_FIELD(TString, SettingPath);;
        };

        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return {NEntityTagsManager::EEntityType::Car};
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::NoUnique;
        }

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    public:
        [[nodiscard]] bool OnBeforeRemove(const TDBTag& tag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;

    private:
        static TFactory::TRegistrator<TEnableSessionTag> Registrator;
    };

    class TTaxiCompanyTag : public IUniqueTypeNameTag {
    public:
        class TDescription : public TTagDescription {
        public:
            virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        private:
            virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
            virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

        private:
            static TFactory::TRegistrator<TDescription> Registrator;

        public:
            R_FIELD(TString, TaxiCompanyName);
            R_FIELD(TString, City);
            R_FIELD(double, Score, 0);
            R_FIELD(double, TelematicsCarsScore, 0);
            R_FIELD(ui64, Tin, 0);
        };

    private:
        using TBase = IUniqueTypeNameTag;

    public:
        using TBase::TBase;

    private:
        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { NEntityTagsManager::EEntityType::Car };
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::Rewrite;
        }

        [[nodiscard]] virtual TVector<TDBTag> GetTagsToRemove(const TTaggedObject& object) const override;

    public:
        static const TString TypeName;
        static TFactory::TRegistrator<TTaxiCompanyTag> Registrator;
    };

    class TLeasingCompanyTag : public IUniqueTypeNameTag {
    public:
        class TDescription : public TTagDescription {
        public:
            virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        private:
            virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
            virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

        private:
            static TFactory::TRegistrator<TDescription> Registrator;
            R_FIELD(TString, LeasingCompanyName);
            R_FIELD(double, Score, 0);
        };

    private:
        using TBase = IUniqueTypeNameTag;

    public:
        using TBase::TBase;

    private:
        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { NEntityTagsManager::EEntityType::Car };
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::Rewrite;
        }

        [[nodiscard]] virtual TVector<TDBTag> GetTagsToRemove(const TTaggedObject& object) const override;

    public:
        static const TString TypeName;
        static TFactory::TRegistrator<TLeasingCompanyTag> Registrator;
    };

    class TBrandTag : public IUniqueTypeNameTag {
    public:
        class TDescription : public TTagDescription {
        public:
            virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

        private:
            virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
            virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& value) override;

        private:
            static TFactory::TRegistrator<TDescription> Registrator;

        public:
            R_FIELD(TString, BrandName);
            R_FIELD(TString, BrandId, 0);
            R_FIELD(TVector<TString>, TaxiCompaniesTins);
        };

    private:
        using TBase = IUniqueTypeNameTag;

    public:
        using TBase::TBase;

    private:
        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { NEntityTagsManager::EEntityType::Car };
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::Rewrite;
        }

        [[nodiscard]] virtual TVector<TDBTag> GetTagsToRemove(const TTaggedObject& object) const override;

    public:
        static const TString TypeName;
        static TFactory::TRegistrator<TBrandTag> Registrator;
    };

    class THasTelematicsTag : public IUniqueTypeNameTag {
    private:
        using TBase = IUniqueTypeNameTag;

    public:
        using TBase::TBase;

    private:
        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { NEntityTagsManager::EEntityType::Car };
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::Rewrite;
        }

        [[nodiscard]] virtual TVector<TDBTag> GetTagsToRemove(const TTaggedObject& object) const override;

    public:
        static inline const TString TypeName{"leasing_has_telematics_tag"};
        static inline TFactory::TRegistrator<THasTelematicsTag> Registrator{TypeName};
    };


    class THasSignalqTag : public IUniqueTypeNameTag {
    private:
        using TBase = IUniqueTypeNameTag;

    public:
        using TBase::TBase;

    private:
        virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { NEntityTagsManager::EEntityType::Car };
        }

        virtual EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::SkipIfExists;
        }

        [[nodiscard]] virtual TVector<TDBTag> GetTagsToRemove(const TTaggedObject& object) const override;

    public:
        static inline const TString TypeName{"leasing_has_signalq_tag"};
        static inline TFactory::TRegistrator<THasSignalqTag> Registrator{TypeName};
    };
}
