#include "long_term.h"

#include "chargable.h"
#include "support_tags.h"
#include "user_tags.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <memory>
#include <rtline/library/json/proto/adapter.h>
#include <rtline/util/algorithm/type_traits.h>

const TString TLongTermOfferHolderTag::Preparation = "long_term_preparation";

bool TLongTermCarDeliveryTag::OnBeforePerform(TDBTag& /*self*/, const TUserPermissions& /*permissions*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& session) {
    if (!UserId || !OfferId) {
        session.SetErrorInfo("LongTermCarDeliveryTag::OnBeforePerform", "empty data: user_id '" + UserId + "', offer_id '" + OfferId + "'");
        return false;
    }
    return true;
}

bool TLongTermCarDeliveryTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (GetPerformer()) {
        const auto driveApi = Yensured(server)->GetDriveAPI();
        auto permissions = driveApi->GetUserPermissions(userId, {});
        if (!permissions) {
            session.SetErrorInfo("LongTermCarDeliveryTag::OnAfterRemove", "cannot recreate permissions");
            return false;
        }

        if (!UserId || !OfferId) {
            session.SetErrorInfo("LongTermCarDeliveryTag::OnAfterRemove", "empty data: user_id '" + UserId + "', offer_id '" + OfferId + "'");
            return false;
        }
        const auto& userTagsManager = Yensured(driveApi)->GetTagsManager().GetUserTags();
        auto targetUser = userTagsManager.RestoreObject(UserId, session);
        if (!targetUser) {
            return false;
        }
        bool found = false;
        for (auto&& tag : targetUser->GetTags()) {
            auto offerContainer = tag.GetTagAs<IOfferContainer>();
            auto offer = offerContainer ? offerContainer->GetOffer() : nullptr;
            if (!offer) {
                continue;
            }
            if (offer->GetOfferId() == OfferId) {
                auto bookingTag = MakeAtomicShared<TOfferBookingUserTag>();
                bookingTag->SetName(TOfferBookingUserTag::Type());
                bookingTag->SetCarId(self.GetObjectId());
                if (!userTagsManager.EvolveTag(tag, bookingTag, *permissions, server, session)) {
                    return false;
                }
                found = true;
                break;
            }
        }
        if (!found) {
            session.SetErrorInfo("LongTermCarDeliveryTag::OnAfterRemove", "cannot find offer " + OfferId);
            return false;
        }
    }
    return true;
}

bool TLongTermCarDeliveryTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return
        TBase::DoSpecialDataFromJson(value, errors) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

void TLongTermCarDeliveryTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}

NDrive::TScheme TLongTermOfferHolderTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    if (server) {
        const auto& tagMeta = server->GetDriveDatabase().GetTagsManager().GetTagsMeta();
        auto carTagDescriptions = tagMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Car);
        auto carTagNames = NContainer::Keys(carTagDescriptions);
        result.Add<TFSVariants>("child_seat_tag", "Car tag to add on evolution if child seat is requested").SetVariants(carTagNames);
        auto userTagNames = tagMeta.GetRegisteredTagNames({ TSupportOutgoingCommunicationTag::TypeName });
        result.Add<TFSVariants>("communication_tag", "Communication tag to add on evolution").SetVariants(userTagNames);
        result.Add<TFSBoolean>("check_deposit", "Check deposit on evolution").SetDefault(CheckDeposit);
        result.Add<TFSBoolean>("start_billing_task", "Start billing").SetDefault(StartBillingTask);
    }
    return result;
}

NJson::TJsonValue TLongTermOfferHolderTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TLongTermOfferHolderTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TBase::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

TOptionalDBTags TLongTermOfferHolderTag::RestoreOfferHolderTags(const TString& behaviourConstructorId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
    auto registeredTagNames = tagsMeta.GetRegisteredTagNames({ Type() });
    auto tagNames = MakeVector(registeredTagNames);
    return TBase::RestoreOfferHolderTags(behaviourConstructorId, tagNames, server, session);
}

bool TLongTermOfferHolderTag::AssignCarId(const TDBTag& tag, const TString& carId, const TString& tagName, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& session, bool overwrite) {
    const auto& userTagManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
    auto from = tag.GetTagAs<TLongTermOfferHolderTag>();
    if (!from) {
        session.SetErrorInfo("LongTermOfferHolderTag::SetObjectId", "cannot cast tag to LongTermOfferHolderTag");
        return false;
    }
    auto offer = std::dynamic_pointer_cast<IOffer>(from->GetOffer());
    if (!offer) {
        session.SetErrorInfo("LongTermOfferHolderTag::SetObjectId", "cannot find offer");
        return false;
    }
    if (offer->GetObjectId() && !overwrite) {
        session.SetErrorInfo("LongTermOfferHolderTag::SetObjectId", "car_id is already present: " + offer->GetObjectId());
        return false;
    }

    auto targetTag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName);
    auto communication = std::dynamic_pointer_cast<TSupportOutgoingCommunicationTag>(targetTag);
    auto to = std::dynamic_pointer_cast<TLongTermOfferHolderTag>(targetTag);
    Y_ENSURE(!communication || !to);
    if (!to && communication) {
        auto linked = communication->MutableLinkedTag(server, session);
        to = std::dynamic_pointer_cast<TLongTermOfferHolderTag>(linked);
    }
    if (!to) {
        session.SetErrorInfo("LongTermOfferHolderTag::AssignCarId", "cannot create long_term_offer_holder_tag " + tagName);
        return false;
    }
    to->SetAutoassigned(true);
    to->SetCarId(carId);

    if (communication) {
        auto optionalCommunicationTag = userTagManager.RestoreTag(from->GetCommunicationTagId(), session);
        if (!optionalCommunicationTag) {
            return false;
        }
        auto communicationTag = std::move(*optionalCommunicationTag);
        if (auto container = communicationTag.GetTagAs<IContainerTag>()) {
            auto optionalUndeferredTag = IContainerTag::UndeferContainer(communicationTag, permissions.Self(), &server, userTagManager, session);
            if (!optionalUndeferredTag) {
                return false;
            }
            communicationTag = std::move(*optionalUndeferredTag);
            if (!communicationTag) {
                session.SetErrorInfo("LongTermOfferHolderTag::AssignCarId", "cannot restore tag after undeferring");
                return false;
            }
        }
        if (auto impl = communicationTag.GetTagAs<TSupportOutgoingCommunicationTag>()) {
            communication->SetOriginalSupportLine(impl->GetOriginalSupportLine());
            communication->SetTopicLink(impl->GetTopicLink());
        }
        if (!userTagManager.EvolveTag(communicationTag, communication, permissions, &server, session)) {
            return false;
        }
    } else {
        if (!userTagManager.EvolveTag(tag, to, permissions, &server, session)) {
            return false;
        }
    }
    session.Committed().Subscribe([tag, carId](const NThreading::TFuture<void>& c) {
        if (c.HasValue()) {
            NDrive::TEventLog::Log("LongTermAssignCarId", NJson::TMapBuilder
                ("car_id", carId)
                ("tag", NJson::ToJson(tag))
            );
        }
    });
    return true;
}

bool TLongTermOfferHolderTag::IsPreparation() const {
    return GetName().Contains("preparation");
}

const TString& TLongTermOfferHolderTag::GetCarId() const {
    return CarId.GetOrElse(Default<TString>());
}

const TString& TLongTermOfferHolderTag::GetStage(const TTaggedObject* /*object*/) const {
    if (IsPreparation()) {
        return Preparation;
    }
    return TChargableTag::Prereservation;
}

bool TLongTermOfferHolderTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeAdd(objectId, userId, server, session)) {
        return false;
    }

    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& tagMeta = database.GetTagsManager().GetTagsMeta();
    const auto& userTagManager = database.GetTagsManager().GetUserTags();

    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        return false;
    }

    if (const auto& communicationTagName = description->GetCommunicationTag()) {
        auto tag = tagMeta.CreateTag(communicationTagName);
        auto communicationTag = std::dynamic_pointer_cast<TSupportOutgoingCommunicationTag>(tag);
        if (!communicationTag) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeAdd", "cannot create tag " + communicationTagName);
            return false;
        }
        auto added = userTagManager.AddTag(tag, userId, objectId, server, session);
        if (!added) {
            return false;
        }
        if (added->size() != 1) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeAdd", "could not AddTag " + communicationTagName);
            return false;
        }
        CommunicationTagId = added->front().GetTagId();
    }

    return true;
}

bool TLongTermOfferHolderTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterAdd(self, userId, server, session)) {
        return false;
    }

    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& userTagManager = database.GetTagsManager().GetUserTags();

    auto description = GetDescriptionAs<TDescription>(*server, session);
    if (!description) {
        return false;
    }

    if (CommunicationTagId) {
        auto communicationDbTag = userTagManager.RestoreTag(CommunicationTagId, session);
        if (!communicationDbTag) {
            return false;
        }
        auto communicationTag = communicationDbTag->MutableTagAs<TSupportOutgoingCommunicationTag>();
        if (!communicationTag) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnAfterAdd", "cannot cast tag " + communicationDbTag->GetTagId() + " to SupportOutgoingCommunicationTag");
            return false;
        }
        communicationTag->SetLinkedTagId(self.GetTagId());
        if (!userTagManager.UpdateTagData(*communicationDbTag, userId, session)) {
            return false;
        }
    }
    if (CommunicationTagId) {
        auto communicationDbTag = userTagManager.RestoreTag(CommunicationTagId, session);
        if (!communicationDbTag) {
            return false;
        }
        auto communicationTag = communicationDbTag->MutableTagAs<TSupportOutgoingCommunicationTag>();
        if (!communicationTag) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnAfterAdd", "cannot cast tag " + communicationDbTag->GetTagId() + " to SupportOutgoingCommunicationTag after update");
            return false;
        }
        {
            const auto offer = GetOffer();
            const auto longTermOffer = offer ? std::dynamic_pointer_cast<const TLongTermOffer>(offer) : nullptr;
            if (longTermOffer && longTermOffer->GetShouldDeferCommunication()) {
                TInstant deferUntil = Now() + longTermOffer->GetDefaultDeferPeriod();
                if (longTermOffer->GetSince() > Now() + longTermOffer->GetDeferDelay()) {
                    TInstant targetDay = longTermOffer->GetSince() - longTermOffer->GetDelayedDeferPeriod();
                    deferUntil = TInstant::Days(targetDay.Days());
                    deferUntil += longTermOffer->GetDelayedHours();
                }
                auto permissions = server->GetDriveAPI()->GetUserPermissions(self.GetObjectId(), TUserPermissionsFeatures());
                if (!communicationTag->OnDeferUntil(*communicationDbTag, {}, deferUntil, *permissions, session, server)) {
                    return false;
                }
            }
        }
    }

    if (description->GetStartBillingTask()) {
        if (!StartBillingTask(server, session)) {
            return false;
        }
    }

    return true;
}

bool TLongTermOfferHolderTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterRemove(self, userId, server, session)) {
        return false;
    }

    const auto api = Yensured(server)->GetDriveAPI();
    auto offer = GetOffer();
    if (!offer) {
        session.SetErrorInfo("LongTermOfferHolderTag::OnAfterRemove", "null offer");
        return false;
    }
    TBillingTask billingTask;
    if (api && api->HasBillingManager()) {
        const auto& billingManager = api->GetBillingManager();
        auto optionalBillingTask = billingManager.GetActiveTasksManager().GetTask(offer->GetOfferId(), session);
        if (!optionalBillingTask) {
            return false;
        }
        billingTask = std::move(*optionalBillingTask);
    }
    if (billingTask) {
        const auto& billingManager = api->GetBillingManager();
        if (!billingManager.FinishingBillingTask(offer->GetOfferId(), session)) {
            return false;
        }
        auto billingSession = CreateOfferHolderSession(self, TOfferHolderSessionOptions{}, api->GetTagsManager().GetUserTags(), session);
        if (!billingSession) {
            session.AddErrorMessage("LongTermOfferHolderTag::OnAfterRemove", "cannot create billing session from " + self.GetTagId());
            return false;
        }

        auto permissions = api->GetUserPermissions(self.GetObjectId(), {});
        if (!permissions) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnAfterRemove", "cannot create permissions for " + self.GetObjectId());
            return false;
        }
        session.Committed().Subscribe([api, billingSession, permissions, server](const NThreading::TFuture<void>& w) {
            if (!w.HasValue()) {
                return;
            }
            auto closed = Yensured(api)->CloseSession(billingSession, *Yensured(permissions), *server);
            if (!closed) {
                NDrive::TEventLog::Log("CloseSessionError", NJson::TMapBuilder
                    ("session", billingSession ? billingSession->GetDebugInfo() : NJson::TJsonValue())
                    ("error", NJson::GetExceptionInfo(closed.GetError()))
                );
            }
        });
    }

    auto permissions = api->GetUserPermissions(userId, {});
    if (!permissions) {
        session.SetErrorInfo("LongTermOfferHolderTag::OnAfterRemove", "cannot create permissions for " + userId);
        return false;
    }
    if (!RemoveCommunicationTag(permissions, server, session)) {
        return false;
    }
    if (!RemoveSubtags(userId, server, session)) {
        return false;
    }

    return true;
}

bool TLongTermOfferHolderTag::OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    if (!TBase::OnBeforeEvolve(from, to, permissions, server, session, eContext)) {
        return false;
    }

    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "LongTermOfferHolderTag_OnBeforeEvolve")
        );
    }

    const auto api = Yensured(server)->GetDriveAPI();
    const auto& tagMeta = Yensured(api)->GetTagsManager().GetTagsMeta();
    const auto& deviceTagManager = Yensured(api)->GetTagsManager().GetDeviceTags();
    const auto& userTagManager = Yensured(api)->GetTagsManager().GetUserTags();

    auto d = tagMeta.GetDescriptionByName(to->GetName());
    auto description = std::dynamic_pointer_cast<const TDescription>(d);

    auto recipient = std::dynamic_pointer_cast<TLongTermOfferHolderTag>(to);
    if (recipient && recipient->IsPreparation()) {
        const auto& deviceTagManager = api->GetTagsManager().GetDeviceTags();
        const auto& targetCarId = recipient->GetCarId();
        if (!targetCarId) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "no CarId provided", EDriveSessionResult::IncorrectRequest);
            return false;
        }
        auto device = deviceTagManager.RestoreObject(targetCarId, session);
        if (!device) {
            return false;
        }
        for (auto&& deviceTag : device->GetTags()) {
            if (!deviceTag) {
                continue;
            }
            if (deviceTag->GetPerformer()) {
                session.SetErrorInfo(
                    "LongTermOfferHolderTag::OnBeforeEvolve",
                    TStringBuilder() << "car tag " << deviceTag->GetName() << " has performer"
                );
                return false;
            }
        }
        auto offer = std::dynamic_pointer_cast<IOffer>(Coalesce(recipient->GetOffer(), GetOffer()));
        if (!offer) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "offer is missing in tag " + from.GetTagId());
            return false;
        }
        offer->SetObjectId(targetCarId);
        recipient->SetOffer(offer);
    } else if (recipient && recipient->HasCarId()) {
        const auto& carId = recipient->GetCarId();
        if (!carId && !RemoveSubtags(permissions.GetUserId(), server, session)) {
            return false;
        }
        auto offer = std::dynamic_pointer_cast<IOffer>(Coalesce(recipient->GetOffer(), GetOffer()));
        if (offer) {
            offer->SetObjectId(carId);
        }
    }
    if (recipient && !recipient->GetOffer()) {
        recipient->SetOffer(GetOffer());
    }
    if (recipient && !recipient->HasCarId()) {
        recipient->SetCarId(CarId);
    }
    if (recipient) {
        recipient->CarTagIds = CarTagIds;
        recipient->UserTagIds = UserTagIds;
        recipient->SetCommunicationTagId(CommunicationTagId);
    }

    if (description && description->GetCheckDeposit()) {
        auto offer = GetOffer();
        if (!offer) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "null offer");
            return false;
        }
        const auto& billingManager = api->GetBillingManager();
        auto optionalBillingTask = billingManager.GetActiveTasksManager().GetTask(offer->GetOfferId(), session);
        if (!optionalBillingTask) {
            return false;
        }
        auto billingTask = std::move(*optionalBillingTask);
        if (billingTask && !billingTask.CheckDeposit()) {
            session.SetErrorInfo(
                "LongTermOfferHolderTag::OnBeforeEvolve",
                TStringBuilder() << "deposit status: " << billingTask.GetTaskStatus(),
                NDrive::MakeError("deposit_check_failed")
            );
            return false;
        }
    }

    if (!description) {
        return RemoveCommunicationTag(permissions.Self(), server, session);
    }

    for (auto&& tagName : description->GetCarTags()) {
        const auto& carId = recipient->GetCarId();
        if (!carId) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "no CarId provided");
            return false;
        }
        auto tag = tagMeta.CreateTag(tagName);
        if (!tag) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "cannot create tag " + tagName);
            return false;
        }
        if (auto deliveryTag = std::dynamic_pointer_cast<TLongTermCarDeliveryTag>(tag)) {
            auto offer = std::dynamic_pointer_cast<TLongTermOffer>(GetOffer());
            if (!offer) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "no LongTermOffer provided");
                return false;
            }
            if (!offer->HasDeliveryLocation()) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "no DeliveryLocation provided");
                return false;
            }
            deliveryTag->SetLatitude(offer->GetDeliveryLocationRef().Y);
            deliveryTag->SetLongitude(offer->GetDeliveryLocationRef().X);
            deliveryTag->SetOfferId(offer->GetOfferId());
            deliveryTag->SetUserId(from.GetObjectId());
        }
        auto added = deviceTagManager.AddTag(tag, permissions.GetUserId(), carId, server, session);
        if (!added) {
            return false;
        }
        for (auto&& i : *added) {
            recipient->CarTagIds.insert(i.GetTagId());
        }
    }
    for (auto&& tagName : description->GetUserTags()) {
        auto tag = tagMeta.CreateTag(tagName);
        if (!tag) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "cannot create user tag " + tagName);
            return false;
        }
        auto added = userTagManager.AddTag(tag, permissions.GetUserId(), from.GetObjectId(), server, session);
        if (!added) {
            return false;
        }
        for (auto&& i : *added) {
            recipient->UserTagIds.insert(i.GetTagId());
        }
    }
    if (const auto& tagName = description->GetChildSeatTag()) {
        const auto& carId = recipient->GetCarId();
        if (!carId) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "no CarId provided");
            return false;
        }
        auto offer = std::dynamic_pointer_cast<TLongTermOffer>(GetOffer());
        if (!offer) {
            session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "no LongTermOffer provided");
            return false;
        }
        if (offer->GetChildSeatDef(false)) {
            auto tag = tagMeta.CreateTag(tagName);
            if (!tag) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeEvolve", "cannot create tag " + tagName);
                return false;
            }
            auto added = deviceTagManager.AddTag(tag, permissions.GetUserId(), carId, server, session);
            if (!added) {
                return false;
            }
            for (auto&& i : *added) {
                recipient->CarTagIds.insert(i.GetTagId());
            }
        }
    }

    return true;
}

bool TLongTermOfferHolderTag::OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    if (!TBase::OnAfterEvolve(from, to, permissions, server, session, eContext)) {
        return false;
    }

    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "LongTermOfferHolderTag_OnAfterEvolve")
        );
    }

    const auto& database = Yensured(server)->GetDriveDatabase();
    const auto& tagMeta = database.GetTagsManager().GetTagsMeta();

    auto d = tagMeta.GetDescriptionByName(to->GetName());
    auto description = std::dynamic_pointer_cast<const TDescription>(d);
    if (!description) {
        session.SetErrorInfo("LongTermOfferHolderTag::OnAfterEvolve", "cannot get description");
        return false;
    }

    if (description->GetStartBillingTask()) {
        if (!StartBillingTask(server, session)) {
            return false;
        }
    }

    return true;
}

bool TLongTermOfferHolderTag::OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    Y_UNUSED(self);
    auto api = Yensured(server)->GetDriveAPI();
    auto target = std::dynamic_pointer_cast<TLongTermOfferHolderTag>(to);
    if (!target) {
        session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeUpdate", "cannot cast target tag to LongTermOfferHolderTag");
        return false;
    }
    if (!target->GetOffer()) {
        target->SetOffer(GetOffer());
    }
    target->SetCarId(CarId);
    target->CarTagIds = CarTagIds;
    target->UserTagIds = UserTagIds;
    target->SetCommunicationTagId(CommunicationTagId);

    auto offer = target->GetOffer();
    auto longTermOffer = std::dynamic_pointer_cast<TLongTermOffer>(offer);
    bool locationUpdated = false;
    if (longTermOffer) {
        if (target->HasSince()) {
            auto duration = longTermOffer->GetUntil() - longTermOffer->GetSince();
            auto availableSince = longTermOffer->GetAvailableSince();
            auto since = target->GetSinceRef();
            if (since < availableSince) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeUpdate", "since < available_since", NDrive::MakeError("available_since_constraint_violated"));
                session.SetCode(HTTP_BAD_REQUEST);
                return false;
            }
            auto availableUntil = longTermOffer->GetAvailableUntil();
            auto until = since + duration;
            if (until > availableUntil && availableUntil) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeUpdate", "until > available_until", NDrive::MakeError("available_until_constraint_violated"));
                session.SetCode(HTTP_BAD_REQUEST);
                return false;
            }
            longTermOffer->SetSince(since);
            longTermOffer->SetUntil(until);
        }
        if (target->HasDeliveryArea()) {
            longTermOffer->SetDeliveryArea(target->GetDeliveryAreaRef());
        }
        if (target->HasDeliveryLocation()) {
            auto action = Yensured(api)->GetRolesManager()->GetAction(longTermOffer->GetBehaviourConstructorId());
            if (!action) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeUpdate", "cannot get action " + longTermOffer->GetBehaviourConstructorId());
                return false;
            }
            auto longTermOfferBuilder = action->GetAs<TLongTermOfferBuilder>();
            if (!longTermOfferBuilder) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeUpdate", "cannot get long_term_offer_builder " + longTermOffer->GetBehaviourConstructorId());
                return false;
            }
            const auto& deliveryLocation = target->GetDeliveryLocationRef();
            if (!longTermOfferBuilder->CheckDeliveryLocation(deliveryLocation, *server)) {
                session.SetCode(HTTP_BAD_REQUEST);
                session.SetErrorInfo(
                    "TLongTermOfferHolderTag::OnBeforeUpdate",
                    TStringBuilder() << "cannot deliver to " << deliveryLocation,
                    NDrive::MakeError("bad_delivery_location")
                );
                return false;
            }
            longTermOffer->SetDeliveryLocation(target->GetDeliveryLocationRef());
            locationUpdated = true;
        }
        if (target->HasDeliveryLocationName()) {
            longTermOffer->SetDeliveryLocationName(target->GetDeliveryLocationNameRef());
        }
    }
    if (locationUpdated) {
        const auto& deviceTagManager = Yensured(api)->GetTagsManager().GetDeviceTags();
        for (auto&& tagId : CarTagIds) {
            auto tag = deviceTagManager.RestoreTag(tagId, session);
            if (!tag) {
                return false;
            }
            auto deliveryTag = tag->MutableTagAs<TLongTermCarDeliveryTag>();
            if (deliveryTag && deliveryTag->GetPerformer()) {
                session.SetErrorInfo("LongTermOfferHolderTag::OnBeforeUpdate", "delivery tag " + tag->GetTagId() + " is already performed", NDrive::MakeError("delivery_in_progress"));
                session.SetCode(HTTP_BAD_REQUEST);
                return false;
            }
            if (deliveryTag) {
                deliveryTag->SetLatitude(target->GetDeliveryLocationRef().Y);
                deliveryTag->SetLongitude(target->GetDeliveryLocationRef().X);
            }
            if (!deviceTagManager.UpdateTagData(*tag, userId, session)) {
                return false;
            }
        }
    }
    session.Committed().Subscribe([offer](const NThreading::TFuture<void>& c) {
        if (c.HasValue() && offer) {
            auto serializedOffer = offer->SerializeToProto();
            NDrive::TEventLog::Log("OfferUpdated", NJson::ToJson(NJson::Proto(serializedOffer)));
        }
    });
    return true;
}

bool TLongTermOfferHolderTag::RemoveCommunicationTag(TUserPermissionsConstPtr permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "LongTermOfferHolderTag_RemoveCommunicationTag")
            ("communication_tag_id", CommunicationTagId)
        );
    }

    if (!CommunicationTagId) {
        return true;
    }

    const auto api = Yensured(server)->GetDriveAPI();
    const auto& userTagManager = Yensured(api)->GetTagsManager().GetUserTags();
    auto optionalCommunicationTag = userTagManager.RestoreTag(CommunicationTagId, session);
    if (!optionalCommunicationTag) {
        return false;
    }
    auto communicationTag = std::move(*optionalCommunicationTag);
    if (!communicationTag) {
        return true;
    }
    auto deferrableTag = communicationTag.GetTagAs<IDeferrableTag>();
    if (!deferrableTag) {
        auto containerTag = communicationTag.GetTagAs<TUserContainerTag>();
        if (!containerTag) {
            session.SetErrorInfo("LongTermOfferHolderTag::RemoveCommunicationTag", "cannot cast " + communicationTag.GetTagId() + " as DeferrableTag and as UserContainerTag");
            return false;
        }
        auto optionalUndeferredTag = IContainerTag::UndeferContainer(communicationTag, permissions, server, userTagManager, session);
        if (!optionalUndeferredTag) {
            return false;
        }
        communicationTag = std::move(*optionalUndeferredTag);
        if (!communicationTag) {
            session.SetErrorInfo("LongTermOfferHolderTag::RemoveCommunicationTag", "cannot restore tag after undeferring");
            return false;
        }
        deferrableTag = communicationTag.GetTagAs<IDeferrableTag>();
    }
    if (!deferrableTag) {
        session.SetErrorInfo("LongTermOfferHolderTag::RemoveCommunicationTag", "cannot cast " + communicationTag.GetTagId() + " as DeferrableTag second time around");
        return false;
    }
    bool closed = deferrableTag->OnClose(communicationTag, {}, *permissions, session, server);
    if (!closed) {
        session.AddErrorMessage("LongTermOfferHolderTag::RemoveCommunicationTag", "OnClose failed");
        return false;
    }
    return true;
}

bool TLongTermOfferHolderTag::StartBillingTask(const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto api = server ? server->GetDriveAPI() : nullptr;
    if (api && api->HasBillingManager()) {
        const auto& billingManager = api->GetBillingManager();
        auto offer = std::dynamic_pointer_cast<IOffer>(GetOffer());
        if (!offer) {
            session.SetErrorInfo("LongTermOfferHolderTag::StartBillingTask", "null offer");
            return false;
        }
        auto optionalBillingTask = billingManager.GetActiveTasksManager().GetTask(offer->GetOfferId(), session);
        if (!optionalBillingTask) {
            return false;
        }
        auto billingTask = std::move(*optionalBillingTask);
        if (!billingTask) {
            if (!billingManager.CreateBillingTask(offer->GetUserId(), offer, session)) {
                return false;
            }
        }
        return true;
    } else {
        session.SetErrorInfo("LongTermOfferHolderTag::StartBillingTask", "BillingManager is missing");
        return false;
    }
}

NDrive::NProto::TOfferHolderTag TLongTermOfferHolderTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TOfferHolderTag result = TBase::DoSerializeSpecialDataToProto();
    if (CarId) {
        result.SetCarId(*CarId);
    }
    if (CommunicationTagId) {
        result.SetCommunicationTagId(CommunicationTagId);
    }
    for (auto&& id : CarTagIds) {
        result.AddCarTagId(id);
    }
    for (auto&& id : UserTagIds) {
        result.AddUserTagId(id);
    }
    return result;
}

bool TLongTermOfferHolderTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TOfferHolderTag& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (proto.HasCarId()) {
        CarId = proto.GetCarId();
    }
    if (proto.HasCommunicationTagId()) {
        CommunicationTagId = proto.GetCommunicationTagId();
    }
    for (auto&& id : proto.GetCarTagId()) {
        CarTagIds.insert(id);
    }
    for (auto&& id : proto.GetUserTagId()) {
        UserTagIds.insert(id);
    }
    return true;
}

bool TLongTermOfferHolderTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return NJson::TryFieldsFromJson(value, GetFields(), errors);
}

ITag::TFactory::TRegistrator<TLongTermCarDeliveryTag> TLongTermCarDeliveryTag::Registrator(TLongTermCarDeliveryTag::Type());
ITag::TFactory::TRegistrator<TLongTermOfferHolderTag> TLongTermOfferHolderTag::Registrator(TLongTermOfferHolderTag::Type());

TTagDescription::TFactory::TRegistrator<TLongTermCarDeliveryTag::TDescription> LongTermCarDeliveryTag(TLongTermCarDeliveryTag::Type());
TTagDescription::TFactory::TRegistrator<TLongTermOfferHolderTag::TDescription> LongTermOfferHolderTagDescriptionRegistrator(TLongTermOfferHolderTag::Type());
