#include "markers.h"

#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/tags/tags.h>

#include <rtline/util/json_processing.h>

void TStringFieldTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE(json, "info", Information);
}

bool TStringFieldTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (json.Has("info")) {
        Information = json["info"].GetStringRobust();
    }
    return TBase::DoSpecialDataFromJson(json, errors);
}

void TAlertTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    JWRITE(json, "status", ::ToString(Status));
    TJsonProcessor::WriteContainerArray(json, "states_history", StatesHistory);

    NJson::TJsonValue& times = json.InsertValue("states_times", NJson::JSON_ARRAY);
    for (auto&& ts : StatesTs) {
        times.AppendValue(ts.Seconds());
    }
}

bool TAlertTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    JREAD_FROM_STRING_OPT(json, "status", Status);
    return TBase::DoSpecialDataFromJson(json, errors);
}


NDrive::NProto::TObjectAlertInfoTag TAlertTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TObjectAlertInfoTag proto = TBase::DoSerializeSpecialDataToProto();
    if (Status != EAlertSatatus::Active) {
        proto.SetAlertState(::ToString(Status));
    }

    for (auto&& state : StatesHistory) {
        proto.AddStatesHistory(state);
    }
    for (auto&& ts : StatesTs) {
        proto.AddStatesTs(ts.Seconds());
    }
    return proto;
}

bool TAlertTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TObjectAlertInfoTag& proto) {
    if (!TryFromString(proto.GetAlertState(), Status)) {
        return false;
    }

    for (auto&& state : proto.GetStatesHistory()) {
        StatesHistory.push_back(state);
    }
    for (auto&& ts : proto.GetStatesTs()) {
        StatesTs.push_back(TInstant::Seconds(ts));
    }
    if (StatesHistory.size() > StatesTs.size()) {
        StatesTs.clear();
        for (ui32 i = 0; i < StatesHistory.size(); ++i) {
            StatesTs.push_back(TInstant::Zero());
        }
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TUniqueTag::TypeName = "unique_tag";
ITag::TFactory::TRegistrator<TUniqueTag> TUniqueTag::Registrator(TUniqueTag::TypeName);

const TString TUniqueTagRewrite::TypeName = "unique_tag_rewrite";
ITag::TFactory::TRegistrator<TUniqueTagRewrite> TUniqueTagRewrite::Registrator(TUniqueTagRewrite::TypeName);

const TString TStringFieldTag::TypeName = "string_field_tag";
ITag::TFactory::TRegistrator<TStringFieldTag> TStringFieldTag::Registrator(TStringFieldTag::TypeName);

const TString TCarAlertTag::TypeName = "car_alert_tag";
ITag::TFactory::TRegistrator<TCarAlertTag> TCarAlertTag::Registrator(TCarAlertTag::TypeName);

const TString TUserAlertTag::TypeName = "user_alert_tag";
ITag::TFactory::TRegistrator<TUserAlertTag> TUserAlertTag::Registrator(TUserAlertTag::TypeName);
