#pragma once

#include <drive/backend/data/common/serializable.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/util/types/messages_collector.h>

namespace NDrive::NProto {
    class TObjectAlertInfoTag;
}

class TUniqueTag: public IJsonSerializableTag {
private:
    using TBase = IJsonSerializableTag;
public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUniqueTag> Registrator;

public:
    using TBase::TBase;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }
};

class TUniqueTagRewrite: public IJsonSerializableTag {
private:
    using TBase = IJsonSerializableTag;

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TUniqueTagRewrite> Registrator;

public:
    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }
};

class TStringFieldTag: public TUniqueTagRewrite {
private:
    using TBase = TUniqueTagRewrite;
    R_FIELD(TString, Information);

public:
    static const TString TypeName;
    static TFactory::TRegistrator<TStringFieldTag> Registrator;

public:
    using TBase::TBase;

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSString>("info", "Информация");
        return result;
    }

protected:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/) override;
};


class TAlertTag : public ISerializableTag<NDrive::NProto::TObjectAlertInfoTag, true> {
public:
    enum class EAlertSatatus {
        Active /* "active" */,
        Mute /* "mute" */,
    };

private:
    using TBase = ISerializableTag<NDrive::NProto::TObjectAlertInfoTag, true>;
    R_FIELD(EAlertSatatus, Status, EAlertSatatus::Active);
    R_FIELD(TVector<TString>, StatesHistory);
    R_FIELD(TVector<TInstant>, StatesTs);

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        result.Add<TFSVariants>("status", "Статус").InitVariants<EAlertSatatus>();
        return result;
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::SkipIfExists;
    }

    void AddHistoryState(const TString& state, const TInstant ts) {
        StatesHistory.push_back(state);
        StatesTs.push_back(ts);
    }

protected:
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* /*errors*/) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const NDrive::NProto::TObjectAlertInfoTag& proto) override;
};

class TCarAlertTag : public TAlertTag {
private:
    using TBase = TAlertTag;
    static TFactory::TRegistrator<TCarAlertTag> Registrator;

public:
    static const TString TypeName;
    using TBase::TBase;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::Car };
    }
};

class TUserAlertTag : public TAlertTag {
private:
    using TBase = TAlertTag;
    static TFactory::TRegistrator<TUserAlertTag> Registrator;

public:
    static const TString TypeName;
    using TBase::TBase;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::User };
    }
};
