#include "model.h"

#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/scheme/scheme.h>

IEntityTagsManager::TExpectedTag TModelTag::Get(const TString& objectId, const NDrive::IServer& server) {
    auto driveApi = server.GetDriveAPI();
    if (!driveApi) {
        return MakeUnexpected(IEntityTagsManager::ETagError::LogicError);
    }
    auto taggedObject = driveApi->GetTagsManager().GetDeviceTags().GetObject(objectId);
    if (!taggedObject) {
        return MakeUnexpected(IEntityTagsManager::ETagError::Absent);
    }
    for (auto&& tag : taggedObject->GetTags()) {
        if (!tag) {
            continue;
        }
        if (driveApi->GetTagsManager().GetTagsMeta().GetTagTypeByName(tag->GetName()) == TModelTag::Type()) {
            return std::move(tag);
        }
    }
    return MakeUnexpected(IEntityTagsManager::ETagError::Absent);
}

const NDrive::TLocationTags& TModelTag::GetAllowRidingTags(const TConstDBTag& modelTag, const NDrive::IServer& server) {
    auto description = GetDescription(modelTag, server);
    if (description) {
        return description->GetAllowRidingTags();
    } else {
        return NDrive::DefaultAllowRidingLocationTags;
    }
}

const NDrive::TLocationTags& TModelTag::GetAllowDropTags(const TConstDBTag& modelTag, const NDrive::IServer& server) {
    auto description = GetDescription(modelTag, server);
    if (description) {
        return description->GetAllowDropTags();
    } else {
        return NDrive::DefaultAllowDropLocationTags;
    }
}

const NDrive::TLocationTags& TModelTag::GetPoiLocationTags(const TConstDBTag& modelTag, const NDrive::IServer& server) {
    auto description = GetDescription(modelTag, server);
    if (description) {
        return description->GetPoiTags();
    } else {
        return NDrive::DefaultPoiLocationTags;
    }
}

TAtomicSharedPtr<const TModelTag::TDescription> TModelTag::GetDescription(const TConstDBTag& modelTag, const NDrive::IServer& server) {
    if (!modelTag) {
        return nullptr;
    }
    auto driveApi = server.GetDriveAPI();
    auto description = Yensured(driveApi)->GetTagsManager().GetTagsMeta().GetDescriptionByName(modelTag->GetName());
    return std::dynamic_pointer_cast<const TDescription>(description);
}

TMaybe<TSet<TString>> TModelTag::GetInsuranceTypes(const TConstDBTag& modelTag, const NDrive::IServer& server) {
    auto description = GetDescription(modelTag, server);
    if (!description) {
        return {};
    }
    const auto& insuranceTypes = description->GetInsuranceTypes();
    if (insuranceTypes.empty()) {
        return {};
    }
    return insuranceTypes;
}

NDrive::TScheme TModelTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSArray>("allow_riding_tags", "Location tags that allow riding").SetElement<TFSString>();
    result.Add<TFSArray>("allow_drop_tags", "Location tags that allow end of lease").SetElement<TFSString>();
    result.Add<TFSArray>("force_allow_drop_tags", "Location tags that allow force end of lease").SetElement<TFSString>();
    result.Add<TFSArray>("deny_drop_tags", "Location tags that block end of lease").SetElement<TFSString>();
    result.Add<TFSArray>("poi_tags", "Location tags enable certain POI").SetElement<TFSString>();
    result.Add<TFSNumeric>("first_mile_distance_threshold", "Show FirstMile block starting from this distance");
    result.Add<TFSVariants>("insurance_types", "Allowed insurance types").SetMultiSelect(true).SetVariants({
        "standart",
        "full",
    });
    return result;
}

NJson::TJsonValue TModelTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::InsertField(result, "insurance_types", InsuranceTypes);
    for (auto&& tag : AllowRidingTags) {
        result["allow_riding_tags"].AppendValue(tag);
    }
    for (auto&& tag : ForceAllowDropTags) {
        result["force_allow_drop_tags"].AppendValue(tag);
    }
    for (auto&& tag : AllowDropTags) {
        result["allow_drop_tags"].AppendValue(tag);
    }
    for (auto&& tag : DenyDropTags) {
        result["deny_drop_tags"].AppendValue(tag);
    }
    for (auto&& tag : PoiTags) {
        result["poi_tags"].AppendValue(tag);
    }
    return result;
}

bool TModelTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    for (auto&& tag : value["allow_riding_tags"].GetArray()) {
        AllowRidingTags.insert(tag.GetStringRobust());
    }
    for (auto&& tag : value["force_allow_drop_tags"].GetArray()) {
        ForceAllowDropTags.insert(tag.GetStringRobust());
    }
    for (auto&& tag : value["allow_drop_tags"].GetArray()) {
        AllowDropTags.insert(tag.GetStringRobust());
    }
    for (auto&& tag : value["deny_drop_tags"].GetArray()) {
        DenyDropTags.insert(tag.GetStringRobust());
    }
    for (auto&& tag : value["poi_tags"].GetArray()) {
        PoiTags.insert(tag.GetStringRobust());
    }
    return
        NJson::ParseField(value["insurance_types"], InsuranceTypes);
}

ITag::TFactory::TRegistrator<TModelTag> ModelTagRegistrator(TModelTag::Type());
TTagDescription::TFactory::TRegistrator<TModelTag::TDescription> ModelTagDescriptionRegistrator(TModelTag::Type());
