#pragma once

#include "communication_channel_tags.h"

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/proto/tags.pb.h>

#include <drive/backend/abstract/notifier.h>

#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/util/types/expected.h>

class INotificationsTagDescription: public TTagDescription {
private:
    using TBase = TTagDescription;

private:
    R_OPTIONAL(TTimeRestrictionsPool<TTimeRestriction>, ActivityInterval);
    R_FIELD(bool, Enabled, true);
    R_FIELD(bool, ExternalEnabled, false);
    R_FIELD(bool, UseOrigin, false);
    R_FIELD(bool, FetchExternalUserIdFromHistorySessions, false);
    R_FIELD(TCommunicationChannelSettings, CommunicationChannelSettings);

public:
    using TBase::TBase;

    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    bool DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) override;
    NJson::TJsonValue DoSerializeMetaToJson() const override;
};

class INotificationTag {
private:
    R_FIELD(TString, ExternalUserId);
    R_READONLY(TInstant, SendInstant, TInstant::Zero());
    R_READONLY(TInstant, Deadline, TInstant::Max());

public:
    virtual TExpected<NDrive::INotifier::TMessage, TString> BuildMessage(ELocalization locale, const NDrive::IServer& server, const INotificationsTagDescription& desc, const TUserContacts& contacts) const = 0;

    void FillScheme(NDrive::TScheme& scheme) const;
    void SerializeToJson(NJson::TJsonValue& json) const;
    bool DeserializeFromJson(const NJson::TJsonValue& json);

    template <class TProto>
    void SerializeToProto(TProto& proto) const {
        if (ExternalUserId) {
            proto.SetExternalUserId(ExternalUserId);
        }
        if (SendInstant != TInstant::Zero()) {
            proto.SetSendInstant(SendInstant.Seconds());
        }
        if (Deadline != TInstant::Max()) {
            proto.SetDeadline(Deadline.Seconds());
        }
    }

    template <class TProto>
    bool DeserializeFromProto(const TProto& proto) {
        ExternalUserId = proto.GetExternalUserId();
        SendInstant = TInstant::Seconds(proto.GetSendInstant());
        if (proto.HasDeadline()) {
            Deadline = TInstant::Seconds(proto.GetDeadline());
        }
        return true;
    }
};

template <class ProtoType>
class INotificationTagBase: public ISerializableTag<ProtoType>, public INotificationTag {
private:
    using TBase = ISerializableTag<ProtoType>;

public:
    using TBase::TBase;

    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override {
        NDrive::TScheme result = TBase::GetScheme(server);
        INotificationTag::FillScheme(result);
        return result;
    }

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override {
        TBase::SerializeSpecialDataToJson(json);
        INotificationTag::SerializeToJson(json);
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
        if (!INotificationTag::DeserializeFromJson(json)) {
            return false;
        }
        return TBase::DoSpecialDataFromJson(json, errors);
    }

    ProtoType DoSerializeSpecialDataToProto() const override {
        ProtoType proto = TBase::DoSerializeSpecialDataToProto();
        INotificationTag::SerializeToProto<ProtoType>(proto);
        return proto;
    }

    bool DoDeserializeSpecialDataFromProto(const ProtoType& proto) override {
        if (!INotificationTag::DeserializeFromProto<ProtoType>(proto)) {
            return false;
        }
        return TBase::DoDeserializeSpecialDataFromProto(proto);
    }
};
