#include "notifications_tags.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/replace.h>
#include <rtline/util/network/neh.h>

namespace {
    TExpected<TVector<TString>, TString> DownloadAttachments(
            const TUserMailTag::TAttachments& attachments) {
        NNeh::THttpClient client;
        TVector<NThreading::TFuture<NUtil::THttpReply>> futures;
        NSimpleMeta::TConfig config;
        config.SetGlobalTimeout(TDuration::Minutes(1))
                .SetConnectTimeout(TDuration::MilliSeconds(100));
        for (auto&& a : attachments) {
            TString host, path;
            SplitUrlToHostAndPath(a.Url, host, path);
            client.RegisterSource(a.Url, config, "MailAttachmentDownload");

            NNeh::THttpRequest request;
            request.SetUri(path);
            futures.push_back(client.SendAsync(request));
        }

        if (!NThreading::WaitExceptionOrAll(futures).Wait(TDuration::Minutes(2))) {
            return MakeUnexpected<TString>("Timeout while downloading attachment");
        }

        TVector<TString> results;
        for (auto&& f : futures) {
            if (f.HasException()) {
                return MakeUnexpected(NThreading::GetExceptionMessage(f));
            }

            if (!f.GetValue().IsSuccessReply()) {
                return MakeUnexpected(f.GetValue().GetDebugReply());
            }
            results.emplace_back(Base64Encode(f.GetValue().Content()));
        }
        return results;
    }

    TString GetDefaultAttachmentName(size_t index, ELocalization locale, const ILocalization& localization) {
        return TStringBuilder() << localization.GetLocalString(locale, "mail.attachment", "Attachment")
                                << "_" << index;
    }

    void SetFeaturesScheme(NDrive::TScheme& scheme, const TString& key, const TString& message) {
        auto& features = scheme.Add<TFSArray>(key, message);
        features.SetRequired(true);
        NDrive::TScheme& fields = features.SetElement<NDrive::TScheme>();
        fields.Add<TFSString>("key", "Идентификатор фичи").SetRequired(true);
        fields.Add<TFSString>("value", "Значение").SetRequired(true);
    }
}  // namespace

NDrive::TScheme TUserMailTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSString>("template_id", "Идентификатор шаблона").SetRequired(true);
    auto& templates = scheme.Add<TFSArray>("template_args", "Аргументы шаблона");
    templates.SetRequired(true);
    NDrive::TScheme& fields = templates.SetElement<NDrive::TScheme>();
    fields.Add<TFSString>("id", "ID аргумента шаблона").SetRequired(true);
    fields.Add<TFSString>("name", "Имя для отображени в форме").SetRequired(true);
    fields.Add<TFSString>("default", "Значение по умолчанию");
    return scheme;
}

bool TUserMailTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    JREAD_STRING_OPT(jsonMeta, "template_id", TemplateId);
    if (!jsonMeta.Has("template_args") || !jsonMeta["template_args"].IsArray()) {
        return false;
    }
    for (auto&& item : jsonMeta["template_args"].GetArraySafe()) {
        if (!item.Has("id") || !item.Has("name"))
            return false;

        TemplateArgs.push_back(TField(
                item["id"].GetString(),
                item["name"].GetString(),
                item["default"].GetString()));
    }
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}

NJson::TJsonValue TUserMailTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    JWRITE(jsonMeta, "template_id", TemplateId);
    NJson::TJsonValue& templateArgsJson = jsonMeta.InsertValue(
            "template_args", NJson::JSON_ARRAY);
    for (auto&& i : TemplateArgs) {
        NJson::TJsonValue& pair = templateArgsJson.AppendValue(NJson::JSON_MAP);
        pair["name"] = i.Name;
        pair["default"] = i.DefaultValue;
        pair["id"] = i.Id;
    }
    return jsonMeta;
}

const TString TUserMailTag::TypeName = "user_mail_notification_tag";
ITag::TFactory::TRegistrator<TUserMailTag> TUserMailTag::Registrator(TUserMailTag::TypeName);
TUserMailTag::TDescription::TFactory::TRegistrator<TUserMailTag::TDescription> TUserMailTag::TDescription::Registrator(TUserMailTag::TypeName);

NJson::TJsonValue TUserMailTag::TAttachInfo::SerializeToJson() const {
    NJson::TJsonValue res(NJson::JSON_MAP);
    res["url"] = Url;
    res["filename"] = Filename;
    res["mime_type"] = MimeType;
    return res;
}

bool TUserMailTag::TAttachInfo::DeserializeFromJson(const NJson::TJsonValue& json) {
    if (!json.IsMap()) {
        return false;
    }

    JREAD_STRING(json, "url", Url);
    JREAD_STRING(json, "filename", Filename);
    JREAD_STRING(json, "mime_type", MimeType);
    return true;
}

NDrive::TScheme TUserMailTag::TAttachInfo::GetScheme() {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("url", "Ссылка").SetRequired(true);
    scheme.Add<TFSString>("mime_type", "Mime-тип").SetDefault("application/pdf").SetRequired(true);
    scheme.Add<TFSString>("filename", "Имя вложения");
    return scheme;
}

NDrive::TScheme TUserMailTag::GetScheme(const NDrive::IServer* server) const {
    auto scheme = TBase::GetScheme(server);
    scheme.Add<TFSString>("session_id", "Идентификатор сессии");

    const TDriveAPI* driveApi = server->GetDriveAPI();
    CHECK_WITH_LOG(!!driveApi);
    auto description = driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    if (!description) {
        ERROR_LOG << "Can't read description for " << GetName() << Endl;
        return scheme;
    }

    auto descriptionImpl = dynamic_cast<const TUserMailTag::TDescription*>(description.Get());
    if (!descriptionImpl) {
        ERROR_LOG << "Wrong description type for " << GetName() << Endl;
        return scheme;
    }

    auto& templates = scheme.Add<TFSStructure>("template_args", "Аргументы шаблона");
    templates.SetRequired(true);
    NDrive::TScheme& args = templates.SetStructure<NDrive::TScheme>();
    for (auto&& field : descriptionImpl->GetTemplateArgs()) {
        args.Add<TFSString>(field.Id, field.Name).SetDefault(field.DefaultValue);
    }
    scheme.Add<TFSArray>("attachments", "Вложения").SetElement(TAttachInfo::GetScheme());

    return scheme;
}

EUniquePolicy TUserMailTag::GetUniquePolicy() const {
    return EUniquePolicy::NoUnique;
}

TSet<NEntityTagsManager::EEntityType> TUserMailTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::User };
}

void TUserMailTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    json.InsertValue("session_id", SessionId);

    NJson::TJsonValue& templateArgs = json.InsertValue("template_args", NJson::JSON_MAP);
    for (auto&& arg : TemplateArgs) {
        templateArgs.InsertValue(arg.first, arg.second);
    }

    NJson::TJsonValue& attachments = json.InsertValue("attachments", NJson::JSON_ARRAY);
    for (auto&& arg : Attachments) {
        attachments.AppendValue(arg.SerializeToJson());
    }
}

bool TUserMailTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!NJson::ParseField(json, "template_args", NJson::NPrivate::TDictionary(TemplateArgs), true) || !NJson::ParseField(json, "session_id", SessionId, false)) {
        return false;
    }

    if (json.Has("attachments") && json["attachments"].IsArray()) {
        for (auto&& item : json["attachments"].GetArraySafe()) {
            TAttachInfo info;
            if (!info.DeserializeFromJson(item)) {
                return false;
            }
            Attachments.push_back(info);
        }
    }
    return TBase::DoSpecialDataFromJson(json, errors);
}

TUserMailTag::TProto TUserMailTag::DoSerializeSpecialDataToProto() const {
    TUserMailTag::TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetSessionId(SessionId);
    for (auto&& arg : TemplateArgs) {
        TProtoField* argProto = proto.AddTemplateArgs();
        argProto->SetKey(arg.first);
        argProto->SetValue(arg.second);
    }

    for (auto&& a : Attachments) {
        TProtoAttachment* protoAttach = proto.AddAttachments();
        protoAttach->SetUrl(a.Url);
        protoAttach->SetFilename(a.Filename);
        protoAttach->SetMimeType(a.MimeType);
    }
    return proto;
}

bool TUserMailTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    SessionId = proto.GetSessionId();
    for (ui32 i = 0; i < proto.TemplateArgsSize(); ++i) {
        TemplateArgs[proto.GetTemplateArgs(i).GetKey()] = proto.GetTemplateArgs(i).GetValue();
    }
    for (auto&& a : proto.GetAttachments()) {
        Attachments.push_back({a.GetUrl(), a.GetFilename(), a.GetMimeType()});
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}


bool TUserMailTag::OnBeforeAdd(const TString& objectId, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    const TDriveAPI* driveApi = server->GetDriveAPI();
    if (!driveApi) {
        session.SetErrorInfo(GetName(), "No api configured", EDriveSessionResult::InternalError);
        return false;
    }

    auto description = driveApi->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    if (!description) {
        session.SetErrorInfo(GetName(), "Unknown type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    const TDescription* descriptionImpl = dynamic_cast<const TDescription*>(description.Get());
    if (!descriptionImpl) {
        session.SetErrorInfo(GetName(), "Incorrect description type", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    TObjectEvent<TFullCompiledRiding> ride;
    TCarsDB::TFetchResult carData;
    const TDriveCarInfo* carPtr = nullptr;
    TModelsDB::TFetchResult modelData;
    const TDriveModelData* modelPtr = nullptr;

    if (SessionId) {
        auto ydbTx = driveApi->BuildYdbTx<NSQL::ReadOnly>("user_mail_tag", server);
        auto optionalSessions = driveApi->GetMinimalCompiledRides().Get<TFullCompiledRiding>({ SessionId }, session, ydbTx);
        if (!optionalSessions) {
            return false;
        }
        if (optionalSessions->size() != 1) {
            session.SetErrorInfo(GetName(), TStringBuilder() << "undefined full compiled riding " << SessionId << ": " << optionalSessions->size());
            return false;
        }
        ride = std::move(optionalSessions->front());
        carData = driveApi->GetCarsData()->FetchInfo(ride.GetObjectId(), session);
        carPtr = carData.GetResultPtr(ride.GetObjectId());

        if (carPtr) {
            modelData = driveApi->GetModelsData()->FetchInfo(carPtr->GetModel(), session);
            modelPtr = modelData.GetResultPtr(carPtr->GetModel());
        }
    }
    TDuration shift = TDuration::Hours(3);
    for (auto&& field : descriptionImpl->GetTemplateArgs()) {
        if (TemplateArgs.contains(field.Id)) {
            continue;
        }
        if (field.Id == "user_name") {
            auto gUsers = driveApi->GetUsersData()->FetchInfo(objectId, session);
            const auto* userData = gUsers.GetResultPtr(objectId);
            TemplateArgs["user_name"] = userData ? userData->GetFirstName() : "";
        }
        if (field.Id == "car_model") {
            TString model;
            if (modelPtr) {
                model = modelPtr->GetName();
            } else if (carPtr) {
                model = carPtr->GetModel();
            }
            TemplateArgs["car_model"] = std::move(model);
        }
        if (field.Id == "car_number") {
            TemplateArgs["car_number"] = carPtr ? carPtr->GetNumber() : "";
        }
        if (field.Id == "add_car_image" && IsTrue(field.DefaultValue) && modelPtr) {
            TAttachInfo attach;
            attach.Url = modelPtr->GetImageSmallUrl();
            attach.Filename = "model_image_small.png";
            attach.MimeType = "image/png";
            Attachments.emplace_back(std::move(attach));
        }
        if (field.Id == "start_time") {
            TemplateArgs["start_time"] = server->GetLocalization()->FormatInstantWithYear(ELocalization::Rus, ride.GetStartInstant() + shift);
        }
        if (field.Id == "finish_time") {
            TemplateArgs["finish_time"] = server->GetLocalization()->FormatInstantWithYear(ELocalization::Rus, ride.GetFinishInstant() + shift);
        }
        if (field.Id == "bill_report") {
            TStringBuilder strBuilder;
            if (ride.HasBill() && ride.GetBillUnsafe().GetRecords().size()) {
                ui32 discounts = 0;
                for (auto&& i : ride.GetBillUnsafe().GetRecords()) {
                    if (i.GetType() == TBillRecord::DiscountType) {
                        ++discounts;
                    }
                }
                bool insertDiscountSection = true;
                bool insertBillingSection = true;
                for (auto&& i : ride.GetBillUnsafe().GetRecords()) {
                    if (TBillRecord::IsAreaFees(i.GetType())) {
                        strBuilder << Endl;
                    }
                    if (insertDiscountSection && i.GetType() == TBillRecord::DiscountType) {
                        insertDiscountSection = false;
                        strBuilder << (discounts > 1 ? NDrive::TLocalization::DiscountsBillSection() : NDrive::TLocalization::DiscountBillSection()) << Endl;
                    }
                    if (insertBillingSection && i.GetType().StartsWith(TBillRecord::BillingTypePrefix)) {
                        insertBillingSection = false;
                        strBuilder << NDrive::TLocalization::BillingBillSection() << Endl;
                    }
                    strBuilder << i.GetTitle() << (i.GetDetails() ? " (" + i.GetDetails() + ")" : "") << " : " << i.GetCost() << Endl;
                }
            }
            TemplateArgs["bill_report"] = strBuilder;
        }
        if (field.Id == "add_checks" && IsTrue(field.DefaultValue) && SessionId) {
            TCachedPayments payments;
            if (!server->GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetPayments(payments, SessionId, session)) {
                return false;
            }
            for (auto&& payment : payments.GetTimeline()) {
                if (payment.GetPaymentType() != NDrive::NBilling::EAccount::Trust) {
                    continue;
                }
                if (TBillingGlobals::FailedStatuses.contains(payment.GetStatus())) {
                    continue;
                }
                if (payment.GetBillingType() == EBillingType::Deposit && payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Canceled) {
                    continue;
                }
                TAttachInfo attach;
                attach.Url = "https://trust.yandex.ru/checks/" + payment.GetPaymentId() + "/receipts/" + payment.GetPaymentId() + "?mode=pdf";
                attach.Filename = payment.GetPaymentId() + ".pdf";
                attach.MimeType = "application/pdf";
                Attachments.emplace_back(std::move(attach));
            }
        }
    }
    return true;
}

TExpected<NDrive::INotifier::TMessage, TString> TUserMailTag::BuildMessage(ELocalization locale, const NDrive::IServer& server, const INotificationsTagDescription& desc, const TUserContacts& contacts) const {
    auto mailDesc = dynamic_cast<const TUserMailTag::TDescription*>(&desc);
    if (!mailDesc) {
        return MakeUnexpected<TString>("Wrong description tag type");
    }
    if (!contacts.GetEmail()) {
        return MakeUnexpected<TString>("Incorrect user email");
    }

    NThreading::TFuture<NDrive::TGeocoder::TResponse> startLocation;
    NThreading::TFuture<NDrive::TGeocoder::TResponse> finishLocation;
    bool needGeocodedStart = false;
    bool needGeocodedFinish = false;
    if (SessionId) {
        for (auto&& field : mailDesc->GetTemplateArgs()) {
            if (field.Id == "geocoded_start") {
                needGeocodedStart = true;
            }
            if (field.Id == "geocoded_finish") {
                needGeocodedFinish = true;
            }
        }

        NDrive::TEntitySession session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("user_mail_tag", &server);
        auto optionalSessions = server.GetDriveAPI()->GetMinimalCompiledRides().Get<TFullCompiledRiding>({ SessionId }, session, ydbTx);
        if (!optionalSessions || optionalSessions->size() != 1) {
            return MakeUnexpected<TString>("undefined full compiled riding " + SessionId);
        }
        TObjectEvent<TFullCompiledRiding> ride = std::move(optionalSessions->front());

        auto geocoder = server.GetDriveAPI()->HasGeocoderClient() ? &server.GetDriveAPI()->GetGeocoderClient() : nullptr;
        auto language = enum_cast<ELanguage>(locale);
        if (geocoder && ride.HasSnapshotsDiff()) {
            if (needGeocodedStart && ride.GetSnapshotsDiffUnsafe().HasStart()) {
                startLocation = geocoder->Decode(ride.GetSnapshotsDiffUnsafe().GetStartUnsafe().GetCoord(), language);
            }
            if (needGeocodedFinish && ride.GetSnapshotsDiffUnsafe().HasLast()) {
                finishLocation = geocoder->Decode(ride.GetSnapshotsDiffUnsafe().GetLastUnsafe().GetCoord(), language);
            }
        }
    }

    NJson::TJsonValue templatesJson(NJson::JSON_MAP);
    for (auto&& field : GetTemplateArgs()) {
        templatesJson[field.first] = field.second;
    }
    TDuration timeout = server.GetSettings().GetValue<TDuration>("tags.geocoder_timeout_for_user_mail_tag").GetOrElse(TDuration::Zero());
    if (needGeocodedStart && !templatesJson.Has("geocoded_start") && startLocation.Initialized() && startLocation.Wait(timeout) && startLocation.HasValue()) {
        templatesJson["geocoded_start"] = startLocation.GetValue().Title;
    }
    if (needGeocodedFinish && !templatesJson.Has("geocoded_finish") && finishLocation.Initialized() && finishLocation.Wait(timeout) && finishLocation.HasValue()) {
        templatesJson["geocoded_finish"] = finishLocation.GetValue().Title;
    }

    auto res = NDrive::INotifier::TMessage(mailDesc->GetTemplateId(), templatesJson.GetStringRobust());
    if (Attachments) {
        NJson::TJsonValue additionalInfoJson = NJson::JSON_MAP;
        NJson::TJsonValue& attachmentsJson = additionalInfoJson.InsertValue("attachments", NJson::JSON_ARRAY);

        auto base64Attachments = DownloadAttachments(Attachments);
        if (!base64Attachments) {
            return MakeUnexpected(base64Attachments.GetError());
        }

        for (size_t i = 0; i < Attachments.size(); ++i) {
            const auto& attach = Attachments[i];
            NJson::TJsonValue attachJson = NJson::JSON_MAP;
            attachJson["filename"] = attach.Filename
                ? attach.Filename
                : GetDefaultAttachmentName(i + 1, locale, *server.GetLocalization());
            attachJson["mime_type"] = attach.MimeType;
            attachJson["data"] = std::move(base64Attachments.GetValue()[i]);
            attachmentsJson.AppendValue(attachJson);
        }
        res.SetAdditionalInfo(std::move(additionalInfoJson));
    }

    TStringBuilder messageMetaData;
    messageMetaData << "header:" << res.GetHeader() << "body:" << res.GetBody();
    for (size_t i = 0; i < Attachments.size(); ++i) {
        const auto& attach = Attachments[i];
        messageMetaData << "attach" << i << ":url:" << attach.Url << "mimeType:" << attach.MimeType
                        << "filename:" << attach.Filename;
    }

    res.SetMessageHash(MD5::CalcRaw(messageMetaData));
    return res;
}

const TString TSupportAITag::TypeName = "user_support_ai";
ITag::TFactory::TRegistrator<TSupportAITag> TSupportAITag::Registrator(TSupportAITag::TypeName);
TSupportAITag::TDescription::TFactory::TRegistrator<TSupportAITag::TDescription> TSupportAITag::TDescription::Registrator(TSupportAITag::TypeName);

NDrive::TScheme TSupportAITag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TDescriptionBase::GetScheme(server);
    SetFeaturesScheme(result, "default_features", "Фичи звонка по умолчанию");
    result.Add<TFSBoolean>("save_result", "Хранить данные звонка").SetDefault(SaveResult);
    return result;
}

EUniquePolicy TSupportAITag::GetUniquePolicy() const {
    return EUniquePolicy::NoUnique;
}

TSet<NEntityTagsManager::EEntityType> TSupportAITag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::User };
}

NJson::TJsonValue TSupportAITag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TDescriptionBase::DoSerializeMetaToJson();
    if (!DefaultFeatures.empty()) {
        NJson::InsertField(result, "default_features", NJson::KeyValue(DefaultFeatures));
    }
    NJson::InsertNonNull(result, "save_result", SaveResult);
    return result;
}

bool TSupportAITag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    return TDescriptionBase::DoDeserializeMetaFromJson(json)
        && NJson::ParseField(json["save_result"], SaveResult)
        && NJson::ParseField(json["default_features"], NJson::KeyValue(DefaultFeatures));
}

NDrive::TScheme TSupportAITag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    SetFeaturesScheme(result, "features", "Фичи звонка");
    return result;
}

TSupportAITag::TProto TSupportAITag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    for (auto&& [key, value] : Features) {
        NDrive::NProto::TTagField* fieldProto = proto.AddFeatures();
        fieldProto->SetKey(key);
        fieldProto->SetValue(value);
    }
    return proto;
}

bool TSupportAITag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    for (auto&& feature : proto.GetFeatures()) {
        Features.emplace(feature.GetKey(), feature.GetValue());
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

void TSupportAITag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    if (!Features.empty()) {
        NJson::InsertField(json, "override_features", NJson::KeyValue(Features));
    }
}

bool TSupportAITag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return TBase::DoSpecialDataFromJson(json, errors)
        && NJson::ParseField(json["features"], NJson::KeyValue(Features));
}

TExpected<NDrive::INotifier::TMessage, TString> TSupportAITag::BuildMessage(ELocalization /*locale*/, const NDrive::IServer& /*server*/, const INotificationsTagDescription& desc, const TUserContacts& /*contacts*/) const {
    auto messageDesc = dynamic_cast<const TSupportAITag::TDescription*>(&desc);
    if (!messageDesc) {
        return MakeUnexpected<TString>("Wrong description tag type");
    }
    NDrive::TNotifierMessage message("");
    message.SetAdditionalInfo(NJson::ToJson(NJson::Dictionary(GetFeatures().empty() ? messageDesc->GetDefaultFeatures() : GetFeatures())));
    message.SetName(GetName());
    return message;
}

const TString TUserPushTag::TypeName = "user_push";
ITag::TFactory::TRegistrator<TUserPushTag> TUserPushTag::Registrator(TUserPushTag::TypeName);
TUserPushTag::TDescription::TFactory::TRegistrator<TUserPushTag::TDescription>
        TUserPushTag::DescriptionRegistrator(TUserPushTag::TypeName);

const TString TServiceAppPushTag::TypeName = "user_service_app_push";
ITag::TFactory::TRegistrator<TServiceAppPushTag> TServiceAppPushTag::Registrator(TServiceAppPushTag::TypeName);
TServiceAppPushTag::TDescription::TFactory::TRegistrator<TServiceAppPushTag::TDescription> TServiceAppPushTag::DescriptionRegistrator(TServiceAppPushTag::TypeName);

const TString TUserSmsTag::TypeName = "user_sms";
ITag::TFactory::TRegistrator<TUserSmsTag> TUserSmsTag::Registrator(TUserSmsTag::TypeName);
TUserSmsTag::TDescription::TFactory::TRegistrator<TUserSmsTag::TDescription>
        TUserSmsTag::DescriptionRegistrator(TUserSmsTag::TypeName);

const TString TUserWorldSmsTag::TypeName = "user_world_sms";
ITag::TFactory::TRegistrator<TUserWorldSmsTag> TUserWorldSmsTag::Registrator(TUserWorldSmsTag::TypeName);
TUserWorldSmsTag::TDescription::TFactory::TRegistrator<TUserWorldSmsTag::TDescription> TUserWorldSmsTag::DescriptionRegistrator(TUserWorldSmsTag::TypeName);

NDrive::TScheme TUserMessageTagBase::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSJson>("additional_info", "Additional message metadata");
    result.Add<TFSString>("message_text", "Message text");
    result.Add<TFSString>("title", "Message title");
    return result;
}

void TUserMessageTagBase::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    if (!Macros.empty()) {
        json["macros"] = NJson::ToJson(Macros);
    }
    if (MessageText) {
        json["message_text"] = MessageText;
    }
    if (AdditionalInfo.IsDefined()) {
        json["additional_info"] = AdditionalInfo;
    }
    if (Title) {
        json["title"] = Title;
    }
}

bool TUserMessageTagBase::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(json, errors)) {
        return false;
    }
    AdditionalInfo = json["additional_info"];
    return
        NJson::ParseField(json["macros"], Macros) &&
        NJson::ParseField(json["message_text"], MessageText) &&
        NJson::ParseField(json["title"], Title);
}

ISerializableTag<NDrive::NProto::TUserMessageTagData>::TProto TUserMessageTagBase::DoSerializeSpecialDataToProto() const {
    auto result = TBase::DoSerializeSpecialDataToProto();
    result.SetMessageText(MessageText);
    if (AdditionalInfo.IsDefined()) {
        result.SetAdditionalInfo(AdditionalInfo.GetStringRobust());
    }
    if (!Macros.empty()) {
        result.MutableMacros()->insert(Macros.begin(), Macros.end());
    }
    if (Title) {
        result.SetTitle(Title);
    }
    return result;
}

bool TUserMessageTagBase::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (const TString& additionalInfo = proto.GetAdditionalInfo()) {
        if (!NJson::ReadJsonFastTree(additionalInfo, &AdditionalInfo)) {
            return false;
        }
    }
    {
        Macros = {
            proto.GetMacros().begin(),
            proto.GetMacros().end()
        };
    }
    MessageText = proto.GetMessageText();
    Title = proto.GetTitle();
    return true;
}

TExpected<NDrive::INotifier::TMessage, TString> TUserMessageTagBase::BuildMessage(ELocalization locale, const NDrive::IServer& server, const INotificationsTagDescription& desc, const TUserContacts& /*contacts*/) const {
    auto messageDesc = dynamic_cast<const TUserMessageTagBase::TDescription*>(&desc);
    if (!messageDesc) {
        return MakeUnexpected<TString>("Wrong description tag type");
    }

    auto localization = server.GetLocalization();
    TString text;
    if (MessageText) {
        text = MessageText;
    } else {
        text = messageDesc->GetMessageText();
    }
    if (localization) {
        text = localization->ApplyResources(text, locale);
    }

    auto title = Title ? Title : messageDesc->GetTitle();
    if (title && localization) {
        title = localization->ApplyResources(title, locale);
    }

    auto additionalInfo = AdditionalInfo.IsDefined() ? AdditionalInfo : messageDesc->GetAdditionalInfo();

    for (auto&& [name, value] : Macros) {
        SubstGlobal(text, name, value);
        SubstGlobal(title, name, value);
        NJson::ReplaceAll(additionalInfo, name, value);
    }

    NDrive::INotifier::TMessage result(text);
    result.SetName(GetName());
    result.SetMessageHash(MD5::CalcRaw(result.GetBody()));
    if (additionalInfo.IsDefined()) {
        result.SetAdditionalInfo(std::move(additionalInfo));
    }
    if (title) {
        result.SetTitle(title);
    }
    return result;
}

NDrive::TScheme TUserMessageTagBase::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSJson>("additional_info", "Additional message metadata");
    result.Add<TFSString>("message_text", "Текст сообщения");
    result.Add<TFSString>("title", "Заголовок сообщения");
    return result;
}

NJson::TJsonValue TUserMessageTagBase::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue jsonMeta = TBase::DoSerializeMetaToJson();
    NJson::InsertField(jsonMeta, "additional_info", AdditionalInfo);
    JWRITE_DEF(jsonMeta, "message_text", MessageText, "");
    JWRITE_DEF(jsonMeta, "push_text", MessageText, "");
    JWRITE_DEF(jsonMeta, "title", Title, "");
    return jsonMeta;
}

bool TUserMessageTagBase::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& jsonMeta) {
    AdditionalInfo = jsonMeta["additional_info"];
    JREAD_STRING_OPT(jsonMeta, "message_text", MessageText);
    JREAD_STRING_OPT(jsonMeta, "push_text", MessageText);
    JREAD_STRING_OPT(jsonMeta, "title", Title);
    return TBase::DoDeserializeMetaFromJson(jsonMeta);
}
