#include "offer.h"

#include <drive/backend/data/additional_service.h>
#include <drive/backend/data/additional_service_session.h>

bool TOfferHolderTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(objectId);
    Y_UNUSED(userId);
    Y_UNUSED(server);
    if (!Offer) {
        session.SetErrorInfo("OfferHolderTag::OnBeforeAdd", "no offer provided");
        return false;
    }
    return true;
}

NDrive::NProto::TOfferHolderTag TOfferHolderTag::DoSerializeSpecialDataToProto() const {
    NDrive::NProto::TOfferHolderTag result;
    if (Offer) {
        *result.MutableOffer() = Offer->SerializeToProto();
    }
    return result;
}

bool TOfferHolderTag::DoDeserializeSpecialDataFromProto(const NDrive::NProto::TOfferHolderTag& proto) {
    if (proto.HasOffer()) {
        Offer = ICommonOffer::TFactory::Construct(proto.GetOffer().GetInstanceType());
        if (!Offer || !Offer->DeserializeFromProto(proto.GetOffer())) {
            return false;
        }
    }
    return true;
}

TOptionalDBTags TOfferHolderUserTag::RestoreOfferHolderTags(
    TStringBuf behaviourConstructorId,
    TConstArrayRef<TString> tagNames,
    const NDrive::IServer& server,
    NDrive::TEntitySession& session
) {
    const auto& tagManager = Yensured(server.GetDriveAPI())->GetTagsManager().GetUserTags();
    auto optionalTags = tagManager.RestoreTags(TVector<TString>{}, tagNames, session);
    if (!optionalTags) {
        return {};
    }

    TDBTags result;
    for (auto&& tag : *optionalTags) {
        auto offerContainer = tag.GetTagAs<IOfferContainer>();
        if (!offerContainer) {
            session.SetErrorInfo("CalcOfferHolders", "cannot cast " + tag.GetTagId() + " to OfferContainer");
            return {};
        }
        auto offer = offerContainer->GetOffer();
        if (!offer) {
            continue;
        }
        if (behaviourConstructorId && behaviourConstructorId != offer->GetBehaviourConstructorId()) {
            continue;
        }
        result.push_back(tag);
    }
    return result;
}

bool TOfferBookingUserTag::OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const {
    Y_UNUSED(eContext);
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "OfferBookingUserTag_OnAfterEvolve")
        );
    }

    const auto driveApi = server ? server->GetDriveAPI() : nullptr;
    const auto& userTagsManager = Yensured(driveApi)->GetTagsManager().GetUserTags();

    auto offerContainer = fromTag.GetTagAs<IOfferContainer>();
    auto baseOffer = offerContainer ? offerContainer->GetOffer() : nullptr;
    if (!baseOffer) {
        session.SetErrorInfo("OfferBookUserTag::OnAfterEvolve", "cannot find offer in tag " + fromTag.GetTagId(), EDriveSessionResult::InternalError);
        return false;
    }

    auto offer = std::dynamic_pointer_cast<IOffer>(baseOffer);
    if (!offer) {
        session.SetErrorInfo("OfferBookUserTag::OnAfterEvolve", "not a vehicle offer in tag " + fromTag.GetTagId(), EDriveSessionResult::InternalError);
        return false;
    }

    offer->SetTargetHolderTag({});
    if (CarId) {
        offer->SetObjectId(CarId);
    }

    const auto& userId = fromTag.GetObjectId();
    auto userPermissions = driveApi->GetUserPermissions(userId, TUserPermissionsFeatures{});
    if (!userPermissions) {
        session.SetErrorInfo("OfferBookUserTag::OnAfterEvolve", "cannot create user permissions for " + userId, EDriveSessionResult::InternalError);
        return false;
    }

    TChargableTag::TBookOptions bookOptions;
    bookOptions.MultiRent = true;
    auto bookedTag = TChargableTag::Book(offer, *userPermissions, *server, session, bookOptions);
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "BookingResult")
            ("tag", NJson::ToJson(bookedTag))
        );
    }
    if (!bookedTag) {
        return false;
    }

    TDBTag evolvedTag;
    evolvedTag.SetData(toTag);
    evolvedTag.SetObjectId(fromTag.GetObjectId());
    evolvedTag.SetTagId(fromTag.GetTagId());
    if (!userTagsManager.RemoveTag(evolvedTag, permissions.GetUserId(), server, session)) {
        return false;
    }

    return true;
}

bool TOfferBookingUserTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return TBase::DoSpecialDataFromJson(value, errors) &&
        NJson::ParseField(value["car_id"], CarId);
}

void TOfferBookingUserTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::InsertNonNull(value, "car_id", CarId);
}

ITag::TFactory::TRegistrator<TOfferBookingUserTag> TOfferBookingUserTag::Registrator(TOfferBookingUserTag::Type());
ITag::TFactory::TRegistrator<TOfferHolderUserTag> TOfferHolderUserTag::Registrator(TOfferHolderUserTag::Type());

TAtomicSharedPtr<TBillingSession> CreateOfferHolderSession(const TConstDBTag& tag, const TOfferHolderSessionOptions& options, const IEntityTagsManager& tagManager, NDrive::TEntitySession& session) {
    return CreateOfferHolderSession(tag.GetTagId(), options, tagManager, session);
}

TAtomicSharedPtr<TBillingSession> CreateOfferHolderSession(const TString& tagId, const TOfferHolderSessionOptions& options, const IEntityTagsManager& tagManager, NDrive::TEntitySession& session) {
    auto events = tagManager.GetEventsByTag(tagId, session);
    if (!events) {
        return nullptr;
    }

    const auto result = MakeAtomicShared<TBillingSession>();

    IOffer::TPtr lastOffer;
    TString objectId;
    TString performer;
    for (auto&& ev : *events) {
        if (!performer && ev.GetObjectId()) {
            performer = ev.GetObjectId();
        }
        auto offerHolder = ev.GetTagAs<TOfferHolderTag>();
        auto offer = offerHolder ? std::dynamic_pointer_cast<IOffer>(offerHolder->GetOffer()) : nullptr;
        if (offer) {
            lastOffer = offer;
            objectId = offer->GetObjectId();
        }
    }
    TTagHistoryEventId eventId = 0;
    for (auto&& ev : *events) {
        if (!ev) {
            continue;
        }

        auto action = ev.GetHistoryAction();
        switch (action) {
        case EObjectHistoryAction::Add:
            action = EObjectHistoryAction::SetTagPerformer;
            break;
        case EObjectHistoryAction::Remove:
            action = EObjectHistoryAction::DropTagPerformer;
            break;
        default:
            break;
        }

        auto comment = ev.GetHistoryComment();
        auto originalEventId = ev.GetHistoryEventId();
        auto originator = ev.GetHistoryOriginatorId();
        auto timestamp = ev.GetHistoryTimestamp();
        auto userId = ev.GetHistoryUserId();

        TDBTag t{std::move(ev)};
        if (action != EObjectHistoryAction::DropTagPerformer) {
            t->SetPerformer(performer);
        }
        auto offerHolder = t.MutableTagAs<TOfferHolderTag>();
        if (offerHolder && action == EObjectHistoryAction::SetTagPerformer) {
            offerHolder->SetOffer(lastOffer);
        }
        if (offerHolder) {
            t->SetName(offerHolder->GetStage(nullptr));
        }
        t.SetObjectId(objectId);

        auto et = MakeAtomicShared<TObjectEvent<TConstDBTag>>(
            std::move(t),
            action,
            timestamp,
            action == EObjectHistoryAction::DropTagPerformer && result->GetUserId() ? result->GetUserId() : userId,
            originator,
            comment
        );
        if (options.IsStubEventId()) {
            et->SetHistoryEventId(eventId);
            ++eventId;
        } else {
            et->SetHistoryEventId(originalEventId);
        }
        auto category = TBillingSession::TSelector::AcceptImpl(*et);
        result->AddEvent(et, category);
    }

    if (result->GetCurrentState() != ISession::ECurrentState::Started && result->GetCurrentState() != ISession::ECurrentState::Closed) {
        session.SetErrorInfo("CreateOfferHolderSession", "session for tag " + tagId + " is niether Started nor Closed");
        return {};
    }
    bool compiled = result->Compile();
    if (!compiled) {
        session.SetErrorInfo("CreateOfferHolderSession", "can not compile session for tag " + tagId);
        return {};
    }
    return result;
}
