#pragma once

#include "offer_container.h"
#include "chargable.h"

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/offers/offers/abstract.h>
#include <drive/backend/proto/offer.pb.h>

class TBillingSession;
class TAdditionalServiceSession;

class TOfferHolderTag
    : public INativeSerializationTag<NDrive::NProto::TOfferHolderTag>
    , public IOfferContainer
{
private:
    using TBase = INativeSerializationTag<NDrive::NProto::TOfferHolderTag>;

public:
    using TBase::TBase;
    TOfferHolderTag(const TString& name, ICommonOffer::TPtr offer)
        : TBase(name)
        , Offer(offer)
    {
    }

    ICommonOffer::TPtr GetOffer() const override {
        return Offer;
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::NoUnique;
    }

    void SetOffer(ICommonOffer::TPtr offer) {
        Offer = offer;
    }

    const TString& GetStage(const TTaggedObject* /*object*/) const override {
        return GetName();
    }

    virtual bool OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;

protected:
    virtual NDrive::NProto::TOfferHolderTag DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const NDrive::NProto::TOfferHolderTag& proto) override;

private:
    ICommonOffer::TPtr Offer;
};

class TOfferHolderUserTag: public TOfferHolderTag {
private:
    using TBase = TOfferHolderTag;

public:
    static TOptionalDBTags RestoreOfferHolderTags(
        TStringBuf behaviourConstructorId,
        TConstArrayRef<TString> tagNames,
        const NDrive::IServer& server,
        NDrive::TEntitySession& session
    );

public:
    using TBase::TBase;

    static TString Type() {
        return "offer_holder_user_tag";
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::User };
    }

private:
    static TFactory::TRegistrator<TOfferHolderUserTag> Registrator;
};

class TOfferBookingUserTag: public IJsonSerializableTag {
private:
    using TBase = IJsonSerializableTag;

public:
    using TBase::TBase;

    static TString Type() {
        return "offer_booking_user_tag";
    }

    virtual EUniquePolicy GetUniquePolicy() const override {
        return EUniquePolicy::Rewrite;
    }

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { NEntityTagsManager::EEntityType::User };
    }

    bool OnAfterAdd(const TDBTag& /*self*/, const TString& /*userId*/, const NDrive::IServer* /*server*/, NDrive::TEntitySession& session) const override {
        session.SetErrorInfo("OfferBookUserTag::OnAfterAdd", "tag cannot be added", EDriveSessionResult::IncorrectRequest);
        return false;
    }

    virtual bool OnAfterEvolve(const TDBTag& fromTag, ITag::TPtr toTag, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;

protected:
    virtual bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    virtual void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;

private:
    R_FIELD(TString, CarId);

private:
    static TFactory::TRegistrator<TOfferBookingUserTag> Registrator;
};

struct TOfferHolderSessionOptions {
    R_FIELD(bool, StubEventId, true);
};

TAtomicSharedPtr<TBillingSession> CreateOfferHolderSession(const TString& tagId, const TOfferHolderSessionOptions& options, const IEntityTagsManager& tagManager, NDrive::TEntitySession& session);
TAtomicSharedPtr<TBillingSession> CreateOfferHolderSession(const TConstDBTag& tag, const TOfferHolderSessionOptions& options, const IEntityTagsManager& tagManager, NDrive::TEntitySession& session);
