#include "parking_payment.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/parking/inhouse/client.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

namespace {
    const TString FakeToken = "FAKE";
}

NDrive::TScheme TParkingInfoTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSJson>("groups", "Tokens + Groups");
    result.Add<TFSNumeric>("critical_usage_count", "Critical usage count of a token");
    result.Add<TFSNumeric>("required_slots_count", "Required slots count");
    result.Add<TFSBoolean>("use_fake_tokens", "Use fake tokens");
    result.Add<TFSString>("skip_aggregators_on_remove", "Skip aggregators check on remove");
    return result;
}

NJson::TJsonValue TParkingInfoTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result;
    result["groups"] = NJson::ToJson(Groups);
    result["critical_usage_count"] = CriticalUsageCount;
    result["required_slots_count"] = RequiredSlotsCount;
    result["use_fake_tokens"] = UseFakeTokens;
    NJson::InsertNonNull(result, "skip_aggregators_on_remove", JoinSeq(",", SkipAggregatorsOnRemove));
    return result;
}

bool TParkingInfoTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    {
        TString skipAggr;
        if (!NJson::ParseField(value["skip_aggregators_on_remove"], skipAggr)) {
            return false;
        }
        TSet<TString> skipSettings = StringSplitter(skipAggr).Split(',').SkipEmpty();
        for (auto&& aggregator : skipSettings) {
            i64 val = 0;
            if (TryFromString(aggregator, val)) {
                SkipAggregatorsOnRemove.insert(val);
            }
        }
    }
    return
        NJson::ParseField(value["groups"], Groups) &&
        NJson::ParseField(value["critical_usage_count"], CriticalUsageCount) &&
        NJson::ParseField(value["use_fake_tokens"], UseFakeTokens) &&
        NJson::ParseField(value["required_slots_count"], RequiredSlotsCount);
}

NJson::TJsonValue TParkingInfoTag::ToJson() const {
    NJson::TJsonValue data;
    SerializeSpecialDataToJson(data);
    return data;
}

bool TParkingInfoTag::OnBeforeAdd(const TString& objectId, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) try {
    auto api = Yensured(server)->GetDriveAPI();
    auto objects = Yensured(api)->GetCarsData();
    auto fetchResult = Yensured(objects)->FetchInfo(objectId, session);
    if (!fetchResult) {
        return false;
    }
    auto description = api->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    auto parkingInfoDescription = dynamic_cast<const TDescription*>(description.Get());
    if (!parkingInfoDescription) {
        parkingInfoDescription = &Default<TDescription>();
    }
    auto objectInfo = fetchResult.GetResultPtr(objectId);
    if (Slots.empty() && parkingInfoDescription->ShouldUseFakeTokens()) {
        for (size_t i = 0; i < parkingInfoDescription->GetRequiredSlotsCount(); ++i) {
            Slots.emplace_back().Token = FakeToken;
        }
    }
    if (Slots.empty()) {
        TVector<TTaggedDevice> devices;
        if (!api->GetTagsManager().GetDeviceTags().GetObjectsFromCache({ GetName() }, devices, TInstant::Zero())) {
            session.SetErrorInfo("ParkingInfoTag::OnBeforeAdd", "cannot get " + GetName() + " tags from cache", EDriveSessionResult::InternalError);
            return false;
        }

        TMap<TString, TString> groups = parkingInfoDescription->GetGroups();
        TMap<TString, ui32> tokens;
        for (auto&& token : NContainer::Keys(groups)) {
            tokens[token] = 0;
        }
        for (auto&& device : devices) {
            auto optionalTag = device.GetTag(GetName());
            if (!optionalTag) {
                continue;
            }

            auto parkingInfo = optionalTag->GetTagAs<TParkingInfoTag>();
            if (parkingInfo) {
                for (auto&& slot : parkingInfo->GetSlots()) {
                    auto it = tokens.find(slot.Token);
                    if (it == tokens.end()) {
                        tokens[slot.Token] = 1;
                    } else {
                        it->second += 1;
                    }
                }
            }
        }

        std::multimap<ui32, TString> invertedTokens;
        for (auto&&[token, count] : tokens) {
            invertedTokens.emplace(count, token);
        }

        TSet<TString> selectedTokens;
        TSet<TString> selectedGroups;
        for (auto&&[count, token] : invertedTokens) {
            if (count > parkingInfoDescription->GetCriticalUsageCount()) {
                session.SetErrorInfo("ParkingInfoTag::OnBeforeAdd", "critical usage count of tokens exceeded: " + ToString(count), EDriveSessionResult::InternalError);
                return false;
            }
            const TString& group = groups[token];
            if (!selectedGroups.insert(group).second) {
                continue;
            }
            if (!selectedTokens.insert(token).second) {
                session.SetErrorInfo("ParkingInfoTag::OnBeforeAdd", "duplicate selection of token " + token, EDriveSessionResult::DataCorrupted);
                return false;
            }

            TSlot slot;
            slot.Token = token;
            Slots.push_back(std::move(slot));
            if (Slots.size() >= parkingInfoDescription->GetRequiredSlotsCount()) {
                break;
            }
        }
        if (Slots.size() < parkingInfoDescription->GetRequiredSlotsCount()) {
            session.SetErrorInfo("ParkingInfoTag::OnBeforeAdd", "could not assign required slots count: " + ToString(Slots.size()), EDriveSessionResult::InternalError);
            return false;
        }
    }
    for (auto&& slot : Slots) {
        if (!slot.Token) {
            session.SetErrorInfo("ParkingInfoTag::OnBeforeAdd", "Token is empty", EDriveSessionResult::DataCorrupted);
            return false;
        }
        if (slot.Token == FakeToken) {
            continue;
        }
        if (slot.ParkingVehicleId) {
            continue;
        }
        NDrive::TParkingPaymentClient3 client(slot.Token);
        NDrive::TEventLog::Log("ParkingAddingVehicle", NJson::TMapBuilder("id", objectId)("token", slot.Token));
        auto vehicle = client.AddVehicleRobust(objectId, Yensured(objectInfo)->GetNumber());
        slot.ParkingVehicleId = vehicle.Reference;
        NDrive::TEventLog::Log("ParkingAddedVehicle", NJson::TMapBuilder("id", objectId)("token", slot.Token)("vehicle", vehicle.ToJson()));
    }
    return true;
} catch (const yexception& e) {
    session.SetErrorInfo("ParkingInfoTag::OnBeforeAdd", e.what(), EDriveSessionResult::InternalError);
    for (auto&& slot : Slots) {
        if (!slot.Token || !slot.ParkingVehicleId) {
            continue;
        }
        try {
            NDrive::TParkingPaymentClient3 client(slot.Token);
            auto vehicles = client.GetVehicles();
            for (auto&& vehicle : vehicles) {
                if (vehicle.Reference == slot.ParkingVehicleId) {
                    NDrive::TEventLog::Log("ParkingRollingBackVehicle", NJson::TMapBuilder
                        ("id", objectId)
                        ("token", slot.Token)
                        ("vehicle", vehicle.ToJson())
                    );
                    client.DeleteVehicleRobust(vehicle);
                    NDrive::TEventLog::Log("ParkingRolledBackVehicle", NJson::TMapBuilder
                        ("id", objectId)
                        ("token", slot.Token)
                        ("vehicle", vehicle.ToJson())
                    );
                }
            }
        } catch (const std::exception& e) {
            session.AddErrorMessage("ParkingInfoTag::OnBeforeAdd::rollback", FormatExc(e));
        }
    }
    return false;
}

bool TParkingInfoTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const try {
    auto api = Yensured(server)->GetDriveAPI();
    auto description = Yensured(api)->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
    auto parkingInfoDescription = dynamic_cast<const TDescription*>(description.Get());
    if (!parkingInfoDescription) {
        session.SetErrorInfo("ParkingInfoTag::OnAfterRemove", "unknown description", EDriveSessionResult::InternalError);
    }
    const TSet<i64>& skipAggregator = parkingInfoDescription->GetSkipAggregatorsOnRemove();
    TString carNumber;
    {
        auto objects = Yensured(api)->GetCarsData();
        auto fetchResult = Yensured(objects)->FetchInfo(self.GetObjectId(), session);
        if (!fetchResult) {
            return false;
        }
        auto object = fetchResult.GetResultPtr(self.GetObjectId());
        if (!object) {
            session.SetErrorInfo("ParkingInfoTag::OnAfterRemove", "unknown object", EDriveSessionResult::InternalError);
            return false;
        }
        carNumber = object->GetNumber();
    }
    auto manager = Yensured(server)->GetParkingManager();
    for (auto& slot : Slots) {
        if (!skipAggregator.contains(slot.SessionAggregatorId)) {
            Yensured(manager)->StopParking(self.GetObjectId(), userId, carNumber, slot);
        }
    }
    return true;
} catch (const yexception& e) {
    session.SetErrorInfo("ParkingInfoTag::OnAfterRemove", e.what(), EDriveSessionResult::InternalError);
    return false;
}

void TParkingInfoTag::SerializeSpecialDataToJson(NJson::TJsonValue& data) const {
    TBase::SerializeSpecialDataToJson(data);

    NJson::TJsonValue& slots = data.InsertValue("slots", NJson::JSON_ARRAY);
    for (auto&& slot : Slots) {
        slots.AppendValue(slot.ToJson());
    }
}

bool TParkingInfoTag::DoSpecialDataFromJson(const NJson::TJsonValue& data, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(data, errors)) {
        return false;
    }

    const NJson::TJsonValue& slots = std::as_const(data)["slots"];
    const NJson::TJsonValue::TArray& input = data.IsArray() ? data.GetArray() : slots.GetArray();
    Slots.clear();
    for (auto&& s : input) {
        TSlot slot;
        slot.Token = s["token"].GetStringRobust();
        slot.ParkingVehicleId = s["parking_vehicle_id"].GetString();
        slot.ParkingSessionId = s["parking_session_id"].GetString();
        slot.Session = std::move(s["session"]);
        slot.Refund = std::move(s["refund"]);
        slot.Cost = s["cost"].GetDouble();
        if (s.Has("session_aggregator_id")) {
            slot.SessionAggregatorId = s["session_aggregator_id"].GetInteger();
        }
        if (s.Has("session_finish")) {
            slot.SessionFinish = TInstant::Seconds(s["session_finish"].GetUIntegerRobust());
        }
        Slots.push_back(std::move(slot));
    }
    return true;
}

double TParkingInfoTag::GetCost(TInstant now) const {
    double result = 0;
    for (auto&& slot : Slots) {
        if (slot.IsActive(now)) {
            result += slot.Cost;
        }
    }
    return result;
}

bool TParkingInfoTag::IsActive(TInstant now) const {
    for (auto&& slot : Slots) {
        if (slot.IsActive(now)) {
            return true;
        }
    }
    return false;
}

TTagDescription::TFactory::TRegistrator<TParkingInfoTag::TDescription> DryRunParkingInfoTagDescriptionRegistrator(TParkingInfoTag::DryRunTagName());
NDrive::ITag::TFactory::TRegistrator<TParkingInfoTag> DryRunParkingInfoTagRegistrator(TParkingInfoTag::DryRunTagName());

TTagDescription::TFactory::TRegistrator<TParkingInfoTag::TDescription> ParkingInfoTagDescriptionRegistrator(TParkingInfoTag::TagName());
NDrive::ITag::TFactory::TRegistrator<TParkingInfoTag> ParkingInfoTagRegistrator(TParkingInfoTag::TagName());
