#include "chargable.h"
#include "notifications_tags.h"
#include "radar.h"

#include <drive/backend/actions/filter.h>
#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/radar/radar_geohash.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users_controller/ucontroller.h>

const TString TRadarUserTag::TypeName = "radar_user_tag";

void TRadarUserTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "radar_position", RadarPosition);
    NJson::InsertField(json, "start_timestamp", StartTimestamp.Seconds());
    NJson::InsertField(json, "walk_duration", WalkDuration.Seconds());
    NJson::InsertField(json, "action", ToString(Action));
    NJson::InsertField(json, "filter_identifiers", FilterIdentifiers);
    NJson::InsertField(json, "search_area", SearchArea);
}

bool TRadarUserTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    return NJson::ParseField(json, "radar_position", RadarPosition, false, errors)
        && NJson::ParseField(json, "start_timestamp", StartTimestamp, false, errors)
        && NJson::ParseField(json, "walk_duration", WalkDuration, false, errors)
        && NJson::ParseField(json, "action", NJson::Stringify(Action), false, errors)
        && NJson::ParseField(json, "filter_identifiers", FilterIdentifiers, false, errors)
        && NJson::ParseField(json, "search_area", SearchArea, false, errors)
        && TBase::DoSpecialDataFromJson(json, errors);
}

TRadarUserTag::TProto TRadarUserTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetStartTimestamp(StartTimestamp.Seconds());
    proto.SetWalkDuration(WalkDuration.Seconds());
    proto.SetAction(static_cast<ui32>(Action));
    for (auto&& i : FilterIdentifiers) {
        proto.AddFilterIdentifier(i);
    }
    *proto.MutableRadarPosition() = RadarPosition.SerializeHP();
    for (auto&& coord : SearchArea) {
        *proto.AddSearchArea() = coord.SerializeHP();
    }
    return proto;
}

bool TRadarUserTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    Action = static_cast<EAction>(proto.GetAction());
    WalkDuration = TDuration::Seconds(proto.GetWalkDuration());
    StartTimestamp = TInstant::Seconds(proto.GetStartTimestamp());
    for (auto&& i : proto.GetFilterIdentifier()) {
        FilterIdentifiers.emplace_back(i);
    }
    if (!proto.HasRadarPosition() || !RadarPosition.Deserialize(proto.GetRadarPosition())) {
        return false;
    }
    for (auto&& i : proto.GetSearchArea()) {
        TGeoCoord coord;
        if (!coord.Deserialize(i)) {
            return false;
        }
        SearchArea.emplace_back(std::move(coord));
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

NJson::TJsonValue TRadarUserTag::GetPublicReport(ELocalization locale, const NDrive::IServer& server) const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    result.InsertValue("position", RadarPosition.ToString());
    result.InsertValue("start", StartTimestamp.Seconds());
    result.InsertValue("deadline", GetSLAInstant().Seconds());
    result.InsertValue("walk_duration", WalkDuration.Seconds());
    result.InsertValue("action", ToString(Action));
    result.InsertValue("filter", FilterIdentifiers.size() ? FilterIdentifiers.front() : "");
    TJsonProcessor::WriteContainerArray(result, "filters", FilterIdentifiers);
    result.InsertValue("is_finished", GetSLAInstant() == TInstant::Zero());
    if (GetSLAInstantUnsafe() < ModelingNow()) {
        result.InsertValue("finished_message", NDrive::TLocalization::ScannerExpired());
    }
    const ILocalization* localization = server.GetLocalization();
    const TDriveAPI* api = server.GetDriveAPI();
    if (localization) {
        const TInstant now = Now();
        const TString& since = localization->FormatTimeOfTheDay(locale, StartTimestamp);
        const TString& until = localization->FormatTimeOfTheDay(locale, GetSLAInstant());

        TString message;
        if (now >= StartTimestamp) {
            message = localization->GetLocalString(locale, "radar_message_in_progress");
        } else {
            message = localization->GetLocalString(locale, "radar_message_pending");
        }
        SubstGlobal(message, "_RadarSince_", since);
        SubstGlobal(message, "_RadarUntil_", until);
        result["localizations"].InsertValue("message", message);

        TVector<TString> details;
        if (Action == EAction::Order) {
            details.push_back(localization->GetLocalString(locale, "radar_details_autobook"));
        }
        if (now < StartTimestamp) {
            auto value = localization->GetLocalString(locale, "radar_details_until");
            SubstGlobal(value, "_RadarUntil_", until);
            details.push_back(value);
        }
        if (FilterIdentifiers.size()) {
            auto value = localization->GetLocalString(locale, "radar_details_with_filter");
            TSet<TString> filterIds;
            for (auto&& fId : FilterIdentifiers) {
                StringSplitter(fId).SplitBySet("* ").SkipEmpty().AddTo(&filterIds);
            }
            TSet<TString> filterNames;
            for (auto&& filterId : filterIds) {
                auto action = api->GetRolesManager()->GetAction(filterId);
                auto filter = action ? action->GetAs<TFilterAction>() : nullptr;
                filterNames.emplace(filter ? localization->ApplyResources(filter->GetText(), locale) : filterId);
            }
            SubstGlobal(value, "_FilterName_", JoinSeq(", ", filterNames));
            details.push_back(value);
        } else {
            details.push_back(localization->GetLocalString(locale, "radar_details_no_filter"));
        }
        result["localizations"].InsertValue("details", JoinSeq("\n", details));
    }
    return result;
}

NDrive::TScheme TRadarUserTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSVariants>("action", "Действие радара").InitVariants<EAction>();
    result.Add<TFSNumeric>("start_timestamp", "Время запуска радара")
        .SetVisual(NDrive::TFSNumeric::EVisualType::DateTime);
    result.Add<TFSArray>("radar_position", "Позиция радара на карте").SetElement<NDrive::TFSNumeric>();
    result.Add<TFSDuration>("walk_duration", "Дистанция радиуса измеримая временем ходьбой").SetDefault(TDuration::Zero());
    result.Add<TFSArray>("filter_identifiers", "Фильтры настроенные пользователем").SetElement<TFSString>();

    return result;
}

bool TRadarUserTag::OnAfterAdd(const TDBTag& dbTag, const TString& /*userId*/, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (server->GetSettings().GetValue<bool>("radar.use_geohash_add").GetOrElse(false)) {
        const TRadarGeohashManager* radarManager = Yensured(server->GetRadarGeohashManager());
        const TRadarUserTag* tag = Yensured(dbTag.GetTagAs<TRadarUserTag>());

        ui8 setPrecision = server->GetSettings().GetValue<ui8>("radar.set_geohash_precision").GetOrElse(6);
        if (!radarManager->AddTagId(tag->GetSearchArea(), dbTag.GetTagId(), setPrecision, session)) {
            session.AddErrorMessage("TRadarUserTag::OnAfterAdd", "add tile id error");
            return false;
        }
    }
    return true;
}

bool TRadarUserTag::OnBeforeRemove(const TDBTag& dbTag, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (server->GetSettings().GetValue<bool>("radar.use_geohash_remove").GetOrElse(false)) {
        const TRadarGeohashManager* radarManager = Yensured(server->GetRadarGeohashManager());

        if (!radarManager->RemoveTagId(dbTag.GetTagId(), session)) {
            session.AddErrorMessage("TRadarUserTag::OnBeforeRemove", "remove tile by id error");
            return false;
        }
    }
    return true;
}

bool TRadarUserTag::OnBeforeRemove(const TDBTag& dbTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_UNUSED(userId);
    return OnBeforeRemove(dbTag, server, session);
}

bool TRadarUserTag::OnBeforeEvolve(const TDBTag& from, ITag::TPtr /*to*/, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* /*evolutionContext*/) const {
    Y_UNUSED(permissions);
    if (!OnBeforeRemove(from, server, tx)) {
        return false;
    }
    return true;
}

bool TRadarUserTag::OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& /*permissions*/, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* /*eContext*/) const {
    TDBTag evolvedTag;
    evolvedTag.SetData(to);
    evolvedTag.SetObjectId(from.GetObjectId());
    evolvedTag.SetTagId(from.GetTagId());
    return OnAfterAdd(evolvedTag, from.GetObjectId(), server, session);
}

bool TRadarUserTag::OnAfterUpdate(const TDBTag& dbTag, ITag::TPtr toTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!OnBeforeRemove(dbTag, server, session)) {
        return false;
    }
    TDBTag updatedTag;
    updatedTag.SetData(toTag);
    updatedTag.SetObjectId(dbTag.GetObjectId());
    updatedTag.SetTagId(dbTag.GetTagId());
    return OnAfterAdd(updatedTag, userId, server, session);
}

TSet<NEntityTagsManager::EEntityType> TRadarUserTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::User };
}

EUniquePolicy TRadarUserTag::GetUniquePolicy() const {
    return EUniquePolicy::Rewrite;
}

TFilterActionSet TRadarUserTag::GetFiltersSet(const NDrive::IServer& server, const TSet<TString>& ids, TMap<TString, TCompositeFilter>& CompositeFilters) {
    TVector<const TCompositeFilter*> filters;
    for (auto&& i : ids) {
        auto it = CompositeFilters.find(i);
        if (it == CompositeFilters.end()) {
            TCompositeFilter cf(i, &server);
            it = CompositeFilters.emplace(i, std::move(cf)).first;
        }
        filters.emplace_back(&it->second);
    }
    return TFilterActionSet(std::move(filters));
}

bool TRadarUserTag::FilterByFilterSet(const NDrive::IServer& server, const TDBTag& tagDB, const TTaggedObject& objectTags) {
    const TRadarUserTag* tag = Yensured(tagDB.GetTagAs<TRadarUserTag>());
    const TSet<TString>& filterIds = MakeSet(tag->GetFilterIdentifiers());
    TMap<TString, TCompositeFilter> compositeFilters;
    const TFilterActionSet carFilters = GetFiltersSet(server, filterIds, compositeFilters);
    return carFilters.Match(objectTags);
}

void TRadarUserTag::FilterByFilterSet(const NDrive::IServer& server, TDBTags& radarTags, const TString& carId) {
    if (radarTags.empty()) {
        return;
    }
    auto objectTags = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(carId);
    if (!objectTags) {
        return;
    }
    const auto pred = [&server, &objectTags](const TDBTag& tagDB) -> bool {
        return !TRadarUserTag::FilterByFilterSet(server, tagDB, *objectTags);
    };
    radarTags.erase(std::remove_if(radarTags.begin(), radarTags.end(), pred), radarTags.end());
}

bool TRadarUserTag::FilterByCoords(const TDBTag& tagDB, const TGeoCoord& carPos) {
    const TRadarUserTag* tag = tagDB.GetTagAs<TRadarUserTag>();
    if (!tag) {
        return false;
    }
    if (tag->GetSearchArea().empty()) {
        return true;
    }
    auto line = TPolyLine<TGeoCoord>(tag->GetSearchArea());
    return line.IsPointInternal(carPos);
}

void TRadarUserTag::FilterByCoords(TDBTags& radarTags, const TGeoCoord& carPos) {
    const auto pred = [&carPos](const TDBTag& tagDB) -> bool {
        return !TRadarUserTag::FilterByCoords(tagDB, carPos);
    };
    radarTags.erase(std::remove_if(radarTags.begin(), radarTags.end(), pred), radarTags.end());
}

void TRadarUserTag::Sort(TDBTags& radarTags) {
    std::sort(radarTags.begin(), radarTags.end(), [](const TDBTag& lhs, const TDBTag& rhs) -> bool {
        return Yensured(lhs.GetTagAs<TRadarUserTag>())->GetStartTimestamp() < Yensured(rhs.GetTagAs<TRadarUserTag>())->GetStartTimestamp();
    });
}

TMaybe<TDBTags> TRadarUserTag::FindTags(const NDrive::IServer& server, const TGeoCoord& carPos, const TString& carId, NDrive::TEntitySession& session) {
    TMaybe<TDBTags> radarTags;
    if (server.GetSettings().GetValue<bool>("radar.use_geohash_find").GetOrElse(false)) {
        const TRadarGeohashManager* radarManager = Yensured(server.GetRadarGeohashManager());
        ui8 getPrecision = server.GetSettings().GetValue<ui8>("radar.get_geohash_precision").GetOrElse(6);
        auto tagIds = std::move(radarManager->GetTagIds(carPos, getPrecision, session));
        if (!tagIds) {
            return Nothing();
        }
        radarTags = server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(*tagIds, session);
        if (!radarTags) {
            session.SetErrorInfo("TRadarUserTag::FindTags", "can't restore tags by their tag id");
            return Nothing();
        }
    } else {
        radarTags = server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TVector<TString>{}, {TRadarUserTag::TypeName}, session);
        if (!radarTags) {
            NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                ("stage", "FindTags [old]")
                ("error", "no user with activated radar")
            );
            return Nothing();
        }
    }
    FilterByCoords(*radarTags, carPos);
    FilterByFilterSet(server, *radarTags, carId);
    Sort(*radarTags);
    return radarTags;
}

bool TRadarUserTag::SendPushNotification(const NDrive::IServer& server, const TRadarUserTag* tag, const TString& carId, IOffer::TPtr offer, const TString& userId, NDrive::TEntitySession& dbSession) {
    const ISettings& settings = server.GetSettings();
    TString messageBody;
    NJson::TJsonValue additionalJson;
    auto carMaybeObject = server.GetDriveDatabase().GetCarManager().GetObject(carId);
    if (!carMaybeObject) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "SendPushNotification")
            ("error", "can't take TDriveCarInfo by objectId")
        );
        return false;
    }
    TString model = carMaybeObject->GetModel();
    TString pushTagName;
    additionalJson.InsertValue("number", carMaybeObject->GetNumber());
    if (tag->GetAction() == TRadarUserTag::EAction::Search) {
        messageBody = NDrive::TLocalization::ScannerCarIsFound(model, "");
        pushTagName = settings.GetValue<TString>("radar.found.push_tag_name").GetOrElse("radarpush_found");
        additionalJson.InsertValue("status", "detect_car");
    } else {
        messageBody = NDrive::TLocalization::ScannerOrderReady(model, Yensured(offer)->GetPushReport(tag->GetLocale(), &server));
        pushTagName = settings.GetValue<TString>("radar.booked.push_tag_name").GetOrElse("radarpush_booked");
        additionalJson.InsertValue("status", "ordered_car");
    }

    auto pushTagMaybe = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(pushTagName);
    if (!pushTagMaybe) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "SendPushNotification")
            ("error", "can't create tag")
        );
        return false;
    }
    auto pushTag = std::dynamic_pointer_cast<TUserPushTag>(pushTagMaybe);
    if (!pushTag) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "SendPushNotification")
            ("error", "can't cast push tag")
        );
        return false;
    }
    pushTag->SetMessageText(messageBody);
    pushTag->SetAdditionalInfo(std::move(additionalJson));
    const TString id = settings.GetValue<TString>("radar.robot_user_id").GetOrElse("robot-user-radar");
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(pushTag, id, userId, &server, dbSession)) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "SendPushNotification")
            ("user_id", userId)
            ("id", id)
            ("error", "can't add tag to a user")
        );
        return false;
    }
    return true;
}

bool TRadarUserTag::FinishRadar(const NDrive::IServer& server, TDBTag& dbTag, const TString& userId, NDrive::TEntitySession& dbSession) {
    TRadarUserTag* tag = dbTag.MutableTagAs<TRadarUserTag>();
    if (!tag) {
        return false;
    }
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(dbTag, userId, &server, dbSession, true)) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "FinishRadar")
            ("can_not_remove_tag", "can't remove tag from user")
            ("user_id", userId)
            ("tag", NJson::ToJson(dbTag))
            ("error", dbSession.GetStringReport())
        );
        return false;
    }
    return true;
}

EDriveSessionResult TRadarUserTag::CarBooking(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, IOffer::TPtr offer, NDrive::TEntitySession& dbSession) {
    EDriveSessionResult status = EDriveSessionResult::Success;
    TChargableTag::TBookOptions bookOptions;
    if (!TChargableTag::Book(offer, *permissions, server, dbSession, bookOptions)) {
        if (!dbSession.GetTransaction()->IsFailed()) {
            status = dbSession.DetachResult();
            NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                ("stage", "CarBooking")
                ("error", "EDriveSessionResult: " + ToString(status))
            );
        }
    } else {
        server.GetOffersStorage()->RemoveOffer(offer->GetOfferId(), permissions->GetUserId(), dbSession);
    }
    return status;
}

IOffer::TPtr TRadarUserTag::MakeOffer(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, const TRadarUserTag* tag, const TString& carId, NDrive::TEntitySession& dbSession) {
    TUserOfferContext uoc(&server, permissions, nullptr);
    uoc.SetLocale(tag->GetLocale());
    uoc.SetUserPosition(tag->GetRadarPosition());
    auto api = server.GetDriveAPI();
    auto correctors = permissions->GetOfferCorrections();
    {
        auto additionalActions = api->GetRolesManager()->GetUserAdditionalActions(permissions->GetUserId(), api->GetTagsManager(), false);
        if (!additionalActions) {
            dbSession.SetErrorInfo("RadarUserTag::MakeOffer", "no additional actions");
            return nullptr;
        }
        auto filtered = TAdditionalActions::Filter<IOfferCorrectorAction>(std::move(*additionalActions));
        correctors.insert(correctors.end(), filtered.begin(), filtered.end());
    }
    {
        auto walletAdditionalActions = api->GetRolesManager()->GetAccountsAdditionalActions(uoc.GetRequestUserAccounts(), api->GetTagsManager(), false);
        if (!walletAdditionalActions) {
            dbSession.SetErrorInfo("RadarUserTag::MakeOffer", "no wallet additional actions");
            return nullptr;
        }
        auto filtered = TAdditionalActions::Filter<IOfferCorrectorAction>(std::move(*walletAdditionalActions));
        correctors.insert(correctors.end(), filtered.begin(), filtered.end());
    }
    IOffer::TPtr offer;
    TString offerReport;
    TOffersBuildingContext context(&server);
    context.SetCarId(carId);
    context.SetUserHistoryContext(uoc);
    for (auto&& offerBuilder : permissions->GetOfferBuilders()) {
        const TStandartOfferConstructor* offerConstructor = dynamic_cast<const TStandartOfferConstructor*>(offerBuilder.Get());
        if (offerConstructor && offerConstructor->GetIsPublish() && offerConstructor->GetAvailableForScanner()) {
            TVector<IOfferReport::TPtr> offers;
            auto result = offerConstructor->BuildOffers(*permissions, correctors, offers, context, &server, dbSession);
            if (result != EOfferCorrectorResult::Success) {
                NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                    ("car_id", carId)
                    ("constructor", offerConstructor->GetName())
                    ("stage", "MakeOffer")
                    ("error", "EOfferCorrectorResult: " + ToString(result))
                );
                continue;
            }
            if (offers.size() != 1) {
                NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                    ("car_id", carId)
                    ("constructor", offerConstructor->GetName())
                    ("stage", "MakeOffer")
                    ("error", "offers size not one")
                );
                continue;
            }
            offer = offers.front()->GetOfferPtrAs<IOffer>();
            if (!offer) {
                NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                    ("car_id", carId)
                    ("constructor", offerConstructor->GetName())
                    ("stage", "MakeOffer")
                    ("error", "not a vehicle offer")
                );
                continue;
            }
            offer->SetFromScanner(true);
            break;
        }
    }
    return offer;
}

TMaybe<bool> TRadarUserTag::ApplyCarOrder(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, TDBTag& dbTag, const TString& carId, NDrive::TEntitySession& tx) {
    const TRadarUserTag* tag = Yensured(dbTag.GetTagAs<TRadarUserTag>());
    if (tag->GetStartTimestamp() > Now()) {
        return false;
    }
    const auto& userId = dbTag.GetObjectId();
    auto offer = MakeOffer(server, permissions, tag, carId, tx);
    if (!offer) {
        tx.AddErrorMessage("ApplyCarOrderCarId", carId);
        tx.AddErrorMessage("ApplyCarOrderUserId", userId);
        return {};
    }
    EDriveSessionResult status = CarBooking(server, permissions, offer, tx);
    if (status == EDriveSessionResult::PaymentRequired || status == EDriveSessionResult::NoPermissionsForPerform || status == EDriveSessionResult::CarIsBusy) {
        // In this case transaction should not be broken.
        return false;
    } else if (status != EDriveSessionResult::Success) {
        tx.SetErrorInfo("TRadarUserTag::CarBooking", tx.GetStringReport(), status);
        return {};
    }
    if (!SendPushNotification(server, tag, carId, offer, userId, tx)) {
        tx.SetErrorInfo("TRadarUserTag::SendPushNotification", tx.GetStringReport(), EDriveSessionResult::InternalError);
        TUnistatSignalsCache::SignalAdd("radar-order", "errors", 1);
        return {};
    }
    if (!FinishRadar(server, dbTag, userId, tx)) {
        tx.SetErrorInfo("TRadarUserTag::FinishRadar", tx.GetStringReport(), EDriveSessionResult::InternalError);
        TUnistatSignalsCache::SignalAdd("radar-order", "errors", 1);
        return {};
    }
    TUnistatSignalsCache::SignalAdd("radar-order", "ok", 1);
    return true;
}

TMaybe<bool> TRadarUserTag::ApplyCarSearch(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, TDBTag& dbTag, const TString& carId, NDrive::TEntitySession& tx) {
    Y_UNUSED(permissions);
    const TRadarUserTag* tag = Yensured(dbTag.GetTagAs<TRadarUserTag>());
    if (tag->GetStartTimestamp() > Now()) {
        return false;
    }
    const auto& userId = dbTag.GetObjectId();
    if (!SendPushNotification(server, tag, carId, {}, userId, tx)) {
        tx.SetErrorInfo("TRadarUserTag::SendPushNotification", tx.GetStringReport(), EDriveSessionResult::InternalError);
        TUnistatSignalsCache::SignalAdd("radar-search", "errors", 1);
        return {};
    }
    if (!FinishRadar(server, dbTag, userId, tx)) {
        tx.SetErrorInfo("TRadarUserTag::FinishRadar", tx.GetStringReport(), EDriveSessionResult::InternalError);
        TUnistatSignalsCache::SignalAdd("radar-search", "errors", 1);
        return {};
    }
    TUnistatSignalsCache::SignalAdd("radar-search", "ok", 1);
    return true;
}

bool TRadarUserTag::UserCanRentCar(const NDrive::IServer& server, const TString& userId) {
    if (!userId) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "pre offer")
            ("object_id", userId)
        );
        return false;
    }
    TVector<ISession::TConstPtr> userSessions;
    server.GetDriveAPI()->GetCurrentUserSessions(userId, userSessions, TInstant::Now());
    if (userSessions.size()) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "pre offer")
            ("error", "user have rented car")
        );
        return false;
    }
    return true;
};

bool TRadarUserTag::ApplyRadarByCar(const NDrive::IServer& server, const TString& carId, const TGeoCoord& coord, NDrive::TEntitySession& dbSession) {
    auto maybeAllRadarUserTags = FindTags(server, coord, carId, dbSession);
    if (!maybeAllRadarUserTags) {
        dbSession.SetErrorInfo("TRadarUserTag::FindTags", dbSession.GetStringReport(), EDriveSessionResult::InternalError);
        return false;
    }
    auto device = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreObject(carId, dbSession);
    if (!device) {
        NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
            ("stage", "ApplyRadarByCar")
            ("car_id", carId)
            ("error", "can not retrieve car info")
        );
        return false;
    }
    for (auto&& dbTag : *maybeAllRadarUserTags) {
        const TRadarUserTag* tag = Yensured(dbTag.GetTagAs<TRadarUserTag>());
        if (tag->GetAction() != TRadarUserTag::EAction::Order) {
            continue;
        }
        const TString& userId = dbTag.GetObjectId();
        if (!UserCanRentCar(server, userId)) {
            continue;
        }
        auto api = server.GetDriveAPI();
        auto permissions = Yensured(api)->GetUserPermissions(userId, {}, TInstant::Zero(), nullptr, false, false);
        if (!permissions) {
            NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                ("stage", "ApplyRadarByCar")
                ("user_id", userId)
                ("error", "can't retrieve permissions")
            );
            continue;
        }
        if (permissions->GetVisibility(*device, NEntityTagsManager::EEntityType::Car) != TUserPermissions::EVisibility::Visible) {
            continue;
        }
        NDrive::TEventLog::TUserIdGuard userIdGuard(userId);
        if (!ApplyCarOrder(server, permissions, dbTag, carId, dbSession)) {
            continue;
        }
        return true;
    }

    bool userFound = false;

    for (auto&& dbTag : *maybeAllRadarUserTags) {
        const TRadarUserTag* tag = Yensured(dbTag.GetTagAs<TRadarUserTag>());
        if (tag->GetAction() != TRadarUserTag::EAction::Search) {
            continue;
        }
        const TString& userId = dbTag.GetObjectId();
        if (!UserCanRentCar(server, userId)) {
            continue;
        }
        auto api = server.GetDriveAPI();
        auto permissions = Yensured(api)->GetUserPermissions(userId);
        if (!permissions) {
            NDrive::TEventLog::Log("RadarError", NJson::TMapBuilder
                ("stage", "ApplyRadarByCar")
                ("user_id", userId)
                ("error", "can't retrieve permissions")
            );
            continue;
        }
        if (permissions->GetVisibility(*device, NEntityTagsManager::EEntityType::Car) != TUserPermissions::EVisibility::Visible) {
            continue;
        }
        if (!ApplyCarSearch(server, permissions, dbTag, carId, dbSession)) {
            continue;
        }
        userFound = true;
    }
    return userFound;
}

NDrive::ITag::TFactory::TRegistrator<TRadarUserTag> RadarUserTagRegistrator(TRadarUserTag::TypeName);
