#pragma once

#include <drive/backend/cars/car.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/common/localization.h>
#include <drive/backend/offers/abstract.h>
#include <drive/backend/proto/tags.pb.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/tags/tag.h>

#include <rtline/library/geometry/coord.h>

class TRadarUserTag : public ISerializableTag<NDrive::NProto::TRadarUserTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TRadarUserTag>;

private:
    R_FIELD(ELocalization, Locale, DefaultLocale);
    R_FIELD(TGeoCoord, RadarPosition);
    R_FIELD(TInstant, StartTimestamp, TInstant::Zero());
    R_FIELD(TDuration, WalkDuration, TDuration::Zero());
    R_FIELD(TVector<TString>, FilterIdentifiers);
    R_FIELD(TVector<TGeoCoord>, SearchArea);

public:
    using TBase::TBase;

    enum class EAction {
        Search = 0 /* "search" */,
        Order = 1 /* "order" */
    };

private:
    R_FIELD(EAction, Action, EAction::Order);

public:
    static const TString TypeName;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;

    virtual EUniquePolicy GetUniquePolicy() const override;
    NJson::TJsonValue GetPublicReport(ELocalization locale, const NDrive::IServer& server) const;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

    static void Sort(TDBTags& radarTags);
    static bool FilterByFilterSet(const NDrive::IServer& server, const TDBTag& tagDB, const TTaggedObject& objectTags);
    static bool FilterByCoords(const TDBTag& tagDB, const TGeoCoord& carPos);
    static bool UserCanRentCar(const NDrive::IServer& server, const TString& userId);
    static TMaybe<bool> ApplyCarSearch(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, TDBTag& dbTag, const TString& carId, NDrive::TEntitySession& tx);
    static TMaybe<bool> ApplyCarOrder(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, TDBTag& dbTag, const TString& carId, NDrive::TEntitySession& tx);
    // ApplyRadarByCar searches for radars for the specified vehicle and tries to launch them.
    //
    // This should be used to run on cars that have become available as soon as possible.
    static bool ApplyRadarByCar(const NDrive::IServer& server, const TString& carId, const TGeoCoord& coord, NDrive::TEntitySession& tx);

private:
    virtual bool OnAfterAdd(const TDBTag& dbTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;
    virtual bool OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* evolutionContext) const override;
    virtual bool OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* eContext) const override;
    virtual bool OnBeforeRemove(const TDBTag& dbTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) override;
    virtual bool OnAfterUpdate(const TDBTag& dbTag, ITag::TPtr toTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const override;

    bool OnBeforeRemove(const TDBTag& dbTag, const NDrive::IServer* server, NDrive::TEntitySession& session) const;

private:
    static TFilterActionSet GetFiltersSet(const NDrive::IServer& server, const TSet<TString>& ids, TMap<TString, TCompositeFilter>& CompositeFilters);
    static void FilterByFilterSet(const NDrive::IServer& server, TDBTags& radarTags, const TString& carId);
    static void FilterByCoords(TDBTags& radarTags, const TGeoCoord& carPos);
    static TMaybe<TDBTags> FindTags(const NDrive::IServer& server, const TGeoCoord& carPos, const TString& carId, NDrive::TEntitySession& session);
    static bool SendPushNotification(const NDrive::IServer& server, const TRadarUserTag* tag, const TString& carId, IOffer::TPtr offer, const TString& userId, NDrive::TEntitySession& dbSession);
    // FinishRadar removes radar tag.
    static bool FinishRadar(const NDrive::IServer& server, TDBTag& dbTag, const TString& userId, NDrive::TEntitySession& dbSession);
    // CarBooking books speficied offer.
    static EDriveSessionResult CarBooking(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, IOffer::TPtr offer, NDrive::TEntitySession& dbSession);
    // MakeOffer builds offer for specified user and car.
    static IOffer::TPtr MakeOffer(const NDrive::IServer& server, TUserPermissions::TConstPtr permissions, const TRadarUserTag* tag, const TString& carId, NDrive::TEntitySession& dbSession);

private:
    static TFactory::TRegistrator<TRadarUserTag> Registrator;
};
