#include "rental_offer_holder_tag.h"

#include "rental_service_mode_tag.h"
#include "timetable_builder.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/notifications_tags.h>

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/proto/adapter.h>
#include <rtline/util/algorithm/type_traits.h>

#include <memory>

namespace {
const TString onBeforeAddSourceSession{ "RentalOfferHolderTag::OnBeforeAdd" };
const TString onAfterAddSourceSession{ "RentalOfferHolderTag::OnAfterAdd" };
const TString onBeforeRemoveSourceSession{ "RentalOfferHolderTag::OnBeforeRemove" };
const TString onAfterRemoveSourceSession{ "RentalOfferHolderTag::OnAfterRemove" };
const TString onBeforeUpdateSourceSession{ "RentalOfferHolderTag::OnBeforeUpdate" };
const TString onAfterUpdateSourceSession{ "RentalOfferHolderTag::OnAfterUpdate" };
const TString addStatusNotificationsSourceSession{ "AddRentalStatusChangedNotificationsTags" };
const TString processStatusChangedSourceSession{ "RentalOfferHolderTag::ProcessRentalStatusChanged" };
const TString paidOfferStatus{ "paid" };
const TString confirmedOfferStatus{ "confirmed" };
const TString draftOfferStatus{ "draft" };
const TString userAlreadyServiced{ "rental.book.user_is_already_being_serviced" };
const TString carAlreadyServiced{ "rental.book.car_is_already_being_serviced" };


bool AddRentalStatusChangedNotificationsTags(
    const TString& notificationType,
    const NDrive::IServer* server,
    NDrive::TEntitySession& session,
    const TString& userId,
    TAtomicSharedPtr<TRentalOffer> offer,
    NDrivematics::TUserOrganizationAffiliationTag::TDescriptionConstPtr& affilationTagDescription
) {
    Y_ENSURE(server);
    Y_ENSURE(offer);
    TString smsTagName;
    if (notificationType == TRentalOffer::BookingConfirmedNotificationType) {
        smsTagName = affilationTagDescription->GetBookingConfirmedSmsTagName();
    } else if (notificationType == TRentalOffer::BookingCancelledNotificationType) {
        smsTagName = affilationTagDescription->GetBookingCancelledSmsTagName();
    } else {
        return false;
    }

    auto driveApi = server->GetDriveAPI();

    TMaybe<TString> carmodel;
    // add email tag
    {
        auto emailData = offer->GetEmailData(server, session, notificationType);
        if (!emailData) {
            return false;
        }
        if (!emailData->EmailTagName.Empty()) {
            auto tag = driveApi->GetTagsManager().GetTagsMeta().CreateTag(emailData->EmailTagName);
            if (auto emailTag = std::dynamic_pointer_cast<TUserMailTag>(tag); emailTag) {
                emailTag->SetTemplateArgs(std::move(emailData->Args));
                auto& emailArgs = emailTag->MutableTemplateArgs();
                emailArgs["json_data"] = ToString(emailData->JsonData);
                carmodel = emailArgs[TRentalOffer::CarModelKeyword];

                if (!driveApi->GetTagsManager().GetUserTags().AddTag(tag, userId, offer->GetUserId(), server, session)) {
                    session.SetErrorInfo(addStatusNotificationsSourceSession,
                                         "cannot add " + emailData->EmailTagName + " to user " + offer->GetUserId());
                    return false;
                }
            } else {
                session.SetErrorInfo(addStatusNotificationsSourceSession,
                                     "cannot create tag " + emailData->EmailTagName);
                return false;
            }
        }
    }

    if (!smsTagName.Empty()) {
        if (!carmodel.Defined()) {
            const auto carsFetchResult = driveApi->GetCarsData()->FetchInfo(offer->GetObjectId(), session);
            if (!carsFetchResult) {
                session.SetErrorInfo(addStatusNotificationsSourceSession, "cannot fetch car");
                return false;
            }

            const auto carsInfo = carsFetchResult.GetResult();
            if (carsInfo.size() != 1) {
                session.SetErrorInfo(addStatusNotificationsSourceSession, "wrong carsInfo.size()");
                return false;
            }

            const auto& car = carsInfo.begin()->second;
            auto modelFetchResult = driveApi->GetModelsData()->GetCachedOrFetch(car.GetModel());
            if (!modelFetchResult) {
                session.SetErrorInfo(addStatusNotificationsSourceSession, "cannot fetch model");
                return false;
            }

            auto modelPtr = modelFetchResult.GetResultPtr(car.GetModel());
            if (!modelPtr) {
                session.SetErrorInfo(addStatusNotificationsSourceSession, "cannot find modelPtr");
                return false;
            }
            carmodel = modelPtr->GetName();
        }

        // add sms tag
        auto tag = driveApi->GetTagsManager().GetTagsMeta().CreateTag(smsTagName);
        if (auto smsTag = std::dynamic_pointer_cast<TUserMessageTagBase>(tag); nullptr != smsTag) {
            auto description = smsTag->GetDescriptionAs<TUserMessageTagBase::TDescription>(*server);
            if (!description) {
                session.SetErrorInfo(addStatusNotificationsSourceSession,
                                     "cannot restore sms tag description");
                return false;
            }
            auto messageText = description->GetMessageText();

            const auto dateFormat = offer->OptionalDateFormat().GetOrElse(TRentalOffer::DefaultDateFormat);
            const auto timeFormat = offer->OptionalTimeFormat().GetOrElse(TRentalOffer::DefaultTimeFormat);

            TRentalOffer::FillTimeParameter(TRentalOffer::SinceDateKeyword, offer->GetSince(), dateFormat, messageText, offer->OptionalOfferTimezone());
            TRentalOffer::FillTimeParameter(TRentalOffer::SinceTimeKeyword, offer->GetSince(), timeFormat, messageText, offer->OptionalOfferTimezone());

            TRentalOffer::FillTimeParameter(TRentalOffer::UntilDateKeyword, offer->GetUntil(), dateFormat, messageText, offer->OptionalOfferTimezone());
            TRentalOffer::FillTimeParameter(TRentalOffer::UntilTimeKeyword, offer->GetUntil(), timeFormat, messageText, offer->OptionalOfferTimezone());

            SubstGlobal(messageText, TRentalOffer::CompanynameKeyword, affilationTagDescription->GetCompanyName());
            SubstGlobal(messageText, TRentalOffer::PickupLocationKeyword, offer->OptionalDeliveryLocationName().GetOrElse(""));
            SubstGlobal(messageText, TRentalOffer::CarModelKeyword, carmodel.GetOrElse(""));

            const auto& companyInfo = affilationTagDescription->GetCompanyInformation();
            for (const auto& [key, value]: companyInfo) {
                SubstGlobal(messageText, key, value);
            }

            smsTag->SetMessageText(messageText);

            if (!driveApi->GetTagsManager().GetUserTags().AddTag(tag, userId, offer->GetUserId(), server, session)) {
                session.SetErrorInfo(addStatusNotificationsSourceSession,
                                     "cannot add " + smsTagName + " to user " + offer->GetUserId());
                return false;
            }
        } else {
            session.SetErrorInfo(addStatusNotificationsSourceSession,
                                 "cannot create tag " + smsTagName);
            return false;
        }
    }

    return true;
}
}

TOptionalDBTags TRentalOfferHolderTag::RestoreOfferHolderTags(const TString& behaviourConstructorId, const NDrive::IServer& server, NDrive::TEntitySession& session) {
    const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
    auto registeredTagNames = tagsMeta.GetRegisteredTagNames({ Type() });
    auto tagNames = MakeVector(registeredTagNames);
    return TOfferHolderUserTag::RestoreOfferHolderTags(behaviourConstructorId, tagNames, server, session);
}

const TString& TRentalOfferHolderTag::GetStage(const TTaggedObject* /*object*/) const {
    return TChargableTag::Prereservation;
}

bool TRentalOfferHolderTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_ENSURE(server);
    if (!TBase::OnBeforeAdd(objectId, userId, server, session)) {
        return false;
    }
    auto offer = std::dynamic_pointer_cast<TRentalOffer>(GetOffer());
    if (!offer) {
        session.SetErrorInfo(onBeforeAddSourceSession, "no rental offer provided");
        return false;
    }
    // will be deleted after fronted update
    if (server->GetSettings().GetValue<bool>("rental.is_old_start_riding_flow").GetOrElse(true)) {
        if (paidOfferStatus == ToLowerUTF8(offer->GetStatusRef())) {
            session.SetErrorInfo(onBeforeAddSourceSession, "cannot create paid offer");
            return false;
        }
    }
    return true;
}

bool TRentalOfferHolderTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterAdd(self, userId, server, session)) {
        return false;
    }
    auto offer = std::dynamic_pointer_cast<TRentalOffer>(GetOffer());
    return ProcessRentalStatusChanged(offer, draftOfferStatus, server, session, userId);
}

bool TRentalOfferHolderTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeRemove(self, userId, server, session)) {
        return false;
    }
    auto offer = std::dynamic_pointer_cast<TRentalOffer>(GetOffer());
    if (!offer) {
        session.SetErrorInfo(onBeforeRemoveSourceSession, "no rental offer provided");
        return false;
    }
    const auto offerLock = LockOffer(offer, session);
    if (offerLock.Empty() || !*offerLock) {
        return false;
    }

    if (!CheckTagExists(self, server, session)) {
        session.SetErrorInfo(onBeforeRemoveSourceSession, "cannot remove RentalOfferHolderTag, tag does not exists");
        return false;
    }
    return true;
}

bool TRentalOfferHolderTag::OnAfterRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnAfterRemove(self, userId, server, session)) {
        return false;
    }

    const auto api = Yensured(server)->GetDriveAPI();
    auto offer = std::dynamic_pointer_cast<TRentalOffer>(GetOffer());
    if (!offer) {
        session.SetErrorInfo(onAfterRemoveSourceSession, "null offer");
        return false;
    }

    {
        auto billingSession = CreateOfferHolderSession(self, TOfferHolderSessionOptions{}, api->GetTagsManager().GetUserTags(), session);
        if (!billingSession) {
            session.AddErrorMessage(onAfterRemoveSourceSession, "cannot create billing session from " + self.GetTagId());
            return false;
        }

        auto permissions = api->GetUserPermissions(self.GetObjectId(), {});
        if (!permissions) {
            session.SetErrorInfo(onAfterRemoveSourceSession, "cannot create permissions for " + self.GetObjectId());
            return false;
        }
        session.Committed().Subscribe([api, billingSession, permissions, server](const NThreading::TFuture<void>& w) {
            if (!w.HasValue()) {
                return;
            }
            auto closed = Yensured(api)->CloseSession(billingSession, *Yensured(permissions), *server);
            if (!closed) {
                NDrive::TEventLog::Log("CloseSessionError", NJson::TMapBuilder
                    ("session", billingSession ? billingSession->GetDebugInfo() : NJson::TJsonValue())
                    ("error", NJson::GetExceptionInfo(closed.GetError()))
                );
            }
        });
    }

    return ProcessRentalStatusChanged(offer, PreviousOfferStatus, server, session, userId, true);
}

bool TRentalOfferHolderTag::OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!TBase::OnBeforeUpdate(self, to, userId, server, session)) {
        return false;
    }

    auto target = std::dynamic_pointer_cast<TRentalOfferHolderTag>(to);
    if (!target) {
        session.SetErrorInfo(onBeforeUpdateSourceSession, "cannot cast target tag to RentalOfferHolderTag");
        return false;
    }
    if (!target->GetOffer()) {
        target->SetOffer(GetOffer());
    } else {
        session.SetErrorInfo(onBeforeUpdateSourceSession, "target tag has offer");
        return false;
    }

    auto offer = std::dynamic_pointer_cast<TRentalOffer>(target->GetOffer());
    const auto offerLock = LockOffer(offer, session);
    if (offerLock.Empty() || !*offerLock) {
        return false;
    }

    if (!CheckTagExists(self, server, session)) {
        session.SetErrorInfo(onBeforeUpdateSourceSession, "cannot update RentalOfferHolderTag, tag does not exists");
        return false;
    }

    if (offer->HasStatus()) {
        target->PreviousOfferStatus = ToLowerUTF8(offer->GetStatusRef());
    }
    if (!offer->PatchOffer(target->GetOfferUpdateData(), session, server)) {
        return false;
    }

    session.Committed().Subscribe([offer](const NThreading::TFuture<void>& c) {
        if (c.HasValue() && offer) {
            auto serializedOffer = offer->SerializeToProto();
            NDrive::TEventLog::Log("OfferUpdated", NJson::ToJson(NJson::Proto(serializedOffer)));
        }
    });
    return true;
}

TMaybe<bool> TRentalOfferHolderTag::LockOffer(TAtomicSharedPtr<TRentalOffer> offer, NDrive::TEntitySession& session) {
    const auto objectName = "rental_offer_" + offer->GetOfferId();
    const auto lockState = session.TryLock(objectName);
    if (lockState.Empty()) {
        session.SetErrorInfo("RentalOfferHolderTag::LockOffer", "TryLock failed: rental_offer_" + offer->GetOfferId());
        return {};
    }

    if (!lockState.GetRef()) {
        session.SetErrorInfo("RentalOfferHolderTag::LockOffer", objectName + " is already locked");
        return false;
    }
    return true;
}

bool TRentalOfferHolderTag::OnAfterUpdate(const TDBTag& self, ITag::TPtr toTag, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    Y_ENSURE(server);
    if (!TBase::OnAfterUpdate(self, toTag, userId, server, session)) {
        return false;
    }

    auto offer = std::dynamic_pointer_cast<TRentalOffer>(GetOffer());
    if (!ProcessRentalStatusChanged(offer, PreviousOfferStatus, server, session, userId)) {
        return false;
    }

    // will be deleted after frontend update
    if (server->GetSettings().GetValue<bool>("rental.is_old_start_riding_flow").GetOrElse(true)) {
        if (paidOfferStatus == ToLowerUTF8(offer->GetStatusRef()) && !ProcessPaidOffer(self, userId, server, session, onAfterUpdateSourceSession, offer)) {
            return false;
        }
    }

    return true;
}

bool TRentalOfferHolderTag::ProcessRentalStatusChanged(TAtomicSharedPtr<TRentalOffer> offer,
                                                       const TString& previousStatus,
                                                       const NDrive::IServer* server,
                                                       NDrive::TEntitySession& session,
                                                       const TString& userId,
                                                       bool rentalCancelled) {
    Y_ENSURE(server);
    Y_ENSURE(offer);
    const auto& clientId = offer->GetUserId();

    auto optionalObject = server->GetDriveDatabase().GetTagsManager().GetUserTags().GetCachedOrRestoreObject(clientId, session);
    if (!optionalObject) {
        session.SetErrorInfo(processStatusChangedSourceSession, "cannot restore object " + clientId);
        return false;
    }

    const auto& taggedObject = *optionalObject;
    auto dbTag = taggedObject.GetFirstTagByClass<NDrivematics::TUserOrganizationAffiliationTag>();
    if (!dbTag) {
        return true;
    }

    auto description = server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetDescriptionByName(dbTag->GetName());
    if (!description) {
        session.SetErrorInfo(processStatusChangedSourceSession, "cannot find description " + dbTag->GetName());
        return false;
    }

    auto affilationTagDescription = std::dynamic_pointer_cast<const NDrivematics::TUserOrganizationAffiliationTag::TDescription>(description);
    if (affilationTagDescription && offer->HasStatus()) {
        const auto offerStatus = ToLowerUTF8(offer->GetStatusRef());
        if (confirmedOfferStatus == offerStatus && draftOfferStatus == previousStatus) {
            return AddRentalStatusChangedNotificationsTags(TRentalOffer::BookingConfirmedNotificationType, server, session, userId, offer, affilationTagDescription);
        } else if (rentalCancelled && (confirmedOfferStatus == offerStatus || paidOfferStatus == offerStatus)) {
            return AddRentalStatusChangedNotificationsTags(TRentalOffer::BookingCancelledNotificationType, server, session, userId, offer, affilationTagDescription);
        }
    }

    return true;
}

bool TRentalOfferHolderTag::LockCar(TAtomicSharedPtr<TRentalOffer> offer, NDrive::TEntitySession& session) {
    return TRentalOfferHolderTag::LockCar(offer->GetObjectId(), session);
}

bool TRentalOfferHolderTag::LockUser(const TString& userId, NDrive::TEntitySession& session) {
    auto lockState = session.TryLock("rental_lock_user_" + userId);
    if (lockState.Empty()) {
        ythrow yexception() << "TryLock failed: rental_lock_user_" + userId;
    } else if (!lockState.GetRef()) {
        session.SetErrorInfo("RentalOfferHolderTag::LockUser",
                             "rental_lock_user_" + userId + " is already locked",
                             NDrive::MakeError(userAlreadyServiced));
        return false;
    }
    return true;
}

TRentalOfferHolderTag::TRentalTagNames TRentalOfferHolderTag::GetRentalTagNamesByPermissions(const TUserPermissions& permissions) {
    const auto tagsDescriptions = permissions.GetTagsByAction(NTagActions::ETagAction::Observe);

    TSet<TString> userTagNames;
    TSet<TString> carTagNames;
    for (const auto& tagDescription: tagsDescriptions) {
        if (tagDescription->GetType() == TRentalOfferHolderTag::Type()) {
            userTagNames.insert(tagDescription->GetName());
        }  else if (tagDescription->GetType() == TRentalServiceModeTag::Type()) {
            carTagNames.insert(tagDescription->GetName());
        }
    }
    return {std::move(userTagNames), std::move(carTagNames)};
}

bool TRentalOfferHolderTag::LockCar(const TString& carId, NDrive::TEntitySession& session) {
    auto lockState = session.TryLock("rental_lock_car_" + carId);
    if (lockState.Empty()) {
        ythrow yexception() << "TryLock failed: rental_lock_car_" + carId;
    } else if (!lockState.GetRef()) {
        session.SetErrorInfo("RentalOfferHolderTag::LockCar",
                             "rental_lock_car_" + carId + " is already locked",
                             NDrive::MakeError(carAlreadyServiced));
        return false;
    }
    return true;
}

bool TRentalOfferHolderTag::LockTwoCars(const TString& carId1, const TString& carId2, NDrive::TEntitySession& session) {
    auto lockTwoCars = [&session](const TString& carId1, const TString& carId2) {
        if (!TRentalOfferHolderTag::LockCar(carId1, session) || !TRentalOfferHolderTag::LockCar(carId2, session)) {
            return false;
        }
        return true;
    };

    if (carId1 == carId2 && !TRentalOfferHolderTag::LockCar(carId1, session)) {
        return false;
    } else if (carId1 < carId2 && !lockTwoCars(carId1, carId2)) {
        return false;
    } else if (!lockTwoCars(carId2, carId1)) {
        return false;
    }
    return true;
}

bool TRentalOfferHolderTag::CheckTagExists(const TDBTag& self, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    Y_ENSURE(server);
    auto driveApi = server->GetDriveAPI();
    auto restoredRentalOfferHolderTag = driveApi->GetTagsManager().GetUserTags().RestoreTag(self.GetTagId(), session);
    return restoredRentalOfferHolderTag && *restoredRentalOfferHolderTag && std::dynamic_pointer_cast<TRentalOfferHolderTag>(restoredRentalOfferHolderTag->GetData());
}

// will be deleted after frontend update
bool TRentalOfferHolderTag::ProcessPaidOffer(const TDBTag& selfContainer, const TString& userId, const NDrive::IServer* server,
                           NDrive::TEntitySession& session, const TString& sourceSessionMessage,
                           TAtomicSharedPtr<TRentalOffer> offer) const {
    Y_ENSURE(server);
    auto& tagsManager = server->GetDriveAPI()->GetTagsManager();
    auto newTag = tagsManager.GetTagsMeta().CreateTagAs<TOfferBookingUserTag>(TOfferBookingUserTag::Type());

    if (!newTag) {
        session.SetErrorInfo(sourceSessionMessage, "cannot build offer_booking_user_tag");
        return false;
    }

    newTag->SetCarId(offer->GetObjectId());
    auto permissions = server->GetDriveAPI()->GetUserPermissions(userId);

    if (!permissions) {
        session.SetErrorInfo(sourceSessionMessage, "cannot get user permissions for user_id=" + userId);
        return false;
    }

    auto evolution = permissions->GetEvolutionPtr(selfContainer->GetName(), newTag->GetName());
    if (!evolution) {
        session.SetErrorInfo(sourceSessionMessage, "no permissions to evolve " + selfContainer->GetName() + " into " + newTag->GetName());
        return false;
    }

    NDrive::ITag::TEvolutionContext eContext;
    auto evolved = tagsManager.GetUserTags().EvolveTag(selfContainer, newTag, *permissions, server, session, &eContext);
    if (!evolved) {
        session.SetErrorInfo(sourceSessionMessage, "cannot EvolveTag");
        return false;
    }
    return true;
}

bool TRentalOfferHolderTag::Start(
    const TDBTag& selfContainer,
    const TUserPermissions& permissions,
    const NDrive::IServer* server,
    NDrive::TEntitySession& session,
    const TString& sourceSessionMessage,
    const TString& carId
) {
    Y_ENSURE(server);
    auto& tagsManager = server->GetDriveAPI()->GetTagsManager();
    auto newTag = tagsManager.GetTagsMeta().CreateTagAs<TOfferBookingUserTag>(TOfferBookingUserTag::Type());

    if (!newTag) {
        session.SetErrorInfo(sourceSessionMessage, "cannot build offer_booking_user_tag");
        return false;
    }

    newTag->SetCarId(carId);

    auto evolution = permissions.GetEvolutionPtr(selfContainer->GetName(), newTag->GetName());
    if (!evolution) {
        session.SetErrorInfo(sourceSessionMessage, "no permissions to evolve " + selfContainer->GetName() + " into " + newTag->GetName());
        return false;
    }

    NDrive::ITag::TEvolutionContext eContext;
    auto evolved = tagsManager.GetUserTags().EvolveTag(selfContainer, newTag, permissions, server, session, &eContext);
    if (!evolved) {
        return false;
    }
    return true;
}

TString TRentalOfferHolderTag::GetPaidOfferStatus() {
    return paidOfferStatus;
}

bool TRentalOfferHolderTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    OfferUpdateData = value;
    return TBase::DoSpecialDataFromJson(value, errors);
}

ITag::TFactory::TRegistrator<TRentalOfferHolderTag> TRentalOfferHolderTag::Registrator(TRentalOfferHolderTag::Type());

TTagDescription::TFactory::TRegistrator<TRentalOfferHolderTag::TDescription> RentalOfferHolderTagDescriptionRegistrator(TRentalOfferHolderTag::Type());
