#include "rental_service_mode_tag.h"

#include "rental_offer_holder_tag.h"
#include "timetable_builder.h"

#include <rtline/library/json/field.h>

namespace {
const TString onBeforeAddSourceSession = "RentalServiceModeTag::OnBeforeAdd";
const TString onBeforeEvolveSourceSession = "RentalServiceModeTag::OnBeforeUpdate";
}

bool TRentalServiceModeTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& session) {
    if (!TBase::OnBeforeAdd(objectId, userId, server, session)) {
        return false;
    }

    if (HasUntil() && Since >= *Until) {
        session.SetErrorInfo(onBeforeAddSourceSession, "wrong since/until");
        return false;
    }

    if (!TRentalOfferHolderTag::LockCar(objectId, session)) {
        return false;
    }

    const auto api = Yensured(server)->GetDriveAPI();
    auto permissions = api->GetUserPermissions(userId, {});
    if (!permissions) {
        session.SetErrorInfo(onBeforeAddSourceSession, "cannot create permissions for " + userId);
        return false;
    }
    const auto rentalEvents = TTimetableBuilder::Instance().GetTimetableEvents(*permissions, server, session);
    if (!rentalEvents) {
        return false;
    }

    TMap<TString, TMap<TString, TTimetableEventMetadata>> carsTimetable;
    const auto sinceTimetable = Since;
    const auto untilTimetable = Until.GetOrElse(TInstant::FromValue(std::numeric_limits<ui64>::max()));
    if (!TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable,
                                                                                                 rentalEvents->first,
                                                                                                 rentalEvents->second,
                                                                                                 sinceTimetable,
                                                                                                 untilTimetable,
                                                                                                 *permissions,
                                                                                                 server,
                                                                                                 session)) {
        session.SetErrorInfo(onBeforeAddSourceSession, "cannot build timetable");
        return false;
    }

    if (auto conflictStatus = TTimetableBuilder::Instance().HasBookingTimetableConflicts(carsTimetable, objectId, sinceTimetable, server, session); conflictStatus.first) {
        session.SetErrorInfo(onBeforeAddSourceSession, conflictStatus.second, NDrive::MakeError(conflictStatus.second));
        return false;
    }
    return true;
}

bool TRentalServiceModeTag::OnBeforeEvolve(const TDBTag& self, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session, const TEvolutionContext* evolutionContext) const {
    if (!TBase::OnBeforeEvolve(self, to, permissions, server, session, evolutionContext)) {
        return false;
    }
    auto target = std::dynamic_pointer_cast<TRentalServiceModeTag>(to);
    if (!target) {
        session.SetErrorInfo(onBeforeEvolveSourceSession, "cannot cast target tag to RentalServiceModeTag");
        return false;
    }

    if (!TRentalOfferHolderTag::LockCar(self.GetObjectId(), session)) {
        return false;
    }

    if (target->HasUntil() && target->GetSince() >= *target->OptionalUntil()) {
        session.SetErrorInfo(onBeforeEvolveSourceSession, "wrong since/until");
        return false;
    }

    const auto rentalEvents = TTimetableBuilder::Instance().GetTimetableEvents(permissions, server, session);
    if (!rentalEvents) {
        return false;
    }

    TMap<TString, TMap<TString, TTimetableEventMetadata>> carsTimetable;
    const auto sinceTimetable = target->GetSince();
    const auto untilTimetable = target->OptionalUntil().GetOrElse(TInstant::FromValue(std::numeric_limits<ui64>::max()));
    if (!TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable,
                                                                                                 rentalEvents->first,
                                                                                                 rentalEvents->second,
                                                                                                 sinceTimetable,
                                                                                                 untilTimetable,
                                                                                                 permissions,
                                                                                                 server,
                                                                                                 session)) {
        session.SetErrorInfo(onBeforeEvolveSourceSession, "cannot build timetable");
        return false;
    }

    if (auto carOfferIt = carsTimetable.find(self.GetObjectId()); carsTimetable.end() != carOfferIt) {
        carOfferIt->second.erase(self.GetTagId());
        if (carOfferIt->second.empty()) {
            carsTimetable.erase(carOfferIt);
        }
    }

    if (auto conflictStatus = TTimetableBuilder::Instance().HasBookingTimetableConflicts(carsTimetable, self.GetObjectId(), sinceTimetable, server, session); conflictStatus.first) {
        session.SetErrorInfo(onBeforeAddSourceSession, conflictStatus.second, NDrive::MakeError(conflictStatus.second));
        return false;
    }

    return true;
}

TRentalServiceModeTag::TProto TRentalServiceModeTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetJobType(JobType);
    proto.SetJobComment(JobComment);
    proto.SetSince(Since.MicroSeconds());
    proto.SetMileage(Mileage);
    if (Until) {
        proto.SetUntil(Until->MicroSeconds());
    }
    return proto;
}

bool TRentalServiceModeTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    if (!TBase::DoDeserializeSpecialDataFromProto(proto)) {
        return false;
    }
    if (proto.HasJobType()) {
        JobType = proto.GetJobType();
    }
    if (proto.HasJobComment()) {
        JobComment = proto.GetJobComment();
    }
    if (proto.HasSince()) {
        Since = TInstant::MicroSeconds(proto.GetSince());
    }
    if (proto.HasUntil()) {
        Until = TInstant::MicroSeconds(proto.GetUntil());
    }
    if (proto.HasMileage()) {
        Mileage = proto.GetMileage();
    }
    return true;
}

void TRentalServiceModeTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::FieldsToJson(json, GetFields());
    if (Until) {
        json["until"] = Until->MicroSeconds();
    }
}

bool TRentalServiceModeTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(json, errors)) {
        return false;
    }
    if (json.Has("until")) {
        Until = TInstant::MicroSeconds(json["until"].GetUInteger());
    }
    return NJson::TryFieldsFromJson(json, GetFields());
}

ITag::TFactory::TRegistrator<TRentalServiceModeTag> TRentalServiceModeTag::Registrator(TRentalServiceModeTag::Type());
TTagDescription::TFactory::TRegistrator<TRentalServiceModeTag::TDescription> TRentalServiceModeTag::DescriptionRegistrator(TRentalServiceModeTag::Type());
