#include "timetable_builder.h"

#include "rental_service_mode_tag.h"
#include "timetable_builder_impl.h"

#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tag.h>

namespace {
constexpr TDuration year = TDuration::Days(365);
const TString userAlreadyHasBooking{ "rental.book.user_already_has_booking" };
const TString carIsAlreadyUsedByAnotherBooking{ "rental.book.car_is_already_used_by_another_booking" };
const TString getActualDatesSourceSession{ "TTimetableBuilder::GetActualDatesFromSession" };
const TString buildTimetableFromSessionsSourceSession{ "TTimetableBuilder::BuildTimetableFromSessions" };
const TString buildTimetableFromCompiledRidesSourceSession{ "TTimetableBuilder::BuildTimetableFromCompiledRides" };
const TString getTimetableEventsSourceSession{ "TTimetableBuilder::GetTimetableEvents" };
const TString fetchSessionOfferMetadataSourceSession{ "TTimetableBuilder::FetchSessionOfferMetadata" };

template<class OfferType>
TTimetableEventMetadata GetEventMetadata(OfferType& offer) {
    TTimetableEventMetadata event;
    event.OfferId = offer.GetOfferId();
    event.Since = offer.GetSince();
    event.Until = offer.GetUntil();
    if constexpr (std::is_same<TRentalOffer, OfferType>::value) {
        Y_ENSURE(offer.OptionalStatus());
        event.Status = offer.GetStatusRef();
    }
    event.TagId = "";
    event.UserId = offer.GetUserId();
    event.Stage = TChargableTag::Prereservation;
    return event;
}

bool FillActualDatesFromCompiledRide(TMap<TString, TTimetableEventMetadata *>& offerMetadataPointers,
                                     NDrive::TEntitySession& tx,
                                     const NDrive::IServer* server) {
    TVector<TString> offerIdsWithoutSession;
    for (const auto& [offerId, offerMetadataPointer]: offerMetadataPointers) {
        offerIdsWithoutSession.push_back(offerId);
    }
    auto ydbTx = server->GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("fill_actual_dates", server);
    auto optionalSessions = server->GetDriveAPI()->GetMinimalCompiledRides().Get<TFullCompiledRiding>(offerIdsWithoutSession, tx, ydbTx);
    if (!optionalSessions) {
        auto errorString =  TStringBuilder() << "cannot get full compiled ride ";
        for (const auto& offerId: offerIdsWithoutSession) {
            errorString << offerId  << ", ";
        }
        tx.AddErrorMessage(getActualDatesSourceSession, errorString);
        return false;
    }

    for (const auto& session: *optionalSessions) {
        if (auto offerMetadataPointersIt = offerMetadataPointers.find(session.GetSessionId()); offerMetadataPointers.end()!= offerMetadataPointersIt) {
            auto pOfferMetadata = offerMetadataPointersIt->second;
            if (!pOfferMetadata) {
                tx.AddErrorMessage(getActualDatesSourceSession, "nullptr offer metadata " + session.GetSessionId());
                return false;
            }
            bool acceptanceFound = false;
            for (const auto& event: session.GetLocalEvents()) {
                if (!acceptanceFound && TChargableTag::Acceptance == event.GetTagName()) {
                    pOfferMetadata->ActualSince = event.GetInstant();
                    acceptanceFound = true;
                } else if (EObjectHistoryAction::DropTagPerformer == event.GetHistoryAction()) {
                    pOfferMetadata->ActualUntil = event.GetInstant();
                }
            }
            pOfferMetadata->Stage.clear();
        }
    }
    return true;
}

template<class OfferType>
bool BuildTimetableFromSessions(TTimetableBuilder::TCarsTimetable& carsTimetable,
                                TSet<TString>& fetchedOfferIds, TInstant sinceTimetable,
                                TInstant untilTimetable,  const TUserPermissions& permissions,
                                const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    auto driveApi = server->GetDriveAPI();

    Y_ENSURE(driveApi);
    auto sessionsBuilder = driveApi->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
    Y_ENSURE(sessionsBuilder);
    auto sessions = sessionsBuilder->GetSessionsActual();
    const auto nowTime = Now();
    for (auto& session: sessions) {
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(session);
        if (!billingSession) {
            continue;
        }
        auto offer = billingSession->GetCurrentOffer();
        if (!offer) {
            continue;
        }

        auto rentalOffer = std::dynamic_pointer_cast<OfferType>(offer);
        if (!rentalOffer) {
            continue;
        }

        const auto& carId = rentalOffer->GetObjectId();
        TMaybe<TTaggedObject> taggedObject = driveApi->GetTagsManager().GetDeviceTags().GetCachedOrRestoreObject(carId, tx);
        if (!taggedObject) {
            return false;
        }
        if (permissions.GetVisibility(*taggedObject, NEntityTagsManager::EEntityType::Car, nullptr, false) == TUserPermissions::EVisibility::NoVisible) {
            continue;
        }
        auto eventMetadata = GetEventMetadata(*rentalOffer);

        TBillingSession::TBillingEventsCompilation eventsCompilation;
        if (!billingSession->FillCompilation(eventsCompilation)) {
            tx.AddErrorMessage(buildTimetableFromSessionsSourceSession, "cannot fill BillingEventsCompilation " + offer->GetOfferId());
            return false;
        }

        fetchedOfferIds.insert(eventMetadata.OfferId);
        bool acceptanceFound = false;
        for (const auto& event: eventsCompilation.GetEvents()) {
            if (!acceptanceFound && TChargableTag::Acceptance == event.GetName()) {
                eventMetadata.ActualSince = event.GetInstant();
                acceptanceFound = true;
            } else if (EObjectHistoryAction::DropTagPerformer == event.GetAction()) {
                eventMetadata.ActualUntil = event.GetInstant();
            }
        }

        const auto actualSince = eventMetadata.ActualSince ? *eventMetadata.ActualSince : eventMetadata.Since;
        TInstant actualUntil;
        if (eventMetadata.ActualUntil) {
            actualUntil = *eventMetadata.ActualUntil;
        } else if (eventMetadata.Until < nowTime) {
            actualUntil = nowTime;
        } else {
            actualUntil = eventMetadata.Until;
        }

        const auto bookingDatesOutsideTimetable = eventMetadata.Since >= untilTimetable || eventMetadata.Until <= sinceTimetable;
        const auto actualDatesOutsideTimetable = actualSince >= untilTimetable || actualUntil <= sinceTimetable;
        if (bookingDatesOutsideTimetable && actualDatesOutsideTimetable) {
            continue;
        }
        auto chargableTag = taggedObject->GetFirstTagByClass<TChargableTag>();
        if (chargableTag) {
            eventMetadata.Stage = chargableTag->GetName();
        }

        carsTimetable[carId][eventMetadata.OfferId] = eventMetadata;
    }
    return true;
}

template<class TOfferType>
bool BuildTimetableFromCompiledRides(TTimetableBuilder::TCarsTimetable& carsTimetable,
                                     const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    TMap<TString, TTimetableEventMetadata *> eventMetadataPointers;
    for (auto& [carId, events]: carsTimetable) {
        for (auto& [sessionId, eventMetadata]: events) {
            eventMetadataPointers[sessionId] = &eventMetadata;
        }
    }

    if (!eventMetadataPointers.empty()) {
        return FillActualDatesFromCompiledRide(eventMetadataPointers, tx, server);
    }

    return true;
}

void ReplaceDatesByActual(TTimetableBuilder::TCarsTimetable& carsTimetable, const TInstant sinceTimetable, const TInstant untilTimetable) {
    for (auto& [carId, events]: carsTimetable) {
        for (auto eventsIt = events.begin(); eventsIt != events.end();) {
            auto& eventMetadata = eventsIt->second;

            if (eventMetadata.ActualSince) {
                eventMetadata.Since = *eventMetadata.ActualSince;
            }
            if (eventMetadata.ActualUntil) {
                eventMetadata.Until = *eventMetadata.ActualUntil;
            }

            if (eventMetadata.Since >= untilTimetable || eventMetadata.Until <= sinceTimetable) {
                eventsIt = events.erase(eventsIt);
            } else {
                ++eventsIt;
            }
        }
    }

    for (auto carsTimetableIt = carsTimetable.begin(); carsTimetableIt != carsTimetable.end();) {
        if (carsTimetableIt->second.empty()) {
            carsTimetableIt = carsTimetable.erase(carsTimetableIt);
        } else {
            ++carsTimetableIt;
        }
    }
}
}

TTimetableBuilder& TTimetableBuilder::Instance() {
    static TTimetableBuilder instance;
    return instance;
}

bool TTimetableBuilder::BuildLongTermTimetable(TCarsTimetable& carsTimetable, const TEvents& userEvents,
                                               const TEvents& carEvents, const TInstant sinceTimetable,
                                               const TInstant untilTimetable, const TUserPermissions& permissions,
                                               const NDrive::IServer* server, NDrive::TEntitySession& tx,
                                               bool replaceDatesByActual) const {
    TSet<TString> fetchedOfferIds;
    if (!BuildTimetableFromSessions<TLongTermOffer>(carsTimetable, fetchedOfferIds, sinceTimetable, untilTimetable, permissions, server, tx)) {
        return false;
    }
    ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fetchedOfferIds);
    if (!BuildTimetableFromCompiledRides<TLongTermOffer>(carsTimetable, server, tx)) {
        return false;
    }
    if (replaceDatesByActual) {
        ReplaceDatesByActual(carsTimetable, sinceTimetable, untilTimetable);
    }
    return true;
}

bool TTimetableBuilder::BuildRentalTimetable(TCarsTimetable& carsTimetable, const TEvents& userEvents,
                                             const TEvents& carEvents, const TInstant sinceTimetable,
                                             const TInstant untilTimetable, const TUserPermissions& permissions,
                                             const NDrive::IServer* server, NDrive::TEntitySession& tx,
                                             bool replaceDatesByActual) const {
    TSet<TString> fetchedOfferIds;
    if (!BuildTimetableFromSessions<TRentalOffer>(carsTimetable, fetchedOfferIds, sinceTimetable, untilTimetable, permissions, server, tx)) {
        return false;
    }
    ProcessTimetable<TRentalOfferHolderTag, TRentalOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fetchedOfferIds);
    if (!BuildTimetableFromCompiledRides<TRentalOffer>(carsTimetable, server, tx)) {
        return false;
    }
    if (replaceDatesByActual) {
        ReplaceDatesByActual(carsTimetable, sinceTimetable, untilTimetable);
    }
    return true;
}

std::pair<bool, TString> TTimetableBuilder::HasBookingTimetableConflicts(
    TCarsTimetable& carsTimetable, const TString& carId, const TInstant since, const NDrive::IServer* server,
    NDrive::TEntitySession& session, TMaybe<TStringBuf> userId, bool checkRiding) {
    if (userId) {
        for (const auto& [carId, events]: carsTimetable) {
            for (const auto& [until, eventMetadata]: events) {
                if (eventMetadata.UserId == *userId) {
                    return {true, userAlreadyHasBooking};
                }
            }
        }
    }

    if (!carsTimetable[carId].empty()) {
        return {true, carIsAlreadyUsedByAnotherBooking};
    }

    if (checkRiding) {
        const auto nowTime = TInstant::Now();
        if (nowTime + TDuration::Minutes(1) > since) {
            auto carObject = server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreObject(carId, session);
            if (!carObject) {
                return {true, "cannot restore car"};
            }

            auto chargableDBTag = carObject->GetFirstTagByClass<TChargableTag>();
            if (chargableDBTag) {
                auto chargableCarTag = chargableDBTag.GetTagAs<const TChargableTag>();
                if (chargableCarTag->HasPerformer()) {
                    return {true, carIsAlreadyUsedByAnotherBooking};
                }
            }
        }
    }
    return {false, ""};
}

TTimetableBuilder::TOptionalEvents TTimetableBuilder::GetTimetableEvents(const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    TOptionalEvents result;
    const auto [userTagNames, carTagNames] = TRentalOfferHolderTag::GetRentalTagNamesByPermissions(permissions);
    const auto& userTagManager = server->GetDriveDatabase().GetTagsManager().GetUserTags();
    TTagEventsManager::TQueryOptions queryOptions;
    const auto nowTime = TInstant::Now();
    TOptionalTagHistoryEvents userTagEvents, carTagEvents;
    if (!userTagNames.empty()) {
        queryOptions.SetTags(userTagNames);
        queryOptions.SetOrderBy(TVector<TString>{"history_timestamp"});

        userTagEvents = userTagManager.GetEvents({}, {nowTime - year}, session, queryOptions);
        if (!userTagEvents) {
            session.SetErrorInfo(getTimetableEventsSourceSession, "cannot get events");
            return result;
        }
    }

    if (!carTagNames.empty()) {
        queryOptions.SetTags(carTagNames);
        queryOptions.SetOrderBy(TVector<TString>{"history_timestamp"});

        carTagEvents = server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetEvents({}, {nowTime - year}, session, queryOptions);
        if (!carTagEvents) {
            session.SetErrorInfo(getTimetableEventsSourceSession, "cannot get events");
            return result;
        }
    }
    result = std::pair<TEvents, TEvents>{};
    if (userTagEvents) {
        result->first = std::move(*userTagEvents);
    }
    if (carTagEvents) {
        result->second = std::move(*carTagEvents);
    }

    return result;
}
