#pragma once

#include <drive/backend/database/history/event.h>
#include <drive/backend/tags/tags.h>

#include <util/datetime/base.h>
#include <util/generic/fwd.h>
#include <util/generic/maybe.h>
#include <util/generic/vector.h>

struct TTimetableEventMetadata {
    TString OfferId;
    TInstant Since;
    TInstant Until;
    TString UserId;
    TString TagId;
    TString Status;
    TMaybe<TInstant> ActualSince;
    TMaybe<TInstant> ActualUntil;
    TString Stage;
};

class TTimetableBuilder {
public:
    enum class ETimetableType {
        LongTerm,
        Rental
    };

    struct TActualDatesFromSession {
        bool SessionExists = false;
        TMaybe<TInstant> Since;
        TMaybe<TInstant> Until;
    };

    using TCarsTimetable = TMap<TString, TMap<TString, TTimetableEventMetadata>>;
    using TEvents = TVector<TObjectEvent<TConstDBTag>>;
    using TOptionalEvents = TMaybe<std::pair<TEvents, TEvents>>;
    using TOptionalActualDates = TMaybe<TActualDatesFromSession>;

public:
    static TTimetableBuilder& Instance();
    TTimetableBuilder(const TTimetableBuilder&) = delete;
    TTimetableBuilder& operator=(const TTimetableBuilder&) = delete;

    template <ETimetableType TimetableType>
    bool BuildTimetable(TCarsTimetable& carsTimetable, const TEvents& userEvents,
                        const TEvents& carEvents, const TInstant sinceTimetable,
                        const TInstant untilTimetable, const TUserPermissions& permissions,
                        const NDrive::IServer* server, NDrive::TEntitySession& tx,
                        bool replaceDatesByActual = true) const {
        if constexpr (TimetableType == ETimetableType::LongTerm) {
            return BuildLongTermTimetable(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, permissions, server, tx, replaceDatesByActual);
        } else {
            return BuildRentalTimetable(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, permissions, server, tx, replaceDatesByActual);
        }
    }

    std::pair<bool, TString> HasBookingTimetableConflicts(TCarsTimetable& carsTimetable, const TString& carId, const TInstant since,
                                                         const NDrive::IServer* server, NDrive::TEntitySession& session, TMaybe<TStringBuf> userId = {},
                                                         bool checkRiding = true);
    TOptionalEvents GetTimetableEvents(const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& session) const;

private:
    bool BuildLongTermTimetable(TCarsTimetable& carsTimetable, const TEvents& userEvents,
                                const TEvents& carEvents, const TInstant sinceTimetable,
                                const TInstant untilTimetable, const TUserPermissions& permissions,
                                const NDrive::IServer* server, NDrive::TEntitySession& tx,
                                bool replaceDatesByActual) const;

    bool BuildRentalTimetable(TCarsTimetable& carsTimetable, const TEvents& userEvents,
                              const TEvents& carEvents, const TInstant sinceTimetable,
                              const TInstant untilTimetable, const TUserPermissions& permissions,
                              const NDrive::IServer* server, NDrive::TEntitySession& tx,
                              bool replaceDatesByActual) const;

private:
    TTimetableBuilder() = default;

};
