#pragma once
#include "timetable_builder.h"

#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/tags/tag.h>

void ProcessCarChanged(auto& carsTimetable, auto& offerCarsMap, const auto& offerId, const auto& carId) {
    if (auto offerCarMapIt = offerCarsMap.find(offerId); offerCarsMap.end() != offerCarMapIt) {
        if (offerCarMapIt->second != carId) {
            auto carsTimetableIt = carsTimetable.find(offerCarMapIt->second);

            carsTimetableIt->second.erase(offerId);
            if (carsTimetableIt->second.empty()) {
                carsTimetable.erase(carsTimetableIt);
            }

            offerCarsMap.erase(offerCarMapIt);
        }
    }
}

void ProcessOfferRemoved(auto& carsTimetable, auto& offerCarsMap, const auto& offerId, const auto& carId) {
    if (auto carsTimetableIt = carsTimetable.find(carId); carsTimetable.end() != carsTimetableIt) {

        carsTimetableIt->second.erase(offerId);

        if (carsTimetableIt->second.empty()) {
            carsTimetable.erase(carsTimetableIt);
        }

        offerCarsMap.erase(offerId);
    }
}

void ProcessOffer(auto& carsTimetable, auto& offerCarsMap,
                  const TInstant sinceTimetable, const TInstant untilTimetable,
                  const TInstant sinceOffer, const TInstant untilOffer,
                  const TString& offerId, const TString& carId, const TString& userId,
                  const auto& event, const TString& offerStatus) {

    if (sinceOffer >= untilTimetable || untilOffer <= sinceTimetable) {
        ProcessOfferRemoved(carsTimetable, offerCarsMap, offerId, carId);
        return;
    }

    ProcessCarChanged(carsTimetable, offerCarsMap, offerId, carId);

    if (carId.empty()) {
        return;
    }

    if (event.GetHistoryAction() == EObjectHistoryAction::Remove) {
        ProcessOfferRemoved(carsTimetable, offerCarsMap, offerId, carId);
        return;
    }

    offerCarsMap[offerId] = carId;

    carsTimetable[carId][offerId] = TTimetableEventMetadata{offerId, sinceOffer, untilOffer, userId, event.GetTagId(), offerStatus, {}, {}, TChargableTag::Prereservation};
}

template<class HodlerTagType, class OfferType, class ServiceTagType>
void ProcessTimetable(auto& carsTimetable, auto& userEvents, auto& carEvents,
                      const TInstant sinceTimetable, const TInstant untilTimetable,
                      const TSet<TString>& fetchedOfferIds) {
    TMap<TString, TString> offerCarsMap;
    const TString holderStatus = "";
    const TString* pHolderStatus = &holderStatus;
    for (const auto& event: userEvents) {
        if (const auto offerHolderTag = dynamic_cast<const HodlerTagType*>(event.GetData().Get()); offerHolderTag != nullptr) {

            if (const auto offer = std::dynamic_pointer_cast<OfferType>(offerHolderTag->GetOffer());
                offer != nullptr) {

                const auto& offerId = offer->GetOfferId();
                if (fetchedOfferIds.find(offerId) != fetchedOfferIds.end()) {
                    continue;
                }

                if constexpr (std::is_same<TRentalOffer, OfferType>::value) {
                    pHolderStatus = &holderStatus;
                    if (offer->HasStatus()) {
                        pHolderStatus = offer->GetStatusPtr();
                    }
                }

                const auto sinceOffer = offer->GetSince();
                const auto untilOffer = offer->GetUntil();
                const auto& carId = offer->GetObjectId();
                const auto& userId = offer->GetUserId();
                ProcessOffer(carsTimetable, offerCarsMap,
                             sinceTimetable, untilTimetable, sinceOffer, untilOffer,
                             offerId, carId, userId, event, *pHolderStatus);
            }
        }
    }

    const auto unboundaryService = TInstant::FromValue(std::numeric_limits<ui64>::max());
    for (const auto& event: carEvents) {
        if (const auto serviceTag = dynamic_cast<const ServiceTagType*>(event.GetData().Get()); serviceTag != nullptr) {

            const auto sinceOffer = serviceTag->GetSince();
            auto untilOffer = unboundaryService;
            if (serviceTag->HasUntil()) {
                untilOffer = *serviceTag->OptionalUntil();
            }

            const auto& carId = event.GetObjectId();
            const auto& tagId = event.GetTagId();

            ProcessOffer(carsTimetable, offerCarsMap,
                         sinceTimetable, untilTimetable, sinceOffer,
                         untilOffer, tagId, carId, "", event, TRentalOffer::ServiceModeStatus);
        }
    }
}
