#include "scoring.h"

NJson::TJsonValue IScoringBaseTag::TScoringDetails::SerializeToJson() const {
    NJson::TJsonValue json;
    NJson::InsertField(json, "lateral_acceleration_cornering", LateralAccelerationCornering);
    NJson::InsertField(json, "lateral_acceleration_straight", LateralAccelerationStraight);
    NJson::InsertField(json, "high_engine_speed", HighEngineSpeed);
    NJson::InsertField(json, "harsh_braking", HarshBraking);
    NJson::InsertField(json, "sharp_acceleration", SharpAcceleration);
    NJson::InsertField(json, "sections_with_speeding", SectionsWithSpeeding);
    return json;
}

bool IScoringBaseTag::TScoringDetails::DeserializeFromJson(const NJson::TJsonValue& json) {
    return
        NJson::ParseField(json, "lateral_acceleration_cornering", LateralAccelerationCornering, false) &&
        NJson::ParseField(json, "lateral_acceleration_straight", LateralAccelerationStraight, false) &&
        NJson::ParseField(json, "high_engine_speed", HighEngineSpeed, false) &&
        NJson::ParseField(json, "harsh_braking", HarshBraking, false) &&
        NJson::ParseField(json, "sharp_acceleration", SharpAcceleration, false) &&
        NJson::ParseField(json, "sections_with_speeding", SectionsWithSpeeding, false);
}

NDrive::TScheme IScoringBaseTag::TScoringDetails::GetScheme() {
    NDrive::TScheme scheme;
    scheme.Add<TFSNumeric>("lateral_acceleration_cornering", "Боковое ускорение на повороте");
    scheme.Add<TFSNumeric>("lateral_acceleration_straight", "Боковое ускорение на прямой");
    scheme.Add<TFSNumeric>("high_engine_speed", "Высокие обороты двигателя");
    scheme.Add<TFSNumeric>("harsh_braking", "Резкое торможение");
    scheme.Add<TFSNumeric>("sharp_acceleration", "Резкое ускорение");
    scheme.Add<TFSNumeric>("sections_with_speeding", "Участки с превышением скорости");
    return scheme;
}

IScoringBaseTag::TScoringDetails::TProto IScoringBaseTag::TScoringDetails::SerializeToProto() const {
    TScoringDetails::TProto proto;
    proto.SetLateralAccelerationCornering(LateralAccelerationCornering);
    proto.SetLateralAccelerationStraight(LateralAccelerationStraight);
    proto.SetHighEngineSpeed(HighEngineSpeed);
    proto.SetHarshBraking(HarshBraking);
    proto.SetSharpAcceleration(SharpAcceleration);
    proto.SetSectionsWithSpeeding(SectionsWithSpeeding);
    return proto;
}

bool IScoringBaseTag::TScoringDetails::DeserializeFromProto(const IScoringBaseTag::TScoringDetails::TProto& proto) {
    LateralAccelerationCornering = proto.GetLateralAccelerationCornering();
    LateralAccelerationStraight = proto.GetLateralAccelerationStraight();
    HighEngineSpeed = proto.GetHighEngineSpeed();
    HarshBraking = proto.GetHarshBraking();
    SharpAcceleration = proto.GetSharpAcceleration();
    SectionsWithSpeeding = proto.GetSectionsWithSpeeding();
    return true;
}

NJson::TJsonValue IScoringBaseTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeMetaToJson();
    NJson::InsertField(json, "kind", ::ToString(Kind));
    return json;
}

bool IScoringBaseTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    if (!NJson::ParseField(json, "kind", NJson::Stringify(Kind))) {
        return false;
    }
    return TBase::DoDeserializeMetaFromJson(json);
}

NDrive::TScheme IScoringBaseTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSVariants>("kind", "Тип скоринга").InitVariants<EScoringKind>();
    return result;
}

EUniquePolicy IScoringBaseTag::GetUniquePolicy() const {
    return EUniquePolicy::Rewrite;
}

NDrive::TScheme IScoringBaseTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("value", "Значение скоринга");
    result.Add<TFSNumeric>("timestamp", "Время расчета скоринга")
        .SetVisual(NDrive::TFSNumeric::EVisualType::DateTime);
    result.Add<TFSNumeric>("previous_value", "Предыдущее значение скоринга");
    result.Add<TFSNumeric>("rank");
    result.Add<TFSStructure>("details", "Расшифровка скоринга")
        .SetStructure<NDrive::TScheme>(TScoringDetails::GetScheme());
    return result;
}

void IScoringBaseTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "value", Value);
    NJson::InsertField(json, "timestamp", Timestamp.Seconds());
    NJson::InsertField(json, "previous_value", PreviousValue);
    NJson::InsertNonNull(json, "rank", Rank);
    if (Details) {
        json["details"] = Details->SerializeToJson();
    }
}

bool IScoringBaseTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    RawData = json;
    if (!NJson::ParseField(json, "value", Value, false)) {
        return false;
    }
    if (!NJson::ParseField(json, "timestamp", Timestamp, false)) {
        return false;
    }
    if (!NJson::ParseField(json, "previous_value", PreviousValue, false)) {
        return false;
    }
    if (!NJson::ParseField(json, "rank", Rank)) {
        return false;
    }
    if (json.Has("details")) {
        Details.ConstructInPlace();
        if (!Details->DeserializeFromJson(json["details"])) {
            return false;
        }
    }
    return TBase::DoSpecialDataFromJson(json, errors);
}

IScoringBaseTag::TProto IScoringBaseTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    proto.SetValue(Value);
    proto.SetTimestamp(Timestamp.Seconds());
    if (PreviousValue) {
        proto.SetPreviousValue(*PreviousValue);
    }
    if (Rank) {
        proto.SetRankDouble(*Rank);
    }
    if (Details) {
        *proto.MutableDetails() = Details->SerializeToProto();
    }
    return proto;
}

bool IScoringBaseTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    Value = proto.GetValue();
    Timestamp = TInstant::Seconds(proto.GetTimestamp());
    if (proto.HasPreviousValue()) {
        PreviousValue = proto.GetPreviousValue();
    }
    if (proto.HasRankDouble()) {
        Rank = proto.GetRankDouble();
    } else if (proto.HasRank()) {
        Rank = proto.GetRank();
    }
    if (proto.HasDetails()) {
        Details.ConstructInPlace();
        if (!Details->DeserializeFromProto(proto.GetDetails())) {
            return false;
        }
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

const TString TScoringCarTag::TypeName = "scoring_car_tag";

TSet<NEntityTagsManager::EEntityType> TScoringCarTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::Car };
}

NDrive::ITag::TFactory::TRegistrator<TScoringCarTag> ScoringCarTagRegistrator(TScoringCarTag::TypeName);
TTagDescription::TFactory::TRegistrator<TScoringCarTag::TDescription> ScoringCarTagDescriptionRegistrator(TScoringCarTag::TypeName);

const TString TScoringUserTag::TypeName = "scoring_user_tag";

TSet<NEntityTagsManager::EEntityType> TScoringUserTag::GetObjectType() const {
    return { NEntityTagsManager::EEntityType::User };
}

NDrive::ITag::TFactory::TRegistrator<TScoringUserTag> ScoringUserTagRegistrator(TScoringUserTag::TypeName);
TTagDescription::TFactory::TRegistrator<TScoringUserTag::TDescription> ScoringUserTagDescriptionRegistrator(TScoringUserTag::TypeName);

NJson::TJsonValue TScoringTraceTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue json = TBase::DoSerializeMetaToJson();
    NJson::InsertField(json, "signal_enabled", SignalEnabled);
    return json;
}

bool TScoringTraceTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& json) {
    if (!NJson::ParseField(json["signal_enabled"], SignalEnabled)) {
        return false;
    }
    return TBase::DoDeserializeMetaFromJson(json);
}

NDrive::TScheme TScoringTraceTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSBoolean>("signal_enabled", "Enable drivematics signals");
    return result;
}

const TString TScoringTraceTag::TypeName = "scoring_trace_tag";

bool TScoringTraceTag::AddEvent(TEvent&& ev) {
    auto before = Events.size();
    Events.push_back(std::move(ev));
    std::stable_sort(Events.begin(), Events.end());
    Events.erase(
        std::unique(Events.begin(), Events.end()),
        Events.end()
    );
    auto after = Events.size();
    return after != before;
}

NJson::TJsonValue TScoringTraceTag::GetDetailReport() const {
    NJson::TJsonValue result;
    result.InsertValue("score", GetValue());
    return result;
}

NDrive::TScheme TScoringTraceTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSJson>("events", "События");
    return result;
}

void TScoringTraceTag::SerializeSpecialDataToJson(NJson::TJsonValue& json) const {
    TBase::SerializeSpecialDataToJson(json);
    NJson::InsertField(json, "events", Events);
}

bool TScoringTraceTag::DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
    if (!NJson::ParseField(json, "events", Events, false)) {
        return false;
    }
    std::stable_sort(Events.begin(), Events.end());
    return TBase::DoSpecialDataFromJson(json, errors);
}

TScoringTraceTag::TProto TScoringTraceTag::DoSerializeSpecialDataToProto() const {
    TProto proto = TBase::DoSerializeSpecialDataToProto();
    for (auto&& event : Events) {
        auto protoEvent = proto.AddEvent();
        protoEvent->SetTimestamp(event.Timestamp.Seconds());
        *protoEvent->MutableLocation() = event.Location.SerializeHP();
        protoEvent->SetValue(event.Value);
        if (event.Kind != TEvent::EKind::Unknown) {
            protoEvent->SetKind(static_cast<NDrive::NProto::TScoringTagEvent::EKind>(event.Kind));
        }
    }
    if (auto signalTagTraits = proto.MutableSignalTagTraits()) {
        Serialize(*signalTagTraits);
    }
    return proto;
}

bool TScoringTraceTag::DoDeserializeSpecialDataFromProto(const TProto& proto) {
    for (auto&& event : proto.GetEvent()) {
        TEvent tagEvent;
        tagEvent.Timestamp = TInstant::Seconds(event.GetTimestamp());
        if (!tagEvent.Location.Deserialize(event.GetLocation())) {
            return false;
        }
        tagEvent.Value = event.GetValue();
        tagEvent.Kind = static_cast<TEvent::EKind>(event.GetKind());
        Events.push_back(std::move(tagEvent));
    }
    if (proto.HasSignalTagTraits()) {
        if (!Deserialize(proto.GetSignalTagTraits())) {
            return false;
        }
    }
    return TBase::DoDeserializeSpecialDataFromProto(proto);
}

bool TScoringTraceTag::OnBeforeAdd(const TString& entityId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    Y_UNUSED(userId);
    bool enableSignals = SignalsEnabled(server);
    if (enableSignals) {
        Y_ASSERT(std::is_sorted(Events.begin(), Events.end()));
        auto start = TInstant::Zero();
        if (!start && !Events.empty()) {
            start = Events.front().Timestamp;
        }
        if (!start) {
            start = GetTimestamp();
        }
        if (!start) {
            start = Now();
        }
        auto entityUuid = TUuid::TryParse(entityId);
        if (!entityUuid) {
            tx.SetErrorInfo("ScoringTraceTag::OnBeforeAdd", TStringBuilder() << "cannot parse " << entityId << " as UUID");
            return false;
        }
        if (!Fill(*entityUuid, start, *server, tx)) {
            return false;
        }

        auto finish = !Events.empty() ? Events.back().Timestamp : TInstant::Zero();
        if (!SetFinish(finish, tx)) {
            return false;
        }
    }
    return true;
}

bool TScoringTraceTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    Y_UNUSED(userId);
    bool enableSignals = SignalsEnabled(server);
    if (enableSignals) {
        if (!UpdateSignal(self, tx)) {
            return false;
        }
    }
    return true;
}

bool TScoringTraceTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    Y_UNUSED(userId);
    bool enableSignals = SignalsEnabled(server);
    if (enableSignals) {
        if (!RemoveSignal(self, tx)) {
            return false;
        }
    }
    return true;
}

bool TScoringTraceTag::OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    auto impl = std::dynamic_pointer_cast<TScoringTraceTag>(to);
    if (!impl) {
        return true;
    }
    return impl->OnBeforeAdd(self.GetObjectId(), userId, server, tx);
}

bool TScoringTraceTag::OnAfterUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    Y_UNUSED(to);
    return OnAfterAdd(self, userId, server, tx);
}

bool TScoringTraceTag::SignalsEnabled(const NDrive::IServer* server) const {
    if (!server) {
        return false;
    }
    auto description = GetDescriptionAs<TDescription>(*server);
    return description ? description->IsSignalEnabled() : false;
}

NDrive::ITag::TFactory::TRegistrator<TScoringTraceTag> ScoringTraceTagRegistrator(TScoringTraceTag::TypeName);
TTagDescription::TFactory::TRegistrator<TScoringTraceTag::TDescription> ScoringTraceTagDescriptionRegistrator(TScoringTraceTag::TypeName);

template <>
NJson::TJsonValue NJson::ToJson(const TScoringTraceTag::TEvent& event) {
    NJson::TJsonValue json;
    NJson::InsertField(json, "timestamp", event.Timestamp.Seconds());
    NJson::InsertField(json, "location", event.Location);
    NJson::InsertField(json, "value", event.Value);
    if (event.Kind != TScoringTraceTag::TEvent::EKind::Unknown) {
        NJson::InsertField(json, "kind", ::ToString(event.Kind));
    }
    return json;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& json, TScoringTraceTag::TEvent& event) {
    return
        NJson::ParseField(json, "timestamp", event.Timestamp, false) &&
        NJson::ParseField(json, "location", event.Location, false) &&
        NJson::ParseField(json, "value", event.Value, false) &&
        NJson::ParseField(json, "kind", NJson::Stringify(event.Kind), false);
}
