#pragma once

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/drivematics/signal/tag.h>

#include <drive/backend/proto/tags.pb.h>
#include <rtline/library/geometry/coord.h>

class IScoringBaseTag : public ISerializableTag<NDrive::NProto::TScoringTag> {
private:
    using TBase = ISerializableTag<NDrive::NProto::TScoringTag>;

    class TScoringDetails {
    public:
        using TProto = NDrive::NProto::TScoringDetails;

    public:
        static NDrive::TScheme GetScheme();

        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& json);

        TScoringDetails::TProto SerializeToProto() const;
        bool DeserializeFromProto(const TScoringDetails::TProto& proto);

    private:
        R_FIELD(ui64, LateralAccelerationCornering, 0);
        R_FIELD(ui64, LateralAccelerationStraight, 0);
        R_FIELD(ui64, HighEngineSpeed, 0);
        R_FIELD(ui64, HarshBraking, 0);
        R_FIELD(ui64, SharpAcceleration, 0);
        R_FIELD(ui64, SectionsWithSpeeding, 0);
    };

private:
    R_FIELD(double, Value, 0);
    R_FIELD(TInstant, Timestamp, TInstant::Zero());
    R_OPTIONAL(TScoringDetails, Details);
    R_OPTIONAL(double, PreviousValue);
    R_OPTIONAL(double, Rank);

public:
    using TBase::TBase;

    enum class EScoringKind {
        // Aggressive used for storing aggression scoring value.
        Aggressive = 0 /* "aggressive" */,
        // Speeding used for storing speeding value.
        Speeding = 1 /* "speeding" */,
        // TrialMileage used for storing trial mileage value.
        TrialMileage = 2 /* "trial_mileage" */,
    };

    class TDescription: public TTagDescription {
    private:
        using TBase = TTagDescription;

    public:
        R_FIELD(EScoringKind, Kind);

    public:
        using TBase::TBase;

        virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        virtual NJson::TJsonValue DoSerializeMetaToJson() const override;
        virtual bool DoDeserializeMetaFromJson(const NJson::TJsonValue& json) override;

    private:
        static TFactory::TRegistrator<TDescription> Registrator;
    };

public:
    virtual EUniquePolicy GetUniquePolicy() const override;
    virtual NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;

protected:
    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    virtual TProto DoSerializeSpecialDataToProto() const override;
    virtual bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

protected:
    NJson::TJsonValue RawData;
};

class TScoringCarTag : public IScoringBaseTag {
public:
    static const TString TypeName;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;

private:
    static TFactory::TRegistrator<TScoringCarTag> Registrator;
};

class TScoringUserTag : public IScoringBaseTag {
public:
    static const TString TypeName;

    virtual TSet<NEntityTagsManager::EEntityType> GetObjectType() const override;

private:
    static TFactory::TRegistrator<TScoringUserTag> Registrator;
};

class TScoringTraceTag
    : public IScoringBaseTag
    , public NDrivematics::TSignalTagTraits
{
private:
    using TBase = IScoringBaseTag;

public:
    class TDescription: public TBase::TDescription {
    private:
        using TBase = IScoringBaseTag::TDescription;

    public:
        R_FIELD(bool, SignalEnabled, false);

    public:
        using TBase::TBase;

        NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;
        NJson::TJsonValue DoSerializeMetaToJson() const override;
        bool DoDeserializeMetaFromJson(const NJson::TJsonValue& json) override;
    };

    class TEvent {
    public:
        enum class EKind {
            Unknown = 0 /* "unknown" */,
            Acceleration = 1 /* "acceleration" */,
            Braking = 2 /* "braking" */,
            StraightLateralAcceleration = 3 /* "straight_lateral_acceleration" */,
            TurningLateralAcceleration = 4 /* "turning_lateral_acceleration" */,
            // Do not forget for NDrive::NProto::TScoringTagEvent::EKind.
        };

        TInstant Timestamp = TInstant::Zero();
        TGeoCoord Location = {};
        double Value = 0;
        EKind Kind = EKind::Unknown;

    public:
        inline bool operator<(const TEvent& other) const {
            return Timestamp < other.Timestamp;
        }
        inline bool operator==(const TEvent& other) const {
            return Timestamp == other.Timestamp;
        }
    };
    using TEvents = TVector<TEvent>;

private:
    R_FIELD(TEvents, Events);

public:
    static const TString TypeName;

public:
    bool AddEvent(TEvent&& ev);

    TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
        return { GetEntityType() };
    }

    NJson::TJsonValue GetDetailReport() const override;
    NDrive::TScheme GetScheme(const NDrive::IServer* server) const override;

    bool OnBeforeAdd(const TString& entityId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;
    bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const override;

    bool OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;

    bool OnBeforeUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const override;
    bool OnAfterUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const override;

    void SerializeSpecialDataToJson(NJson::TJsonValue& json) const override;
    bool SignalsEnabled(const NDrive::IServer* server) const;

protected:
    NEntityTagsManager::EEntityType GetEntityType() const override {
        return NEntityTagsManager::EEntityType::Trace;
    }

    bool DoSpecialDataFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override;
    TProto DoSerializeSpecialDataToProto() const override;
    bool DoDeserializeSpecialDataFromProto(const TProto& proto) override;

private:
    static TFactory::TRegistrator<TScoringTraceTag> Registrator;
};
