#include "service_session.h"

#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/device_snapshot/manager.h>

#include <drive/backend/database/drive_api.h>

NJson::TJsonValue TServiceSession::TCompilation::GetReport(ELocalization /*locale*/, const NDrive::IServer* /*server*/, ISessionReportCustomization::TPtr /*customization*/) const {
    NJson::TJsonValue result;
    result["fuel_level_before"] = FuelLevelBefore;
    result["fuel_level_after"] = FuelLevelAfter;
    result["tag_id"] = TagId;
    result["tag_name"] = TagName;
    result["finish_action"] = ToString(FinishAction);

    NJson::TJsonValue& imagesJson = result.InsertValue("images", NJson::TJsonValue(NJson::JSON_ARRAY));
    for (auto&& image : Images) {
        NJson::TJsonValue imgReport;
        imgReport["marker"] = image.second.Marker;
        imgReport["path"] = image.second.Path;
        imgReport["timestamp"] = image.second.Timestamp.Seconds();
        imagesJson.AppendValue(imgReport);
    }
    return result;
}

bool TServiceSession::TCompilation::Fill(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) {
    if (events.empty()) {
        return false;
    }

    TAtomicSharedPtr<TCarTagHistoryEvent> firstEvent = events.front();
    TagId = firstEvent->GetTagId();
    TagName = (*firstEvent)->GetName();
    SessionId = firstEvent->GetTagId() + "_" + ToString(timeline.front().GetEventId());

    for (auto&& timeEvent : timeline) {
        if (timeEvent.GetEventIndex() >= events.size()) {
            continue;
        }
        auto ev = *events[timeEvent.GetEventIndex()];
        FinishAction = ev.GetHistoryAction();
        if (ev.GetHistoryAction() == EObjectHistoryAction::SetTagPerformer
            || ev.GetHistoryAction() == EObjectHistoryAction::DropTagPerformer
            || ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            const THistoryDeviceSnapshot* ds = ev->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            if (ds) {
                double fuelLevel = 0;
                if (ds->GetFuelLevel(fuelLevel)) {
                    if (timeline.front().GetEventId() == (i64)ev.GetHistoryEventId()) {
                        FuelLevelBefore = fuelLevel;
                    }
                    FuelLevelAfter = fuelLevel;
                }
            }
        }

        if ((ev.GetHistoryAction() == EObjectHistoryAction::Add
            || ev.GetHistoryAction() == EObjectHistoryAction::UpdateData
            || ev.GetHistoryAction() == EObjectHistoryAction::TagEvolve)
            && ev)
        {
            if (!ServiceTagsReport.contains(ev->GetName())) {
                ServiceTags.push_back(ev->GetName());
            }
            ServiceTagsReport[ev->GetName()] = ev->GetServiceReport(TagsManager);
        }

        if (ev.GetHistoryAction() == EObjectHistoryAction::AddSnapshot) {
            const TString& userId = ev.GetHistoryUserId();
            if (UserId && UserId != userId) {
                return false;
            }
            UserId = userId;
            const TImagesSnapshot* snapshot = ev->GetObjectSnapshotAs<TImagesSnapshot>();
            if (!!snapshot) {
                for (auto&& image : snapshot->GetImages()) {
                    TImageData data;
                    data.Path = image.Path;
                    data.Marker = image.Marker;
                    data.Timestamp = ev.GetHistoryInstant();
                    data.MD5 = image.MD5;
                    Images[image.ExternalId] = std::move(data);
                }
            }
        }
    }
    return true;
}

TString TServiceSession::TCompilation::GetServiceReport(const NDrive::IServer* server) const {
    TString report;
    const auto& manager = server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
    for (const auto& tagName : ServiceTags) {
        auto description = manager.GetDescriptionByName(tagName);
        auto it = ServiceTagsReport.find(tagName);
        if (!description || !description->GetEnableReport() || it == ServiceTagsReport.end()) {
            continue;
        }
        report += description->GetDisplayName() + "\n" + it->second + "\n";
    }
    if (!report.empty()) {
        report.pop_back();
    }
    return report;
}

bool TServiceSession::TestEvent(const TCarTagHistoryEvent& histEvent) const {
    if (histEvent.GetObjectId() != GetObjectId()) {
        return false;
    }
    return true;
}

NEventsSession::EEventCategory TServiceSessionSelector::Accept(const TCarTagHistoryEvent& histEvent) const {
    if (!histEvent->GetPerformer() && histEvent.GetHistoryAction() != EObjectHistoryAction::DropTagPerformer) {
        return NEventsSession::EEventCategory::IgnoreExternal;
    }
    NEventsSession::EEventCategory result = NEventsSession::IgnoreExternal;
    if (histEvent.GetHistoryAction() == EObjectHistoryAction::ForceTagPerformer) {
        result = NEventsSession::EEventCategory::Switch;
    } else if (histEvent.GetHistoryAction() == EObjectHistoryAction::Remove || histEvent.GetHistoryAction() == EObjectHistoryAction::DropTagPerformer) {
        result = NEventsSession::EEventCategory::End;
    } else if (histEvent.GetHistoryAction() == EObjectHistoryAction::SetTagPerformer) {
        result = NEventsSession::EEventCategory::Begin;
    } else {
        result = NEventsSession::EEventCategory::Internal;
    }
    auto description = TagsManager.GetDescriptionByName(histEvent->GetName());
    if (!description || !description->GetEnableSessions()) {
        return NEventsSession::IgnoreExternal;
    }
    return result;
}
