#pragma once

#include "device_tags.h"

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/history/session.h>

#include <util/generic/map.h>

class TServiceSessionSelector;

class TServiceSession: public IEventsSession<TCarTagHistoryEvent> {
private:
    using TBase = IEventsSession<TCarTagHistoryEvent>;
    using ICompilation = typename TBase::ICompilation;
    using TTimeEvent = typename TBase::TTimeEvent;

public:
    struct TImageData {
        TInstant Timestamp;
        TString Path;
        TString Marker;
        TString MD5;
    };

    class TCompilation: public ICompilation {
    private:
        TMap<TString, TImageData> Images;
        TMap<TString, TString> ServiceTagsReport;
        TVector<TString> ServiceTags;
        TString SessionId;
        R_READONLY(TString, TagId);
        R_READONLY(TString, TagName);
        R_READONLY(TString, UserId);
        R_READONLY(double, FuelLevelBefore, 0);
        R_READONLY(double, FuelLevelAfter, 0);
        R_READONLY(EObjectHistoryAction, FinishAction, EObjectHistoryAction::Unknown);

        const ITagsMeta& TagsManager;
    public:
        TCompilation(const ITagsMeta& tagsManager)
            : TagsManager(tagsManager)
        {
        }

        const TMap<TString, TImageData>& GetImages() const {
            return Images;
        }

        const TString& GetSessionId() const override {
            return SessionId;
        }

        virtual NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer* /*server*/, ISessionReportCustomization::TPtr /*customization*/) const override;
        virtual bool Fill(const TVector<TTimeEvent>& timeline, const TVector<TAtomicSharedPtr<TCarTagHistoryEvent>>& events) override;
        TString GetServiceReport(const NDrive::IServer* server) const;
    };

    using TCompilationPtr = TAtomicSharedPtr<TCompilation>;

protected:
    virtual bool TestEvent(const TCarTagHistoryEvent& histEvent) const override;

public:
    TServiceSession(const ITagsMeta& tagsManager)
        : TagsManager(tagsManager)
    {
    }

    THolder<ICompilation> BuildDefaultCompilation() const override {
        return MakeHolder<TCompilation>(TagsManager);
    }

protected:
    virtual TUniquePtr DoClone() const override {
        return MakeHolder<TServiceSession>(*this);
    }

private:
    const ITagsMeta& TagsManager;
};

class TServiceSessionSelector: public ISessionSelector<TCarTagHistoryEvent> {
private:
    const ITagsMeta& TagsManager;
    TString Name;
    TSet<TString> AcceptedTagTypes;

public:
    TServiceSessionSelector(const ITagsMeta& tagsManager, const TString& name)
        : TagsManager(tagsManager)
        , Name(name)
    {}

    virtual TString GetName() const override {
        return Name;
    }

    virtual IEventsSession<TCarTagHistoryEvent>::TPtr BuildSession() const override {
        return new TServiceSession(TagsManager);
    }

    virtual NEventsSession::EEventCategory Accept(const TCarTagHistoryEvent& e) const override;
};
