#include "servicing.h"

#include "additional_service.h"
#include "chargable.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/sessions/manager/billing.h>

#include <rtline/library/json/field.h>

NDrive::TScheme TServicingTag::TDescription::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TTagDescription::GetScheme(server);
    result.Add<TFSBoolean>("cancellable", "Can servicing be cancelled").SetDefault(Cancellable);
    result.Add<TFSVariants>("device_tags", "Device tags to be added on servicing start").SetReference("device_tags").SetMultiSelect(true);
    result.Add<TFSVariants>("user_tags", "User tags to be added on servicing start").SetReference("user_tags").SetMultiSelect(true);
    return result;
}

NJson::TJsonValue TServicingTag::TDescription::DoSerializeMetaToJson() const {
    NJson::TJsonValue result = TTagDescription::DoSerializeMetaToJson();
    NJson::FieldsToJson(result, GetFields());
    return result;
}

bool TServicingTag::TDescription::DoDeserializeMetaFromJson(const NJson::TJsonValue& value) {
    return
        TTagDescription::DoDeserializeMetaFromJson(value) &&
        NJson::TryFieldsFromJson(value, GetFields());
}

NDrive::TScheme TServicingTag::GetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSString>("session_id").SetVisual(TFSString::EVisualType::CurrentSession).SetRequired(true);
    return result;
}

bool TServicingTag::IsEligible(const NDrive::IServer& server) const {
    auto sessionBuilder = server.GetDriveDatabase().GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing");
    auto session = sessionBuilder ? sessionBuilder->GetSession(SessionId) : nullptr;
    auto lastEvent = session ? session->GetLastEvent() : Nothing();
    if (!lastEvent) {
        return false;
    }
    return lastEvent.GetRef()->GetName() == TChargableTag::Parking;
}

bool TServicingTag::IsEligible(const TDBTag& sessionTag) const {
    return sessionTag->GetName() == TChargableTag::Parking;
}

bool TServicingTag::OnBeforeAdd(const TString& objectId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    Y_UNUSED(userId);
    auto sessionBuilder = server ? server->GetDriveDatabase().GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing", Now()) : nullptr;
    if (!sessionBuilder) {
        tx.SetErrorInfo("TServicingTag::OnBeforeAdd", "cannot GetSessionsBuilder billing");
        return false;
    }
    if (!SessionId) {
        auto sessions = sessionBuilder->GetObjectSessions(objectId);
        for (auto&& session : sessions) {
            if (!session || session->GetClosed()) {
                continue;
            }
            if (SessionId) {
                tx.SetErrorInfo("ServicingTag::OnBeforeAdd", "several session by object_id");
                return false;
            }
            SessionId = session->GetSessionId();
        }
        if (!SessionId) {
            tx.SetCode(HTTP_BAD_REQUEST);
            tx.SetErrorInfo("ServicingTag::OnBeforeAdd", "session_id is not set");
            return false;
        }
    } else {
        auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetSession(SessionId, tx);
        if (!optionalSession) {
            return false;
        }
        auto session = *optionalSession;
        if (!session || session->GetClosed()) {
            tx.SetCode(HTTP_BAD_REQUEST);
            tx.SetErrorInfo("ServicingTag::OnBeforeAdd", "incorrect session_id");
            return false;
        }
    }
    return true;
}

bool TServicingTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    R_ENSURE(server, {}, "null Server", tx);
    if (IsEligible(*server)) {
        auto permissions = Yensured(server->GetDriveAPI())->GetUserPermissions(userId, {});
        R_ENSURE(permissions, {}, "cannot GetUserPermissions", tx);
        return Start(self, *permissions, *server, tx);
    }
    return true;
}

bool TServicingTag::Start(const TDBTag& self, const TUserPermissions& permissions, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    auto impl = self.GetTagAs<TServicingTag>();
    if (!impl) {
        tx.SetErrorInfo("ServicingTag::Start", "cannot cast to ServicingTag " + self.GetTagId());
        return false;
    }

    const auto& tagsManager = server.GetDriveDatabase().GetTagsManager();
    auto optionalSession = server.GetDriveDatabase().GetSessionManager().GetSession(impl->GetSessionId(), tx);
    if (!optionalSession) {
        return false;
    }
    auto session = *optionalSession;
    if (!session) {
        tx.SetErrorInfo("ServicingTag::Start", "cannot find session " + impl->GetSessionId());
        return false;
    }
    if (session->GetObjectId() != self.GetObjectId()) {
        tx.SetCode(HTTP_BAD_REQUEST);
        tx.SetErrorInfo("ServicingTag::Start", "object_id/session_id mismatch");
        return false;
    }

    const auto& sessionTagId = session->GetInstanceId();
    auto optionalSessionTag = tagsManager.GetDeviceTags().RestoreTag(session->GetInstanceId(), tx);
    if (!optionalSessionTag) {
        return false;
    }
    auto sessionTag = std::move(*optionalSessionTag);
    if (!sessionTag) {
        tx.SetErrorInfo("ServicingTag::Start", "cannot find session tag " + sessionTagId);
        return false;
    }
    if (!impl->IsEligible(sessionTag)) {
        tx.SetErrorInfo("ServicingTag::Start", "session " + impl->GetSessionId() + " is ineligible");
        return false;
    }

    auto description = impl->GetDescriptionAs<TDescription>(server, tx);
    if (!description) {
        return false;
    }

    auto to = tagsManager.GetTagsMeta().CreateTag(TChargableTag::Servicing);
    auto servicing = std::dynamic_pointer_cast<TChargableTag>(to);
    if (!servicing) {
        tx.SetErrorInfo("ServicingTag::Start", "cannot create tag " + TChargableTag::Servicing);
        return false;
    }
    if (auto&& parentUserTagId = impl->GetParentUserTagId()) {
        auto optionalUserTag = tagsManager.GetUserTags().RestoreTag(parentUserTagId, tx);
        if (!optionalUserTag) {
            tx.SetErrorInfo("ServicingTag::Start", "cannot get parent user tag " + parentUserTagId);
            return false;
        }
        if (!TAdditionalServiceOfferHolderTag::OnStarted(*optionalUserTag, tx, *servicing, permissions, server)) {
            return false;
        }
    } else if (!impl->OnStarted(tx, *servicing, permissions.GetUserId(), *session, server)) {
        return false;
    }
    const auto& sessionUserId = session->GetUserId();
    auto userPermissions = server.GetDriveDatabase().GetUserPermissions(sessionUserId, {});
    if (!userPermissions) {
        tx.SetErrorInfo("ServicingTag::Start", "cannot GetUserPermissions for " + sessionUserId);
        return false;
    }
    if (!TChargableTag::DirectEvolve(sessionTag, servicing, *userPermissions, server, tx, nullptr)) {
        return false;
    }
    if (!tagsManager.GetDeviceTags().RemoveTag(self, permissions.GetUserId(), &server, tx)) {
        return false;
    }
    return true;
}

bool TServicingTag::OnStarted(NDrive::TEntitySession& tx, TChargableTag& servicingTag, const TString& userId, const TCommonTagSessionManager::TSession& session, const NDrive::IServer& server) const {
    auto description = GetDescriptionAs<TDescription>(server, tx);
    if (!description) {
        return false;
    }
    const auto& tagsManager = server.GetDriveDatabase().GetTagsManager();
    servicingTag.SetServicingCancellable(description->IsCancellable());
    {
        TVector<ITag::TPtr> subtags;
        for (auto&& tagName : description->GetDeviceTags()) {
            auto tag = tagsManager.GetTagsMeta().CreateTag(tagName);
            if (!tag) {
                tx.SetErrorInfo("ServicingTag::Start", "cannot create tag " + tagName);
                return false;
            }
            subtags.push_back(std::move(tag));
        }
        servicingTag.SetSubtags(std::move(subtags));
    }
    const auto& sessionUserId = session.GetUserId();
    {
        TVector<ITag::TPtr> tags;
        for (auto&& tagName : description->GetUserTags()) {
            auto tag = tagsManager.GetTagsMeta().CreateTag(tagName);
            if (!tag) {
                tx.SetErrorInfo("ServicingTag::Start", "cannot create tag " + tagName);
                return false;
            }
        }
        if (!tagsManager.GetUserTags().AddTags(tags, userId, sessionUserId, &server, tx)) {
            return false;
        }
    }
    return true;
}

bool TServicingTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    if (!TBase::DoSpecialDataFromJson(value, errors)) {
        return false;
    }
    return NJson::TryFieldsFromJson(value, GetFields(), errors);
}

void TServicingTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::FieldsToJson(value, GetFields());
}

TTagDescription::TFactory::TRegistrator<TServicingTag::TDescription> ServicingTagDescriptionRegistrator(TServicingTag::Type());
ITag::TFactory::TRegistrator<TServicingTag> TServicingTag::Registrator(TServicingTag::Type());
